﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsReaderWriterLock.cpp,PageSampleOsReaderWriterLock}
 *
 * @brief
 *  Readers-Writer ロック機能のサンプルプログラム
 */

/**
 * @page PageSampleOsReaderWriterLock ReaderWriterLock
 * @tableofcontents
 *
 * @brief
 *  Readers-Writer ロック機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsReaderWriterLock_SectionBrief 概要
 *  ここでは、Readers-Writer ロック機能を使ったサンプルプログラムについて
 *  説明します。
 *
 *  Readers-Writer ロック機能の使い方については、
 *  Readers-Writer ロック機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsReaderWriterLock_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsReaderWriterLock Samples/Sources/Applications/OsReaderWriterLock @endlink 以下にあります。
 *
 * @section PageSampleOsReaderWriterLock_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsReaderWriterLock_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsReaderWriterLock_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsReaderWriterLock_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsReaderWriterLock_SectionDetail 解説
 *
 * @subsection PageSampleOsReaderWriterLock_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsReaderWriterLock.cpp
 *  @includelineno OsReaderWriterLock.cpp
 *
 * @subsection PageSampleOsReaderWriterLock_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて、3 つのスレッドを生成。それぞれを CPU コア 0～2 に割当て
 *  - 3 つのスレッドからアクセスするグローバル変数 g_Value1 と g_Value2 を用意
 *  - 上記変数へのアクセスを排他するための nn::os::ReaderWriterLockType オブジェクト g_Rwlockを 1 つ用意
 *  - nnMain() の先頭付近で g_Rwlock を初期化
 *  - SubThread1 スレッドでは読み込みロックを利用して、2 つのグローバル変数の値を参照
 *  - SubThread2 スレッドは、1 つ目のスレッドと全く同じ実装
 *  - SubThread3 スレッドでは書き込みロックを利用して、2 つのグローバル変数の値を書換え
 *  - nnMain() にて、3 つのスレッドが終了するまで待機
 *  - nnMain() にて、3 つのスレッドを破棄
 *  - nnMain() にて、g_Rwlock を破棄
 *
 *  SubThread1 と SubThread2 スレッドは、グローバル変数に対して読み込みアクセス
 *  しか行なわないため、読み込みロックを取得して変数へアクセスしています。
 *  この 2 つのスレッドはお互いに排他されることはありません。
 *
 *  一方、3 つ目のスレッドは、グローバル変数に対して書き込みアクセスを
 *  行なうため、書き込みロックを取得して変数へアクセスしています。
 *  この書き込みロック期間中は 1 つ目および 2 つ目のスレッドの
 *  読み込みロック取得操作はブロッキングされます。
 *
 *  仮に、ロックが正しく機能しない場合、1 つ目と 2 つ目のスレッドの中で
 *  2 つのグローバル変数の値を比較しており、そこでアボートすることになります。
 *  この現象は、SubThread3 の中にあるロックを「読み込み用のロック」に
 *  置き換えることで確認することができます。
 *
 *  これらのロックは、nn::os::MutexType を使っても正しく排他制御することが
 *  できますが、Readres-Writer ロック機能を使うと、読み込み目的のスレッドが
 *  複数存在する場合に、それらのスレッド同士をブロッキングせずに動作させる
 *  ことができるため、そのような条件が頻繁に発生するようなケースでは、
 *  nn::os::MutexType よりも効率よく動作させることが期待できます。
 *
 *  本サンプルを実行すると以下のように表示され、
 *  読み込みロックと書き込みロックが正しく排他されているのが分かります。
 *
 *  @verbinclude  OsReaderWriterLock_OutputExample.txt
 *
 */

#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

namespace {

const size_t                    ThreadStackSize = 8192;
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack1[ ThreadStackSize ];
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack2[ ThreadStackSize ];
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack3[ ThreadStackSize ];

int                             g_Value1 = 0;   // ロック対象変数その 1
int                             g_Value2 = 0;   // ロック対象変数その 2
nn::os::ReaderWriterLockType    g_Rwlock;       // Readers-Writer ロック

const int   LoopCount = 10000;                  // ループ回数

}   // namepsace

//-----------------------------------------------------------------------------

//
//  1 つ目のスレッドです。
//
void SubThread1(void *arg)
{
    NN_UNUSED(arg);

    for (int i = 0; i < LoopCount; ++i)
    {
        nn::os::AcquireReadLock( &g_Rwlock );
        auto value1 = g_Value1;
        auto value2 = g_Value2;
        NN_ABORT_UNLESS( value1 * 2 == value2 );
        nn::os::ReleaseReadLock( &g_Rwlock );
    }
}

//
//  2 つ目のスレッドです。
//
void SubThread2(void *arg)
{
    NN_UNUSED(arg);

    for (int i = 0; i < LoopCount; ++i)
    {
        nn::os::AcquireReadLock( &g_Rwlock );
        auto value1 = g_Value1;
        auto value2 = g_Value2;
        NN_ABORT_UNLESS( value1 * 2 == value2 );
        nn::os::ReleaseReadLock( &g_Rwlock );
    }
}

//
//  3 つ目のスレッドです。
//
void SubThread3(void *arg)
{
    NN_UNUSED(arg);

    for (int i = 0; i < LoopCount; ++i)
    {
        nn::os::AcquireWriteLock( &g_Rwlock );
        g_Value1 += 1;
        nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(10) );
        g_Value2 += 2;
        nn::os::ReleaseWriteLock( &g_Rwlock );
    }
}

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::os::ThreadType  thread1;
    nn::os::ThreadType  thread2;
    nn::os::ThreadType  thread3;

    // Rwlock を初期化
    nn::os::InitializeReaderWriterLock( &g_Rwlock );

    // スレッドを生成する
    auto result = nn::os::CreateThread( &thread1, SubThread1, NULL, g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority, 0 );
    NN_ABORT_UNLESS( result.IsSuccess(), "Cannot create thread1." );

    result = nn::os::CreateThread( &thread2, SubThread2, NULL, g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority, 1 );
    NN_ABORT_UNLESS( result.IsSuccess(), "Cannot create thread2." );

    result = nn::os::CreateThread( &thread3, SubThread3, NULL, g_ThreadStack3, ThreadStackSize, nn::os::DefaultThreadPriority, 2 );
    NN_ABORT_UNLESS( result.IsSuccess(), "Cannot create thread3." );

    // スレッドの実行を開始する
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );
    nn::os::StartThread( &thread3 );

    NN_LOG("Start ReaderWriterLock sample.\n");

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );
    nn::os::WaitThread( &thread3 );

    // 最終カウンタ値を確認
    NN_LOG("g_Value1 = %d\n", g_Value1);
    NN_LOG("g_Value2 = %d\n", g_Value2);
    NN_ABORT_UNLESS( g_Value1 * 2 == g_Value2 );
    NN_LOG("Ok.\n");

    // スレッドを破棄する
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );
    nn::os::DestroyThread( &thread3 );

    // Rwlock を破棄
    nn::os::FinalizeReaderWriterLock( &g_Rwlock );
}
