﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsMutex.cpp,PageSampleOsMutex}
 *
 * @brief
 *  ミューテックス機能のサンプルプログラム
 */

/**
 * @page PageSampleOsMutex Mutex
 * @tableofcontents
 *
 * @brief
 *  ミューテックス機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsMutex_SectionBrief 概要
 *  ここでは、ミューテックス機能を使ったサンプルプログラムの説明をします。
 *
 *  ミューテックス機能の使い方については、ミューテックス機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsMutex_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsMutex Samples/Sources/Applications/OsMutex @endlink 以下にあります。
 *
 * @section PageSampleOsMutex_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsMutex_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsMutex_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsMutex_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsMutex_SectionDetail 解説
 *
 * @subsection PageSampleOsMutex_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsMutex.cpp
 *  @includelineno OsMutex.cpp
 *
 * @subsection PageSampleOsMutex_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて、2 つのスレッドを生成
 *  - 2 つのスレッドは、いずれも ThreadSample() を利用
 *  - ThreadSample() では 100 万回のループの中で、競合リソースである g_SharedCounter をインクリメント
 *  - 2 つのスレッドが終了するまで nnMain() で待機
 *  - 最終的な g_SharedCounter の値を表示
 *  - nnMain() にて 2 つのスレッドを破棄
 *
 *  2 つのスレッドでは g_SharedCounter をインクリメントする際に LockMutex() と UnlockMutex() を
 *  行なっているため、このロック区間中は、もう片方のスレッドと排他されています。
 *
 *  そのため、最終的な g_SharedCounter の値は以下のように表示されるはずです。
 *
 *  @verbinclude  OsMutex_OutputExampleSuccessful.txt
 *
 *  一方、サンプルプログラム中の (A) の LockMutex() と (B) の UnlockMutex() を削除（もしくは、コメントアウト）して
 *  実行すると、最終的な g_SharedCounter の値は上記のようにはならず、アサートに失敗し
 *  以下のように表示されます。@n
 *  ※） 表示されているカウンタ値は実行される環境によって異なります。
 *
 *  @verbinclude  OsMutex_OutputExampleFailed.txt
 *
 *  このように、ミューテックス機能を使用することでロック区間を作り、
 *  その中で共有リソースへの操作を行なうことで、排他すべきリソースを
 *  安全に操作することができるようになります。
 *
 */

#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

namespace {

const size_t           ThreadStackSize = 8192;              // ミューテックス操作スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack2[ ThreadStackSize ];   // 2つ目のスレッドのスタック

const int     SharedCounterLoop = 1000000;  // ミューテックスを操作する回数
int           g_SharedCounter   = 0;        // ミューテックスを操作した回数
nn::os::MutexType g_Mutex;                      // 操作対象のミューテックス

}   // namepsace

//-----------------------------------------------------------------------------

//
//  各スレッドが実行する、ミューテックスの操作を行う関数です。
//
void ThreadSample(void *arg)
{
    NN_UNUSED(arg);

    for (int i = 0; i < SharedCounterLoop; ++i)
    {
        nn::os::LockMutex( &g_Mutex );         // ロック区間開始 (A)
        ++g_SharedCounter;
        nn::os::UnlockMutex( &g_Mutex );       // ロック区間終了 (B)
    }
}

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::os::ThreadType  thread1;
    nn::os::ThreadType  thread2;
    nn::Result      result;

    // ミューテックスを初期化する
    nn::os::InitializeMutex( &g_Mutex, false, 0 );

    // スレッドを生成する
    result = nn::os::CreateThread( &thread1, ThreadSample, NULL, g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create thread1." );

    result = nn::os::CreateThread( &thread2, ThreadSample, NULL, g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create thread2." );

    // スレッドの実行を開始する
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );

    // 最終カウンタ値を確認
    NN_LOG("g_SharedCounter = %d\n", g_SharedCounter);
    NN_ASSERT( g_SharedCounter == SharedCounterLoop * 2, "*** Invalid counter value!" );
    NN_LOG("Ok.\n");

    // スレッドを破棄する
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );

    // ミューテックスをファイナライズする
    nn::os::FinalizeMutex( &g_Mutex );
}
