﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsInterruptEvent.cpp,PageSampleOsInterruptEvent}
 *
 * @brief
 *  割込みイベント機能のサンプルプログラム
 */

/**
 * @page PageSampleOsInterruptEvent InterruptEvent
 * @tableofcontents
 *
 * @brief
 *  割込みイベント機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsInterruptEvent_SectionBrief 概要
 *  ここでは、割込みイベント機能を使ったサンプルプログラムの説明をします。
 *
 *  割込みイベント機能の使い方については、割込みイベント機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsInterruptEvent_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsInterruptEvent Samples/Sources/Applications/OsInterruptEvent @endlink 以下にあります。
 *
 * @section PageSampleOsInterruptEvent_SectionNecessaryEnvironment 必要な環境
 *  実機ターゲット環境でのみ割込みイベントを使用します。
 *
 * @section PageSampleOsInterruptEvent_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsInterruptEvent_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsInterruptEvent_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsInterruptEvent_SectionDetail 解説
 *
 * @subsection PageSampleOsInterruptEvent_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsInterruptEvent.cpp
 *  @includelineno OsInterruptEvent.cpp
 *
 *  その他、OsInterruptEvent-hw.＜ターゲット名＞.cpp および
 *          OsInterruptEvent-hw.＜ターゲット名＞.h   が必要になります。
 *
 * @subsection PageSampleOsInterruptEvent_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて、InterruptEvent を設定
 *  - nnMain() にて、タイマーを起動し、タイマー割込みが入るように設定
 *  - nnMain() にて、割込みを待機するスレッドを生成して動作開始
 *  - 上記スレッドにて InterruptEvent を待機しながらログ出力を 20 回繰り返す。
 *  - 上記スレッドが終了するまで nnMain() は待機
 *  - nnMain() にて上記スレッドを破棄
 *  - nnMain() にて、タイマーを停止し、タイマー割込みが入らないように設定
 *  - nnMain() にて、InterruptEvent を破棄
 *
 *  InterruptHandlerThread() は、InterruptEvent 機能を使って、タイマー割込みを
 *  待機しています。 WaitInterruptEvent() にて InterruptEvent がシグナル化
 *  されるまで待機し、その後タイマーのステータスをクリアしてから、
 *  ClearInterruptEvent() して再度割込み待ちを行なうようにします。
 *
 *  このサンプルプログラムでは、手動クリアモードで割込みイベントを
 *  待機しています。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *  InterruptEventThread() が割込みイベントで起床される度に、
 *  InterruptHandlerThread() が文字列をログ出力します。
 *
 *  @verbinclude  OsInterruptEvent_OutputExample.txt
 *
 */

#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/os/os_InterruptEvent.h>

#include "OsInterruptEvent.h"

//-----------------------------------------------------------------------------

namespace {

const size_t           ThreadStackSize = 8192;              // 割込みイベント待機スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack[ ThreadStackSize ];    // 割込みイベント待機スレッドのスタック

nn::os::InterruptEventType  g_InterruptEvent;               // タイマーの割込みイベント

}   // namespace

//-----------------------------------------------------------------------------
//  割込みイベント待機スレッド
//
void InterruptHandlerThread(void *arg)
{
    NN_UNUSED(arg);

    NN_LOG("This is an InterruptEvent handler thread.\n");

    for (int i = 0; i<20; ++i)
    {
        nn::os::WaitInterruptEvent( &g_InterruptEvent );

        NN_LOG("*** Timer Interrupt is triggered. (%d)\n", i);

        nns::os::interruptEvent::ClearTimerInterruptStatus();
        nn::os::ClearInterruptEvent( &g_InterruptEvent );
    }
}

//-----------------------------------------------------------------------------
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::os::ThreadType  thread;

    // 割込みイベントを設定する
    nn::os::InitializeInterruptEvent(
                            &g_InterruptEvent,
                            nns::os::interruptEvent::TimerInterruptName,
                            nn::os::EventClearMode_ManualClear );

    // タイマー動作を開始し、タイマー割込みが入るように設定
    nns::os::interruptEvent::StartTimerInterrupt();

    // スレッドを生成する
    nn::Result result = nn::os::CreateThread( &thread,
                                              InterruptHandlerThread,
                                              NULL,
                                              g_ThreadStack,
                                              ThreadStackSize,
                                              nn::os::DefaultThreadPriority );
    NN_ASSERT(result.IsSuccess(), "*** Cannot create InterruptHandlerThread.");

    // スレッドの実行を開始する
    nn::os::StartThread( &thread );

    // スレッドが終了するのを待って破棄する
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    // タイマー動作を停止し、タイマー割込みが入らないように設定
    nns::os::interruptEvent::StopTimerInterrupt();

    // 割込みイベントを破棄する
    nn::os::FinalizeInterruptEvent( &g_InterruptEvent );
}

