﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsEvent.cpp,PageSampleOsEvent}
 *
 * @brief
 *  イベント機能のサンプルプログラム
 */

/**
 * @page PageSampleOsEvent Event
 * @tableofcontents
 *
 * @brief
 *  イベント機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsEvent_SectionBrief 概要
 *  ここでは、イベント機能を使ったサンプルプログラムの説明をします。
 *
 *  イベント機能の使い方については、イベント機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsEvent_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsEvent Samples/Sources/Applications/OsEvent @endlink 以下にあります。
 *
 * @section PageSampleOsEvent_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsEvent_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsEvent_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsEvent_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsEvent_SectionDetail 解説
 *
 * @subsection PageSampleOsEvent_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsEvent.cpp
 *  @includelineno OsEvent.cpp
 *
 * @subsection PageSampleOsEvent_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて、2 つのスレッドを生成
 *  - 2 つのスレッドは、 ThreadFunction1() と ThreadFunction2()
 *  - ThreadFunction1() では WaitEvent() で同期を取りながら文字列をログ出力
 *  - ThreadFunction2() では適当なタイミングで SignalEvent() を通知
 *  - 2 つのスレッドが終了するまで nnMain() は待機
 *  - nnMain() にて 2 つのスレッドを破棄
 *
 *  ThreadFunction1() と ThreadFunction2() は、Event 機能を使ってタイミングを
 *  計っています。 ThreadFunction2() の SignalEvent() のタイミングを変更
 *  することで、 ThreadFunction1() 側のログ出力のスピードが変わります。
 *
 *  このサンプルプログラムでは、自動クリアモードでイベントを使用しています。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *  ThreadFunction2() がイベント通知する度、ThreadFunction1() が文字列を出力します。
 *
 *  @verbinclude  OsEvent_OutputExample.txt
 *
 */

#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

namespace {

const size_t           ThreadStackSize = 8192;              // イベント操作スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のイベント操作スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack2[ ThreadStackSize ];   // 2つ目のイベント操作スレッドのスタック

nn::os::EventType   g_Event;              // 操作対象のイベント

}   // namespace

//-----------------------------------------------------------------------------

//
//  1 つ目のスレッド（イベント待機する側）
//
void ThreadFunction1(void *arg)
{
    NN_UNUSED(arg);

    NN_LOG("This is an Event sample code.\n");

    nn::os::WaitEvent( &g_Event );
    NN_LOG("1st signal.\n");

    nn::os::WaitEvent( &g_Event );
    NN_LOG("2nd signal.\n");

    nn::os::WaitEvent( &g_Event );
    NN_LOG("3rd signal.\n");

    nn::os::WaitEvent( &g_Event );
    NN_LOG("4th signal.\n");

    nn::os::WaitEvent( &g_Event );
    NN_LOG("5th signal.\n");

    nn::os::WaitEvent( &g_Event );
    NN_LOG("6th signal.\n");
}


//
//  2 つ目のスレッド（イベント通知する側）
//
void ThreadFunction2(void *arg)
{
    NN_UNUSED(arg);

    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
    nn::os::SignalEvent( &g_Event );

    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(500) );
    nn::os::SignalEvent( &g_Event );

    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
    nn::os::SignalEvent( &g_Event );

    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(500) );
    nn::os::SignalEvent( &g_Event );

    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
    nn::os::SignalEvent( &g_Event );

    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(500) );
    nn::os::SignalEvent( &g_Event );
}


//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::os::ThreadType  thread1;
    nn::os::ThreadType  thread2;
    nn::Result      result;

    // イベントを初期化する
    nn::os::InitializeEvent( &g_Event, false, nn::os::EventClearMode_AutoClear );

    // スレッドを生成する
    result = nn::os::CreateThread( &thread1, ThreadFunction1, NULL, g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create thread1." );

    result = nn::os::CreateThread( &thread2, ThreadFunction2, NULL, g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create thread2." );

    // スレッドの実行を開始する
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );

    // スレッドを破棄する
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );

    // イベントをファイナライズする
    nn::os::FinalizeEvent( &g_Event );
}

