﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsBarrier.cpp,PageSampleOsBarrier}
 *
 * @brief
 *  バリア機能のサンプルプログラム
 */

/**
 * @page PageSampleOsBarrier Barrier
 * @tableofcontents
 *
 * @brief
 *  バリア機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsBarrier_SectionBrief 概要
 *  ここでは、バリア機能を使ったサンプルプログラムの説明をします。
 *
 *  バリア機能の使い方については、バリア機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsBarrier_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsBarrier Samples/Sources/Applications/OsBarrier @endlink 以下にあります。
 *
 * @section PageSampleOsBarrier_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsBarrier_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsBarrier_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsBarrier_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsBarrier_SectionDetail 解説
 *
 * @subsection PageSampleOsBarrier_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsBarrier.cpp
 *  @includelineno OsBarrier.cpp
 *
 * @subsection PageSampleOsBarrier_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて、3 つのスレッドを生成
 *  - 3 つのスレッドは、ThreadFunction1(), ThreadFunction2(), ThreadFunction3()
 *  - それぞれの ThreadFunctionN() では適当な時間待ちの後、バリア機能で合流
 *  - バリアで同期した後、それぞれ同時にログを出力して終了
 *  - nnMain() にて 3 つのスレッドを破棄
 *
 *  3 つのスレッドはどれが最初に AwaitBarrier() に到達するかは分かりませんが、
 *  3 つ全てのスレッドが合流した時点で、再度 3 つのスレッドが動作を再開します。
 *  そのため、最終的なログ表示は以下のようになります。
 *
 *  @verbinclude  OsBarrier_OutputExample.txt
 *
 *  それぞれのスレッドが「バリア同期待ち」に入るまでは表示順番はバラバラです。
 *  最後の「実行を再開」は 3 つ全てのスレッドが「バリア同期待ち」に入った後に
 *  並んで表示されます（ただし、その表示順番は実行環境によって異なります）。
 *
 *  このようにバリア機能を使用することで、複数スレッドの合流と実行再開を
 *  制御することができます。
 *
 */

#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

namespace {

const size_t           ThreadStackSize = 8192;              // バリア操作スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack2[ ThreadStackSize ];   // 2つ目のスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack3[ ThreadStackSize ];   // 3つ目のスレッドのスタック

nn::os::BarrierType g_Barrier;      // 操作対象のバリア

}   // namespace

//-----------------------------------------------------------------------------

//
//  ランダムな待ち時間
//
void WaitRandom(const char* name)
{
    for (int i = 0; i < 3; ++i)
    {
        // 乱数の代わりに Tick 値を使う
        int64_t tick     = nn::os::GetSystemTick().GetInt64Value();
        auto    waitTime = nn::TimeSpan::FromMilliSeconds( ((tick & 0xf) + 1) * 10 );
        NN_LOG("%s: Wait %lld msec\n", name, waitTime.GetMilliSeconds());
        nn::os::SleepThread( waitTime );
    }
}


//
//  1 つ目のバリア待機スレッドです。
//
void ThreadFunction1(void *arg)
{
    const char* name = reinterpret_cast<const char*>( arg );

    NN_LOG("Thread1: Start\n");

    WaitRandom( name );

    // バリア同期待ち
    NN_LOG("Thread1: Wait barrier synchronization\n");
    nn::os::AwaitBarrier( &g_Barrier );

    NN_LOG("Thread1: Resume after barrier wait\n");
}


//
//  2 つ目のバリア待機スレッドです。
//
void ThreadFunction2(void *arg)
{
    const char* name = reinterpret_cast<const char*>( arg );

    NN_LOG("Thread2: Start\n");

    WaitRandom( name );

    // バリア同期待ち
    NN_LOG("Thread2: Wait barrier synchronization\n");
    nn::os::AwaitBarrier( &g_Barrier );

    NN_LOG("Thread2: Resume after barrier wait\n");
}


//
//  3 つ目のバリア待機スレッドです。
//
void ThreadFunction3(void *arg)
{
    const char* name = reinterpret_cast<const char*>( arg );

    NN_LOG("Thread3: Start\n");

    WaitRandom( name );

    // バリア同期待ち
    NN_LOG("Thread3: Wait barrier synchronization\n");
    nn::os::AwaitBarrier( &g_Barrier );

    NN_LOG("Thread3: Resume after barrier wait\n");
}


//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::os::ThreadType  thread1;
    nn::os::ThreadType  thread2;
    nn::os::ThreadType  thread3;
    nn::Result      result;

    // バリアを初期化する
    nn::os::InitializeBarrier( &g_Barrier, 3 );

    // スレッドを生成する
    result = nn::os::CreateThread( &thread1, ThreadFunction1, const_cast<char*>("Thread1"), g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "*** Cannot create thread1." );

    result = nn::os::CreateThread( &thread2, ThreadFunction2, const_cast<char*>("Thread2"), g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "*** Cannot create thread2." );

    result = nn::os::CreateThread( &thread3, ThreadFunction3, const_cast<char*>("Thread3"), g_ThreadStack3, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "*** Cannot create thread3." );

    // スレッドの実行を開始する
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );
    nn::os::StartThread( &thread3 );

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );
    nn::os::WaitThread( &thread3 );

    // スレッドを破棄する
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );
    nn::os::DestroyThread( &thread3 );

    // バリアをファイナライズする
    nn::os::FinalizeBarrier( &g_Barrier );

    // 全スレッドの終了を確認する
    NN_LOG("Ok.\n");
}
