﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OeDisplayResolution.cpp,PageSampleOeDisplayResolution}
 *
 * @brief
 *  デフォルトディスプレイの画面解像度取得のサンプルプログラム
 */

/**
 * @page PageSampleOeDisplayResolution デフォルトディスプレイの画面解像度取得
 * @tableofcontents
 *
 * @brief
 *  アプリケーション動作環境操作ライブラリを用いたデフォルトディスプレイの画面解像度取得の解説です。
 *
 * @section PageSampleOeDisplayResolution_SectionBrief 概要
 *  ここでは、アプリケーション動作環境操作ライブラリを用いてデフォルトディスプレイの画面解像度を取得する方法を解説します。
 *
 *  アプリケーション動作環境操作ライブラリの使い方については、
 *  @ref nn::oe "アプリケーション動作環境操作ライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOeDisplayResolution_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OeDisplayResolution
 *  Samples/Sources/Applications/OeDisplayResolution @endlink 以下にあります。
 *
 * @section PageSampleOeDisplayResolution_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX プラットフォームでのみビルドと実行が可能です。
 *
 * @section PageSampleOeDisplayResolution_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOeDisplayResolution_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOeDisplayResolution_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOeDisplayResolution_SectionDetail 解説
 *
 * @subsection PageSampleOeDisplayResolution_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OeDisplayResolution.cpp
 *  @includelineno OeDisplayResolution.cpp
 *
 * @subsection PageSampleOeDisplayResolution_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムはデフォルトディスプレイの画面解像度を取得し、それをログに出力します。
 *  デフォルトディスプレイの画面解像度は以下の場合において変更されます。
 *
 *  - SDEV クレードルや Switch ドックの抜き差しを行う。
 *  - 動作モードが据置モードのとき、DevMenu や HomeMenu の設定画面で画面解像度を変更する。
 *
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - アプリケーション動作環境操作ライブラリを初期化。
 *  - 現在のデフォルトディスプレイの画面解像度の取得。
 *  - デフォルトディスプレイの画面解像度が変更されたときのシステムイベントを取得。
 *  - デフォルトディスプレイの画面解像度が変更されるのを待機。
 *  - 現在のデフォルトディスプレイの画面解像度を取得。
 *
 *  このサンプルプログラムの実行結果例を以下に示します。
 *
 *  @verbinclude  OeDisplayResolution_OutputExample.txt
 *
 */

//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/oe.h>

//-----------------------------------------------------------------------------

//
//  メイン関数
//
extern "C" void nnMain()
{
    int width, height;

    // アプリケーション動作環境操作ライブラリを初期化。
    nn::oe::Initialize();

    // 現在のデフォルトディスプレイの画面解像度を取得。
    nn::oe::GetDefaultDisplayResolution( &width, &height );
    NN_LOG("Default Display Resolution : Width=%d, Height=%d\n", width, height);

    // デフォルトディスプレイの画面解像度が変更されたときのシステムイベントを取得。
    nn::os::SystemEvent defaultDisplayResolutionChangeEvent;
    nn::oe::GetDefaultDisplayResolutionChangeEvent( &defaultDisplayResolutionChangeEvent );

    for (;;)
    {
        // デフォルトディスプレイの画面解像度が変更されるのを待機。
        defaultDisplayResolutionChangeEvent.Wait();

        // 現在のデフォルトディスプレイの画面解像度を取得。
        nn::oe::GetDefaultDisplayResolution( &width, &height );
        NN_LOG("Default Display Resolution : Width=%d, Height=%d\n", width, height);
    }
}
