﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{NvnTutorial11.cpp,PageSampleNvnTutorial11}
 *
 * @brief
 *  This tutorial shows how to load a Jpeg image from a file
 *  and compress it at runtime through a shader.
 */

/**
 * @page PageSampleNvnTutorial11 NVN Tutorial 11: Runtime Texture Compression
 * @tableofcontents
 *
 * @brief
 *  This tutorial shows how to load a Jpeg image from a file
 *  and compress it at runtime through a shader.
 *
 * @section PageSampleNvnTutorial11_SectionBrief Overview
 *  This tutorial shows how to load a Jpeg image from a file
 *  and compress it at runtime through a shader. The Jpeg data
 *  is used to create a linear NVN texture by running it through
 *  the Jpeg decoder and wrapping a memory pool around the resulting
 *  buffer. This texture is compressed into a DXT1 texture by running
 *  it through a shader containing the DXT1 compression algorithm.
 *  In order to render to the compressed texture, a texture view
 *  with a format that can be render to must be used. In the case
 *  DXT1 compression the texture format needs to be NVN_FORMAT_RGBA16UI.
 *
 * @subsection PageSampleNvnTutorial11_SectionExpectedOutput Expected Output
 * @image html NvnTutorial11.png
 *
 * @section PageSampleNvnTutorial11_SectionFileStructure File Structure
 *  The main tutorial file and other files that make up the tutorial as well as the Visual Studio solutions can be found at
 *  @link ../../../Samples/Sources/Applications/NvnTutorial11RuntimeTextureCompression Samples/Sources/Applications/NvnTutorial11RuntimeTextureCompression @endlink
 *
 *  The tutorial library which contains common code shared by the tutorials can be found at
 *  Samples/Sources/Libraries/NvnTutorial
 *
 * @section PageSampleNvnTutorial11_SectionNecessaryEnvironment System Requirements
 *  No extra system requirements.
 *
 * @section PageSampleNvnTutorial11_SectionHowToOperate Operation Procedure
 *  This sample runs on its own without any additional input. Hitting escape on Windows
 *  or touching the touch screen on NX will exit the program.
 *
 * @section PageSampleNvnTutorial11_SectionPrecaution Precautions
 *  None.
 *
 * @section PageSampleNvnTutorial11_SectionHowToExecute Execution Procedure
 *  Build the Visual Solution and run it.
 *
 * @section PageSampleNvnTutorial11_SectionDetail Description
 *
 * @subsection PageSampleNvnTutorial11_SectionSampleProgram Sample Program
 *  Below is the source code for the main tutorial file for this sample.
 *
 *  NvnTutorial11.cpp
 *  @includelineno NvnTutorial11.cpp
 *
 * @subsection PageSampleNvnTutorial11_SectionSampleDetail Sample Program Description
 *  This application renders a spinning cube texture with the compressed
 *  version of the loaded in Jpeg file.
 */

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvn/nvn_FuncPtrImpl.h>
#include <nn/util/util_Matrix.h>
#include <nn/fs.h>
#include <nn/image/image_JpegDecoder.h>

#ifdef _WIN32
    #ifndef WIN32_LEAN_AND_MEAN
    #define WIN32_LEAN_AND_MEAN
    #endif
    #ifndef NOMINMAX
    #define NOMINMAX
    #endif
    #include <nn/nn_Windows.h>
#endif

#include <nvntutorial/TutorialBaseClass.h>
#include <nvntutorial/MemoryPool.h>
#include <nvntutorial/ManagedCommandBuffer.h>
#include <nvntutorial/FrameBufferManager.h>
#include <nvntutorial/AssetFileDataHolder.h>
#include <nvntutorial/AssetFileLoadingHelper.h>
#include <nvntutorial/UniformBufferManager.h>
#include <nvntutorial/TextureIDManager.h>

static const int    g_NumChunks = 2;
static const size_t g_CommandMemoryChunkSize = 2048;
static const size_t g_ControlMemoryChunkSize = 2048;

static const int    g_NumColorBuffers = 2;

class RuntimeTextureCompression : public TutorialBaseClass
{
public:
    RuntimeTextureCompression();
    virtual ~RuntimeTextureCompression();
    virtual void Init(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow);
    virtual void Shutdown();

    virtual void Draw(uint64_t millisec);
    virtual void Resize(int width, int height);

private:
    static void NVNAPIENTRY DebugLayerCallback(
        NVNdebugCallbackSource source,
        NVNdebugCallbackType type,
        int id,
        NVNdebugCallbackSeverity severity,
        const char* message,
        void* pUser
    );

    void UpdateScene(int frame);
    void UpdateRenderTargets();
    void UpdateUniformBuffers();
    void UpdateSceneCommandBuffer();

    void SetupJpegTexture();
    void CompressJpegTexture();
    void LoadJpegImage(const std::string& fileName, void** outImageData, size_t& outSize, uint32_t& outWidth, uint32_t& outHeight);

    NVNdevice                    m_Device;
    NVNqueue                     m_Queue;
    void*                        m_pQueueMemory;

    NVNtextureBuilder            m_RenderTargetBuilder;
    NVNtexture*                  m_RenderTargets[g_NumColorBuffers];
    NVNtexture*                  m_pDepthTarget;

    int                          m_ScreenWidth;
    int                          m_ScreenHeight;

    MemoryPool*                  m_pRenderTargetMemoryPool;

    MemoryPool*                  m_pShaderScratchMemoryPool;
    size_t                       m_ShaderScratchSize;

    MemoryPool                   m_JpegMemoryPool;
    NVNtexture                   m_JpegTexture;
    NVNsampler                   m_JpegSampler;
    uint32_t                     m_JpegSamplerID;
    uint32_t                     m_JpegTextureID;
    NVNtextureHandle             m_JpegTextureHandle;
    uint32_t                     m_JpegWidth;
    uint32_t                     m_JpegHeight;

    MemoryPool                   m_DXT1MemoryPool;
    NVNtexture                   m_DXT1Texture;
    NVNsampler                   m_DXT1Sampler;
    uint32_t                     m_DXT1SamplerID;
    uint32_t                     m_DXT1TextureID;
    NVNtextureHandle             m_DXT1TextureHandle;

    NVNtextureView               m_DXT1CompressView;

    NVNblendState                m_BlendState;
    NVNchannelMaskState          m_ChannelMaskState;
    NVNcolorState                m_ColorState;
    NVNdepthStencilState         m_DepthStencilState;
    NVNmultisampleState          m_MultisampleState;
    NVNpolygonState              m_PolygonState;


    TextureIDManager*            m_pTextureIDManager;
    AssetFileLoadingHelper*      m_pAssetLoader;

    AssetFileDataHolder*         m_pCubeDataHolder;

    AssetFileDataHolder*         m_pCompressShaderDataHolder;

    ManagedCommandBuffer*        m_pSceneManagedCommandBuffer;
    NVNcommandHandle             m_SceneCommandHandle;

    ManagedCommandBuffer*        m_pTextureCompressManagedCommandBuffer;
    NVNcommandHandle             m_TextureCompressCommandHandle;

    NVNwindow*                   m_pWindow;
    NVNwindowBuilder             m_WindowBuilder;
    int                          m_CurrentWindowIndex;
    NVNsync                      m_WindowSync;

    UniformBufferManager*        m_pUniformBufferManager;

    ManagedUniformBuffer*        m_pCubeUniformBufferVS;
    ManagedUniformBuffer*        m_pCubeUniformBufferFS;

    FrameBufferedSyncManager*    m_pFrameBufferedSyncManager;

    size_t                       m_ColorTargetSize;
    size_t                       m_DepthTargetSize;

    nn::util::Matrix4x4fType     m_ProjMtx;
    nn::util::Matrix4x4fType     m_CamMtx;
    nn::util::Matrix4x4fType     m_WorldMtx;
};

/*
 * RuntimeTextureCompression Constructor
 * -------------------------------------
 * Sets up default values for member data.
 */
RuntimeTextureCompression::RuntimeTextureCompression() :
    m_pQueueMemory(NULL),
    m_pDepthTarget(NULL),
    m_ScreenWidth(0),
    m_ScreenHeight(0),
    m_pRenderTargetMemoryPool(NULL),
    m_pShaderScratchMemoryPool(NULL),
    m_ShaderScratchSize(0),
    m_JpegSamplerID(0),
    m_JpegTextureID(0),
    m_JpegTextureHandle(0),
    m_JpegWidth(0),
    m_JpegHeight(0),
    m_DXT1SamplerID(0),
    m_DXT1TextureID(0),
    m_DXT1TextureHandle(0),
    m_pTextureIDManager(NULL),
    m_pAssetLoader(NULL),
    m_pCubeDataHolder(NULL),
    m_pCompressShaderDataHolder(NULL),
    m_pSceneManagedCommandBuffer(NULL),
    m_pTextureCompressManagedCommandBuffer(NULL),
    m_pWindow(NULL),
    m_CurrentWindowIndex(-1),
    m_pUniformBufferManager(NULL),
    m_pCubeUniformBufferVS(NULL),
    m_pCubeUniformBufferFS(NULL),
    m_pFrameBufferedSyncManager(NULL),
    m_ColorTargetSize(0),
    m_DepthTargetSize(0)
{
    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        m_RenderTargets[i] = NULL;
    }

    nn::util::MatrixIdentity(&m_CamMtx);
}

/*
 * RuntimeTextureCompression Destructor
 * ------------------------------------
 * Empty destructor.
 */
RuntimeTextureCompression::~RuntimeTextureCompression()
{
}

/*
 * RuntimeTextureCompression::Init
 * -------------------------------
 * Initialize NVN, load asset files, and create objects needed for the
 * application to run.
 */
void RuntimeTextureCompression::Init(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow)
{
    NN_ASSERT(pLoader != NULL, "Bootstrap loader function pointer is NULL\n");

        /* Load the function pointer to get other nvn function pointers. */
    pfnc_nvnDeviceGetProcAddress = (PFNNVNDEVICEGETPROCADDRESSPROC) (*pLoader)("nvnDeviceGetProcAddress");

    if (pfnc_nvnDeviceGetProcAddress == NULL)
    {
            /* This can happen if an NVN driver is not installed on a Windows PC. */
        NN_ASSERT(0, "BootstrapLoader failed to find nvnDeviceGetProcAddress");
    }

        /*
         * Load function pointers for creating nvn device
         */
    nvnLoadCProcs( NULL, pfnc_nvnDeviceGetProcAddress );

        /* Check the API version. */
    int majorVersion, minorVersion;
    nvnDeviceGetInteger( NULL, NVN_DEVICE_INFO_API_MAJOR_VERSION, &majorVersion );
    nvnDeviceGetInteger( NULL, NVN_DEVICE_INFO_API_MINOR_VERSION, &minorVersion );

    if( majorVersion != NVN_API_MAJOR_VERSION || minorVersion < NVN_API_MINOR_VERSION )
    {
        NN_ASSERT( 0, "NVN SDK not supported by current driver." );
    }

        /* If debug or develop is enabled, turn on NVN's debug layer. */
    int deviceFlags = 0;
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT;
#endif

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
            /*
             * This can fail for a few reasons; the most likely on Horizon is
             * insufficent device memory.
             */
        NN_ASSERT(0, "nvnDeviceInitialize");
    }

        /*
         * Ensure function pointers are all loaded properly
         */
    nvnLoadCProcs(&m_Device, pfnc_nvnDeviceGetProcAddress );

        /* Setup the debug callback for the debug layer. */
    if (deviceFlags & NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT)
    {
        nvnDeviceInstallDebugCallback(
            &m_Device,
            reinterpret_cast<PFNNVNDEBUGCALLBACKPROC>(&DebugLayerCallback),
            NULL, // For testing purposes; any pointer is OK here.
            NVN_TRUE // NVN_TRUE = Enable the callback.
            );
    }

        /* Initialize the queue. */
    NVNqueueBuilder queueBuilder;
    nvnQueueBuilderSetDevice(&queueBuilder, &m_Device);
    nvnQueueBuilderSetDefaults(&queueBuilder);
    nvnQueueBuilderSetComputeMemorySize(&queueBuilder, 0);

    int minQueueCommandMemorySize = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_QUEUE_COMMAND_MEMORY_MIN_SIZE, &minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandMemorySize(&queueBuilder, minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandFlushThreshold(&queueBuilder, minQueueCommandMemorySize);

    size_t neededQueueMemorySize = nvnQueueBuilderGetQueueMemorySize(&queueBuilder);

    if ((neededQueueMemorySize % NVN_MEMORY_POOL_STORAGE_GRANULARITY) != 0)
    {
        NN_ASSERT(0, "Memory size reported for queue is not the proper granularity");
    }

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    m_pQueueMemory = NULL;
#else
    m_pQueueMemory = AlignedAllocate(neededQueueMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
#endif

    nvnQueueBuilderSetQueueMemory(&queueBuilder, m_pQueueMemory, neededQueueMemorySize);

    if (nvnQueueInitialize(&m_Queue, &queueBuilder) == false)
    {
        NN_ASSERT(0, "nvnQueueInitialize failed");
    }

        /* Create a managed Command Buffer */
    m_pSceneManagedCommandBuffer = new ManagedCommandBuffer(&m_Device, g_CommandMemoryChunkSize, g_ControlMemoryChunkSize, g_NumChunks);

    m_pTextureCompressManagedCommandBuffer = new ManagedCommandBuffer(&m_Device, g_CommandMemoryChunkSize, g_ControlMemoryChunkSize, 1);

        /* Set up the texture builder for the render target. */
    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);
    m_ColorTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_DEPTH32F);
    m_DepthTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

        /* Allocate the render target memory. */
    m_pRenderTargetMemoryPool = new MemoryPool();
    m_pRenderTargetMemoryPool->Init(NULL,
                                    m_ColorTargetSize * g_NumColorBuffers + m_DepthTargetSize,
                                    NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT,
                                    &m_Device);

    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);

        /* Create the texture ID manager. */
    m_pTextureIDManager = new TextureIDManager(&m_Device);

        /* Create the asset loader. */
    m_pAssetLoader = new AssetFileLoadingHelper(&m_Device, m_pTextureIDManager);

    size_t cacheSize = 0;
    nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT(result.IsSuccess());

    char* mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

    result = nn::fs::MountRom("rom", mountRomCacheBuffer, cacheSize);
    NN_ASSERT(result.IsSuccess(), "Failed to mount rom.");

    m_pCubeDataHolder = m_pAssetLoader->LoadAssetFile("texturedCube.out");
    m_pCubeDataHolder->SetupAttributeStatesNVN(SimpleTexturedModel::Attributes::GetAttributeLocation);

    m_pCompressShaderDataHolder = m_pAssetLoader->LoadAssetFile("textureCompress.out");

    m_ShaderScratchSize = std::max(m_pCubeDataHolder->GetProgramData()[0]->m_ShaderScratchMemorySize,
                                   m_pCompressShaderDataHolder->GetProgramData()[0]->m_ShaderScratchMemorySize);

    if (m_ShaderScratchSize != 0)
    {
        m_pShaderScratchMemoryPool = new MemoryPool();
        m_pShaderScratchMemoryPool->Init(nullptr,
                                         m_ShaderScratchSize,
                                         NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT,
                                         &m_Device);
    }

        /*! Initialize the window sync. */
    if (!nvnSyncInitialize(&m_WindowSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize window sync");
    }

        /* Setup the window builder. */
    nvnWindowBuilderSetDefaults(&m_WindowBuilder);
    nvnWindowBuilderSetDevice(&m_WindowBuilder, &m_Device);
    nvnWindowBuilderSetNativeWindow(&m_WindowBuilder, nativeWindow);

        /* Create the uniform buffer manager. */
    m_pUniformBufferManager = new UniformBufferManager(&m_Device, 64 * 1024);

        /* Create and set up the sync manager. */
    m_pFrameBufferedSyncManager = new FrameBufferedSyncManager(&m_Device, &m_Queue);
    m_pFrameBufferedSyncManager->RegisterMemoryManager(m_pUniformBufferManager);
    m_pFrameBufferedSyncManager->RegisterMemoryManager(m_pSceneManagedCommandBuffer);

        /* Setup the Jpeg texture. */
    SetupJpegTexture();

    nn::fs::Unmount("rom");
    delete[] mountRomCacheBuffer;

    CompressJpegTexture();

        /* Setup the uniform buffers for the cube. */
    m_pCubeUniformBufferVS = m_pUniformBufferManager->CreateUniformBuffer(sizeof(SimpleTexturedModel::BlockVSUniformBlockData));
    m_pCubeUniformBufferFS = m_pUniformBufferManager->CreateUniformBuffer(sizeof(SimpleTexturedModel::BlockFSUniformBlockData));

        /* Setup the render states for the cube. */
    nvnDepthStencilStateSetDepthTestEnable(&m_DepthStencilState, NVN_TRUE);
    nvnDepthStencilStateSetDepthWriteEnable(&m_DepthStencilState, NVN_TRUE);
    nvnDepthStencilStateSetDepthFunc(&m_DepthStencilState, NVN_DEPTH_FUNC_LESS);

    nvnPolygonStateSetFrontFace(&m_PolygonState, NVNfrontFace::NVN_FRONT_FACE_CCW);
    nvnPolygonStateSetCullFace(&m_PolygonState, NVNface::NVN_FACE_BACK);
    nvnPolygonStateSetPolygonMode(&m_PolygonState, NVN_POLYGON_MODE_FILL);
}//NOLINT(impl/function_size)

/*
 * RuntimeTextureCompression::CompressJpegTexture
 * ----------------------------------------------
 * Compresses the Jpeg texture to the DXT1 format.
 */
void RuntimeTextureCompression::CompressJpegTexture()
{
        /*
         * Setup a texture builder to create a DXT1 compressed
         * texture. This texture will be rendered into using a
         * specific NVN texture view and a shader to perform the
         * compression.
         */
    NVNtextureBuilder textureBuilder;
    nvnTextureBuilderSetDefaults(&textureBuilder);
    nvnTextureBuilderSetDevice(&textureBuilder, &m_Device);
    nvnTextureBuilderSetFormat(&textureBuilder, NVN_FORMAT_RGBA_DXT1);
    nvnTextureBuilderSetTarget(&textureBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetSize2D(&textureBuilder, m_JpegWidth, m_JpegHeight);

        /* Get the size of the compressed texture. */
    size_t compressedSize = nvnTextureBuilderGetStorageSize(&textureBuilder);

        /* Create a memory pool for the compressed texture. */
    m_DXT1MemoryPool.Init(NULL, compressedSize, NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, &m_Device);

    nvnTextureBuilderSetStorage(&textureBuilder, m_DXT1MemoryPool.GetMemoryPool(), 0);

        /* Initialize the tetxure. */
    if (nvnTextureInitialize(&m_DXT1Texture, &textureBuilder) == NVN_FALSE)
    {
        NN_ASSERT(0, "Failed to initialize compressed texture.");
    }

        /* Grab the texture ID. */
    m_DXT1TextureID = m_pTextureIDManager->RegisterTexture(&m_JpegTexture);

        /* Create a sampler and get its ID. */
    NVNsamplerBuilder samplerBuilder;
    nvnSamplerBuilderSetDevice(&samplerBuilder, &m_Device);
    nvnSamplerBuilderSetDefaults(&samplerBuilder);

    NVNboolean samplerSuccess = nvnSamplerInitialize(&m_DXT1Sampler, &samplerBuilder);
    NN_ASSERT(samplerSuccess, "Failed to initialize sampler\n");
    m_DXT1SamplerID = m_pTextureIDManager->RegisterSampler(&m_DXT1Sampler);

        /* Create the texture handle for the compressed texture. */
    m_DXT1TextureHandle = nvnDeviceGetTextureHandle(&m_Device, m_DXT1TextureID, m_DXT1SamplerID);

        /*
         * Create a texture view for the compressed texture. Using
         * this texture view, the compressed texture can be set as
         * a render target for the compression shader. The compression
         * shader will read 4x4 blocks of pixels from the Jpeg texture,
         * perform the compression algorithm, and write it out to the
         * DXT1 texture through the view. For the DXT1 compression
         * algorithm, the texture view need to specifically be of the
         * type NVN_FORMAT_RGBA16UI.
         */
    nvnTextureViewSetDefaults(&m_DXT1CompressView);
    nvnTextureViewSetFormat(&m_DXT1CompressView, NVN_FORMAT_RGBA16UI);

        /* Set default states for the draw. */
    nvnDepthStencilStateSetDefaults(&m_DepthStencilState);
    nvnPolygonStateSetDefaults(&m_PolygonState);
    nvnMultisampleStateSetDefaults(&m_MultisampleState);
    nvnColorStateSetDefaults(&m_ColorState);
    nvnChannelMaskStateSetDefaults(&m_ChannelMaskState);
    nvnBlendStateSetDefaults(&m_BlendState);

        /* Record the command buffer for the compression. */
    m_pTextureCompressManagedCommandBuffer->BeginRecording();
    {
        NVNcommandBuffer* pCommandBuffer = m_pTextureCompressManagedCommandBuffer->GetCommandBuffer();

            /*
             * Calculate the width and height of the compressed
             * texture with the texture views format.
             */
        const int blockSize = 4;
        int compressedWidth  = (m_JpegWidth + blockSize - 1) / blockSize;
        int compressedHeight = (m_JpegHeight + blockSize - 1) / blockSize;

        m_pTextureIDManager->SetSamplerPool(pCommandBuffer);
        m_pTextureIDManager->SetTexturePool(pCommandBuffer);

        NVNtexture* renderTarget = &m_DXT1Texture;
        NVNtextureView* targetView = &m_DXT1CompressView;

            /*
             * Set the compressed texture as the render
             * target along with the texture view.
             */
        nvnCommandBufferSetRenderTargets(pCommandBuffer, 1, &renderTarget, &targetView, NULL, NULL);
        nvnCommandBufferSetViewport(pCommandBuffer, 0, 0, compressedWidth, compressedHeight);
        nvnCommandBufferSetScissor(pCommandBuffer, 0, 0, compressedWidth, compressedHeight);

        if (m_pShaderScratchMemoryPool)
        {
            nvnCommandBufferSetShaderScratchMemory(pCommandBuffer, m_pShaderScratchMemoryPool->GetMemoryPool(), 0, m_ShaderScratchSize);
        }

            /* Bind the shader program. */
        std::vector<NVNProgramData*>&   programData = m_pCompressShaderDataHolder->GetProgramData();
        programData[0]->BindShaderProgram(pCommandBuffer);

            /* Bind the Jpeg texture to the proper shader stage/slot. */
        nvnCommandBufferBindTexture(pCommandBuffer, NVN_SHADER_STAGE_FRAGMENT, 0, m_JpegTextureHandle);

            /* Bind the render states. */
        nvnCommandBufferBindBlendState(pCommandBuffer, &m_BlendState);
        nvnCommandBufferBindChannelMaskState(pCommandBuffer, &m_ChannelMaskState);
        nvnCommandBufferBindColorState(pCommandBuffer, &m_ColorState);
        nvnCommandBufferBindDepthStencilState(pCommandBuffer, &m_DepthStencilState);
        nvnCommandBufferBindMultisampleState(pCommandBuffer, &m_MultisampleState);
        nvnCommandBufferBindPolygonState(pCommandBuffer, &m_PolygonState);

            /*
             * Draw a screen space quad to perform the compression
             * algorithm with. No vertex buffer is bound as the vertex
             * shader outputs positions based on the vertex ID alone.
             */
        nvnCommandBufferDrawArrays(pCommandBuffer, NVN_DRAW_PRIMITIVE_TRIANGLE_STRIP, 0, 4);

            /*
             * Create a barrier in the command buffer to make sure that
             * the texture is finished being compressed before being used.
             */
        nvnCommandBufferBarrier(pCommandBuffer, NVNbarrierBits::NVN_BARRIER_ORDER_FRAGMENTS_BIT | NVNbarrierBits::NVN_BARRIER_INVALIDATE_TEXTURE_BIT);
    }
    m_TextureCompressCommandHandle = m_pTextureCompressManagedCommandBuffer->EndRecording();

        /* Submit the command buffer for the compression. */
    nvnQueueSubmitCommands(&m_Queue, 1, &m_TextureCompressCommandHandle);
}

/*
 * RuntimeTextureCompression::SetupJpegTexture
 * -------------------------------------------
 * Creates an NVN texture out of the loaded in jpeg data.
 */
void RuntimeTextureCompression::SetupJpegTexture()
{
    void* jpegImageData = NULL;
    size_t jpegImageSize = 0;

        /* Loads the jpeg texture data from a file. */
    LoadJpegImage("rom:/Checkerboard.jpg", &jpegImageData, jpegImageSize, m_JpegWidth, m_JpegHeight);

        /* Create the memory pool for the texture. */
    m_JpegMemoryPool.Init(jpegImageData, jpegImageSize, NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, &m_Device);

        /*
         * Setup a texture builder to create a texture out of the
         * linear texture data.
         */
    NVNtextureBuilder textureBuilder;
    nvnTextureBuilderSetDefaults(&textureBuilder);
    nvnTextureBuilderSetDevice(&textureBuilder, &m_Device);
    nvnTextureBuilderSetTarget(&textureBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&textureBuilder, NVN_FORMAT_RGBA8);
    nvnTextureBuilderSetFlags(&textureBuilder, NVN_TEXTURE_FLAGS_LINEAR_BIT);
    nvnTextureBuilderSetSize2D(&textureBuilder, m_JpegWidth, m_JpegHeight);
    nvnTextureBuilderSetStride(&textureBuilder, m_JpegWidth * sizeof(nn::Bit32));
    nvnTextureBuilderSetStorage(&textureBuilder, m_JpegMemoryPool.GetMemoryPool(), 0);

        /* Initialize the texture. */
    if (nvnTextureInitialize(&m_JpegTexture, &textureBuilder) == NVN_FALSE)
    {
        NN_ASSERT(0, "Failed to initialize Jpeg texture.");
    }

        /* Get the texture id. */
    m_JpegTextureID = m_pTextureIDManager->RegisterTexture(&m_JpegTexture);

        /* Create a sampler. */
    NVNsamplerBuilder samplerBuilder;
    nvnSamplerBuilderSetDevice(&samplerBuilder, &m_Device);
    nvnSamplerBuilderSetDefaults(&samplerBuilder);

        /* Initialize the sampler. */
    if (nvnSamplerInitialize(&m_JpegSampler, &samplerBuilder) == NVN_FALSE)
    {
        NN_ASSERT(0, "Failed to initialize sampler\n");
    }

        /* Get the sampler ID. */
    m_JpegSamplerID = m_pTextureIDManager->RegisterSampler(&m_JpegSampler);

        /* Create the NVN texture handle. */
    m_JpegTextureHandle = nvnDeviceGetTextureHandle(&m_Device, m_JpegTextureID, m_JpegSamplerID);
}

/*
 * RuntimeTextureCompression::LoadJpegImage
 * ----------------------------------------
 * Loads in a Jpeg image from a file and decodes it into
 * a buffer of linear texture data.
 */
void RuntimeTextureCompression::LoadJpegImage(const std::string& fileName, void** outImageData, size_t& outSize, uint32_t& outWidth, uint32_t& outHeight)
{
        /* Open the jpeg file. */
    nn::fs::FileHandle file;
    nn::Result fileResult = nn::fs::OpenFile(&file, fileName.c_str(), nn::fs::OpenMode_Read);

    NN_ASSERT(fileResult.IsSuccess(), "Failed to open image file.");

        /* Get the file size. */
    int64_t fileSize;
    fileResult = nn::fs::GetFileSize(&fileSize, file);
    NN_ASSERT(fileResult.IsSuccess(), "Failed to get image file size.");

        /* Allocate a buffer for the file. */
    void* fileData = malloc(static_cast<size_t>(fileSize));

        /* Read the jpeg file. */
    fileResult = nn::fs::ReadFile(file, 0, fileData, static_cast<size_t>(fileSize));
    NN_ASSERT(fileResult.IsSuccess(), "Failed to read image file.");

        /* Close the file. */
    nn::fs::CloseFile(file);

        /* Create the Jpeg decoder. */
    nn::image::JpegDecoder jpegDecoder;

        /* Give the loaded Jpeg data to the decoder. */
    jpegDecoder.SetImageData(fileData, static_cast<size_t>(fileSize));

        /* Analyze the data. */
    nn::image::JpegStatus jpegStatus = jpegDecoder.Analyze();

    switch (jpegStatus)
    {
    case nn::image::JpegStatus_WrongFormat:
        NN_ASSERT(0, "Wrong Format - Jpeg Decoder Analyze");
        break;
    case nn::image::JpegStatus_UnsupportedFormat:
        NN_ASSERT(0, "Unsupported Format - Jpeg Decoder Analyze");
        break;
    default:
        NN_ASSERT(jpegStatus == nn::image::JpegStatus_Ok, "Jpeg decoder analyze failed.");
    }

        /* Grab the dimension of the Jpeg. */
    nn::image::Dimension imageDimension = jpegDecoder.GetAnalyzedDimension();

        /* Allocate a work buffer for the decoder. */
    size_t workBufferSize = jpegDecoder.GetAnalyzedWorkBufferSize();
    void* decodeWorkBuffer = malloc(workBufferSize);

        /* Output the width/height. */
    outWidth = imageDimension.width;
    outHeight = imageDimension.height;

        /* Size of the raw data. */
    outSize = imageDimension.width * imageDimension.height * sizeof(nn::Bit32);

        /* Allocate memory for the future NVN memory pool. */
    *outImageData = AlignedAllocate(outSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);

        /* Decode the Jpeg data. */
    jpegStatus = jpegDecoder.Decode(*outImageData, outSize, sizeof(nn::Bit32), decodeWorkBuffer, workBufferSize);

    switch (jpegStatus)
    {
    case nn::image::JpegStatus_WrongFormat:
        NN_ASSERT(0, "Wrong Format - Jpeg Decoder Decode");
        break;
    case nn::image::JpegStatus_UnsupportedFormat:
        NN_ASSERT(0, "Unsupported Format - Jpeg Decoder Decode");
        break;
    case nn::image::JpegStatus_OutOfMemory:
        NN_ASSERT(0, "Out of Memory - Jpeg Decoder Decode");
        break;
    default:
        NN_ASSERT(jpegStatus == nn::image::JpegStatus_Ok, "Jpeg decode failed.");
    }
}

/*
 * RuntimeTextureCompression::Shutdown
 * -----------------------------------
 * This method cleans up all nvn objects and dynamically allocated memory.
 */
void RuntimeTextureCompression::Shutdown()
{
    nvnQueueFinish(&m_Queue);
    nvnSyncFinalize(&m_WindowSync);

    if (m_pSceneManagedCommandBuffer != NULL)
    {
        delete m_pSceneManagedCommandBuffer;
        m_pSceneManagedCommandBuffer = NULL;
    }

    if (m_pTextureCompressManagedCommandBuffer != NULL)
    {
        delete m_pTextureCompressManagedCommandBuffer;
        m_pTextureCompressManagedCommandBuffer = NULL;
    }

    if (m_pWindow != NULL)
    {
        nvnWindowFinalize(m_pWindow);
        delete m_pWindow;
        m_pWindow = NULL;
    }

    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        if (m_RenderTargets[i] != NULL)
        {
            nvnTextureFinalize(m_RenderTargets[i]);
            delete m_RenderTargets[i];
            m_RenderTargets[i] = NULL;
        }
    }

    if (m_pDepthTarget != NULL)
    {
        nvnTextureFinalize(m_pDepthTarget);
        delete m_pDepthTarget;
        m_pDepthTarget = NULL;
    }

    nvnTextureFinalize(&m_JpegTexture);
    nvnSamplerFinalize(&m_JpegSampler);

    nvnTextureFinalize(&m_DXT1Texture);
    nvnSamplerFinalize(&m_DXT1Sampler);

    if (m_pTextureIDManager != NULL)
    {
        delete m_pTextureIDManager;
        m_pTextureIDManager = NULL;
    }

    if (m_pUniformBufferManager != NULL)
    {
        delete m_pUniformBufferManager;
        m_pUniformBufferManager = NULL;
    }

    if (m_pFrameBufferedSyncManager != NULL)
    {
        delete m_pFrameBufferedSyncManager;
        m_pFrameBufferedSyncManager = NULL;
    }

    if (m_pCompressShaderDataHolder != NULL)
    {
        delete m_pCompressShaderDataHolder;
        m_pCompressShaderDataHolder = NULL;
    }

    if (m_pCubeDataHolder != NULL)
    {
        delete m_pCubeDataHolder;
        m_pCubeDataHolder = NULL;
    }

    if (m_pAssetLoader != NULL)
    {
        delete m_pAssetLoader;
        m_pAssetLoader = NULL;
    }

    m_DXT1MemoryPool.Shutdown();
    m_JpegMemoryPool.Shutdown();

    if (m_pRenderTargetMemoryPool)
    {
        m_pRenderTargetMemoryPool->Shutdown();
        delete m_pRenderTargetMemoryPool;
        m_pRenderTargetMemoryPool = NULL;
    }

    if (m_pShaderScratchMemoryPool)
    {
        m_pShaderScratchMemoryPool->Shutdown();
        delete m_pShaderScratchMemoryPool;
        m_pShaderScratchMemoryPool = NULL;
        m_ShaderScratchSize = 0;
    }

    nvnQueueFinalize(&m_Queue);
    if (m_pQueueMemory)
    {
        AlignedDeallocate(m_pQueueMemory);
        m_pQueueMemory = NULL;
    }

    nvnDeviceFinalize(&m_Device);
}

/*
 * RuntimeTextureCompression::UpdateScene
 * --------------------------------------
 * Updates the world matrix for the spinning cube.
 */
void RuntimeTextureCompression::UpdateScene(int frame)
{
    nn::util::Matrix4x3fType simpleTranslateMat;
    nn::util::MatrixIdentity(&simpleTranslateMat);
    nn::util::Vector3fType simpleTranslateVector;
    nn::util::VectorSet(&simpleTranslateVector, -0.0f, 0.0f, -3.0f);
    nn::util::MatrixSetTranslate(&simpleTranslateMat, simpleTranslateVector);

    nn::util::Matrix4x3fType simpleRotateXMat;
    nn::util::MatrixIdentity(&simpleRotateXMat);
    nn::util::Vector3fType simpleRotateXVector;
    nn::util::VectorSet(&simpleRotateXVector, 30.0f * 3.1459f / 180.0f, 0.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&simpleRotateXMat, simpleRotateXVector);

    nn::util::Matrix4x3fType simpleRotateYMat;
    nn::util::MatrixIdentity(&simpleRotateYMat);
    nn::util::Vector3fType simpleRotateYVector;
    nn::util::VectorSet(&simpleRotateYVector, 0.0f, frame * 3.14159f / 180.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&simpleRotateYMat, simpleRotateYVector);

    nn::util::Matrix4x3fType simpleModelMat;
    nn::util::Matrix4x3fType tempMat1;
    nn::util::MatrixMultiply(&tempMat1, simpleRotateYMat, simpleRotateXMat);
    nn::util::MatrixMultiply(&simpleModelMat, tempMat1, simpleTranslateMat);

    nn::util::MatrixConvert(&m_WorldMtx, simpleModelMat);
}

/*
 * RuntimeTextureCompression::UpdateRenderTargets
 * ----------------------------------------------
 * Gets the index of the current render target from the NVNwindow
 * and records part of a command buffer that sets it up to be rendered to.
 */
void RuntimeTextureCompression::UpdateRenderTargets()
{
        /* Get next render target to be used */
    NVNwindowAcquireTextureResult result = nvnWindowAcquireTexture(m_pWindow, &m_WindowSync, &m_CurrentWindowIndex);

    NN_ASSERT(result == NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS);

        /* Record the command buffer to set and clear the target. */
    NVNcommandBuffer* pCommandBuffer = m_pSceneManagedCommandBuffer->GetCommandBuffer();
    nvnCommandBufferSetRenderTargets(pCommandBuffer, 1, &m_RenderTargets[m_CurrentWindowIndex], NULL, m_pDepthTarget, NULL);

    nvnCommandBufferSetScissor(pCommandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);
    nvnCommandBufferSetViewport(pCommandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);

    float clearColor[4] = { 0.4f, 0.55f, 0.6f, 1.0f };
    nvnCommandBufferClearColor(pCommandBuffer, 0, clearColor, NVN_CLEAR_COLOR_MASK_RGBA);
    nvnCommandBufferClearDepthStencil(pCommandBuffer, 1.0, NVN_TRUE, 0, 0);
}

/*
 * RuntimeTextureCompression::UpdateUniformBuffers
 * -----------------------------------------------
 * Updates the uniform buffers for rendering the spinning cube.
 */
void RuntimeTextureCompression::UpdateUniformBuffers()
{
        /* Update Vertex shader uniform buffer. */

        /* Get the current mapped point fom the managed buffer. */
    void* vertexPoolMap = m_pCubeUniformBufferVS->GetMappedPointer();

    SimpleTexturedModel::BlockVSUniformBlockData vertexShaderData;

    nn::util::Float4x4 temp;
    nn::util::MatrixStore(&temp, m_WorldMtx);
    vertexShaderData.SetUniform_u_modelMtx(*reinterpret_cast<float(*)[16]>(&temp));

    nn::util::MatrixIdentity(&m_CamMtx);
    nn::util::MatrixStore(&temp, m_CamMtx);
    vertexShaderData.SetUniform_u_viewMtx(*reinterpret_cast<float(*)[16]>(&temp));

    nn::util::MatrixStore(&temp, m_ProjMtx);
    vertexShaderData.SetUniform_u_projMtx(*reinterpret_cast<float(*)[16]>(&temp));

        /* Copy in the uniform data. */
    memcpy(vertexPoolMap, &vertexShaderData, sizeof(SimpleTexturedModel::BlockVSUniformBlockData));

        /* Update Fragment shader uniform buffer. */

        /* Get the current mapped point fom the managed buffer. */
    void* fragmentPoolMap = m_pCubeUniformBufferFS->GetMappedPointer();

    SimpleTexturedModel::BlockFSUniformBlockData fragmentShaderData;
    fragmentShaderData.SetUniform_u_bindlessTex(m_DXT1TextureHandle);

        /* Copy in the uniform data. */
    memcpy(fragmentPoolMap, &fragmentShaderData, sizeof(SimpleTexturedModel::BlockFSUniformBlockData));
}

/*
 * RuntimeTextureCompression::UpdateSceneCommandBuffer
 * ---------------------------------------------------
 * Creates the command buffer for rendering the scene.
 */
void RuntimeTextureCompression::UpdateSceneCommandBuffer()
{
    m_pSceneManagedCommandBuffer->BeginRecording();
    {
            /* Update the render targets. */
        UpdateRenderTargets();

        NVNcommandBuffer* pCommandBuffer = m_pSceneManagedCommandBuffer->GetCommandBuffer();

            /* Bind the texture and sampler descriptor pools. */
        m_pTextureIDManager->SetSamplerPool(pCommandBuffer);
        m_pTextureIDManager->SetTexturePool(pCommandBuffer);

            /* Sets the scissor rectangle and viewport to the full screen */
        nvnCommandBufferSetScissor(pCommandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);
        nvnCommandBufferSetViewport(pCommandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);

            /* Clears the currently set render target at a given index. */
        float clear_color[4] = { 0.4f, 0.55f, 0.6f, 1.0f };
        nvnCommandBufferClearColor(pCommandBuffer, 0, clear_color, NVN_CLEAR_COLOR_MASK_RGBA);
        nvnCommandBufferClearDepthStencil(pCommandBuffer, 1.0, NVN_TRUE, 0, 0);

            /* Bind the render state objects. */
        nvnCommandBufferBindBlendState(pCommandBuffer, &m_BlendState);
        nvnCommandBufferBindChannelMaskState(pCommandBuffer, &m_ChannelMaskState);
        nvnCommandBufferBindColorState(pCommandBuffer, &m_ColorState);
        nvnCommandBufferBindDepthStencilState(pCommandBuffer, &m_DepthStencilState);
        nvnCommandBufferBindMultisampleState(pCommandBuffer, &m_MultisampleState);
        nvnCommandBufferBindPolygonState(pCommandBuffer, &m_PolygonState);
        nvnCommandBufferSetSampleMask(pCommandBuffer, static_cast<uint32_t>(~0));

        std::vector<NVNModelData*>&   modelData = m_pCubeDataHolder->GetModelData();
        std::vector<NVNProgramData*>& programData = m_pCubeDataHolder->GetProgramData();

        ShaderTypes::ShaderType shaderType = programData[0]->m_ShaderType;

        Model* model = &modelData[0]->m_Model;
        NVNbufferAddress vboAddr = nvnBufferGetAddress(&modelData[0]->m_VertexBuffer);

            /* Bind the vertex buffer(s). */
        for (size_t j = 0; j < model->m_VertexAttributes.size(); ++j)
        {
            VertexAttribute& attr = model->m_VertexAttributes[j];
            nvnCommandBufferBindVertexBuffer(pCommandBuffer, attr.m_Location, vboAddr + modelData[0]->m_VertexAttributeBufferOffsets[j], attr.m_DataSize);
        }

            /* Bind the uniform buffers. */
        nvnCommandBufferBindUniformBuffer(pCommandBuffer,
                                          NVN_SHADER_STAGE_VERTEX,
                                          ShaderTypes::BlockVS_GetBinding(NVN_SHADER_STAGE_VERTEX, shaderType),
                                          m_pCubeUniformBufferVS->GetCurrentBufferAddress(),
                                          sizeof(SimpleTexturedModel::BlockVSUniformBlockData));

        nvnCommandBufferBindUniformBuffer(pCommandBuffer,
                                          NVN_SHADER_STAGE_FRAGMENT,
                                          ShaderTypes::BlockFS_GetBinding(NVN_SHADER_STAGE_FRAGMENT, shaderType),
                                          m_pCubeUniformBufferFS->GetCurrentBufferAddress(),
                                          sizeof(SimpleTexturedModel::BlockFSUniformBlockData));

            /* Bind the vertex states. */
        nvnCommandBufferBindVertexAttribState(pCommandBuffer, static_cast<int>(modelData[0]->m_VertexAttributeStates.size()), &modelData[0]->m_VertexAttributeStates[0]);
        nvnCommandBufferBindVertexStreamState(pCommandBuffer, static_cast<int>(modelData[0]->m_VertexStreamStates.size()), &modelData[0]->m_VertexStreamStates[0]);

        if (m_pShaderScratchMemoryPool)
        {
            nvnCommandBufferSetShaderScratchMemory(pCommandBuffer, m_pShaderScratchMemoryPool->GetMemoryPool(), 0, m_ShaderScratchSize);
        }

            /* Bind the shader program. */
        programData[0]->BindShaderProgram(pCommandBuffer);

            /* Draw the primitives. */
        NVNindexType indexType = (NVNindexType)model->m_IndexData.m_IndexType;
        uint32_t numIndices = model->m_IndexData.m_DataSize / model->m_IndexData.m_Stride;
        nvnCommandBufferDrawElements(pCommandBuffer,
                                     (NVNdrawPrimitive)model->m_NvnDrawPrimitiveType,
                                     indexType,
                                     numIndices,
                                     nvnBufferGetAddress(&modelData[0]->m_IndexBuffer));
    }

    m_SceneCommandHandle = m_pSceneManagedCommandBuffer->EndRecording();
}

/*
 * RuntimeTextureCompression::Draw
 * -------------------------------
 * Updates and draws the scene.
 */
void RuntimeTextureCompression::Draw(uint64_t millisec)
{
    NN_UNUSED(millisec);
    static int s_Frame = 0;

        /* Get the current render target and setup/submit a command buffer to set it. */
    UpdateScene(s_Frame);
    UpdateUniformBuffers();
    UpdateSceneCommandBuffer();

        /* Insert a fence in the sync manager. */
    m_pFrameBufferedSyncManager->InsertFence();

        /*
         * Wait on sync that was received in UpdateRenderTargets now that we are
         * actually ready to use the render target
         */
    nvnSyncWait(&m_WindowSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        /* Submit command buffer for switching render targets */
    nvnQueueSubmitCommands(&m_Queue, 1, &m_SceneCommandHandle);

        /* Present the render target to the screen. */
    nvnQueuePresentTexture(&m_Queue, m_pWindow, m_CurrentWindowIndex);

        /* Swap memory for command buffer and uniform buffers */
    m_pFrameBufferedSyncManager->SwapPools();

    ++s_Frame;
}

/*
 * RuntimeTextureCompression::Resize
 * ---------------------------------
 * This method is called everytime the window is resized and is passed
 * the new size of the window. It frees the old render target and creates a new
 * render target with the new screen size. A new set of commands for the command buffer
 * is recorded.
 */
void RuntimeTextureCompression::Resize(int width, int height)
{
    m_ScreenWidth = width;
    m_ScreenHeight = height;

        /* Check for the window being minimized or having no visible surface. */
    if (width == 0 || height == 0)
    {
        return;
    }

        /* If it's the first time Resize is called, allocate the NVNwindow. */
    if (m_pWindow == NULL)
    {
        m_pWindow = new NVNwindow;
    }
        /*
         * Otherwise finalize (free) the NVNwindow used for the previous window size.
         * The NVNWindow must be finalized before a render target it owns is finalized.
         */
    else
    {
        nvnWindowFinalize(m_pWindow);
    }

        /* Set up the builder for the render target. */
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, m_ScreenWidth, m_ScreenHeight);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);

    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
            /* If it's the first time Resize is called, allocate the render target. */
        if(m_RenderTargets[i] == NULL)
        {
            m_RenderTargets[i] = new NVNtexture;
        }
            /* Otherwise finalize (free) the render target used for the previous window size. */
        else
        {
            nvnTextureFinalize(m_RenderTargets[i]);
        }

        nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * i);

            /* Create the texture using the current state of the texture builder. */
        nvnTextureInitialize(m_RenderTargets[i], &m_RenderTargetBuilder);
    }

    if (m_pDepthTarget == NULL)
    {
        m_pDepthTarget = new NVNtexture;
    }
    else
    {
        nvnTextureFinalize(m_pDepthTarget);
    }

        /* Initialize depth buffer for render target. */
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, m_ScreenWidth, m_ScreenHeight);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_DEPTH32F);
    nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * g_NumColorBuffers);
    nvnTextureInitialize(m_pDepthTarget, &m_RenderTargetBuilder);

        /* Pass off the render targets to the window. */
    nvnWindowBuilderSetTextures(&m_WindowBuilder, g_NumColorBuffers, m_RenderTargets);
    nvnWindowInitialize(m_pWindow, &m_WindowBuilder);

        /* Remake the projection matrix with the new screen size. */
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&m_ProjMtx, 60.0f * 3.14159f / 180.0f, static_cast<float>(m_ScreenWidth) / static_cast<float>(m_ScreenHeight), 1.0f, 1000.0f);
}

// static
void RuntimeTextureCompression::DebugLayerCallback(
    NVNdebugCallbackSource source,
    NVNdebugCallbackType type,
    int id,
    NVNdebugCallbackSeverity severity,
    const char* message,
    void* pUser
)
{
    NN_ASSERT(pUser == NULL);

    NN_LOG("NVN Debug Layer Callback:\n");
    NN_LOG("  source:       0x%08x\n", source);
    NN_LOG("  type:         0x%08x\n", type);
    NN_LOG("  id:           0x%08x\n", id);
    NN_LOG("  severity:     0x%08x\n", severity);
    NN_LOG("  message:      %s\n", message);

    NN_ASSERT(0, "NVN Debug layer callback hit");
}

TutorialBaseClass* t()
{
    static RuntimeTextureCompression tut;
    return (&tut);
}

extern "C" void nnMain()
{
    TutorialRun();
}
