﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>

#include <nn/vi.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

/**
* @brief 描画処理を行う GraphicsSystem クラスです。
*
* @details 画面の描画処理を行うクラスです。
*/
class GraphicsSystem
{
    NN_DISALLOW_COPY(GraphicsSystem);
    NN_DISALLOW_MOVE(GraphicsSystem);

public:
    GraphicsSystem() NN_NOEXCEPT;

    void SetApplicationHeap(nn::mem::StandardAllocator* pValue) NN_NOEXCEPT;
    nn::gfx::Device& GetDevice() NN_NOEXCEPT;
    nn::gfx::CommandBuffer& GetCommandBuffer() NN_NOEXCEPT;
    nn::gfx::util::DebugFontTextWriter& GetDebugFont() NN_NOEXCEPT;

    int GetDisplayWidth() const NN_NOEXCEPT;
    int GetDisplayHeight() const NN_NOEXCEPT;
    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    void BeginDraw() NN_NOEXCEPT;
    void EndDraw() NN_NOEXCEPT;

    int RegisterTextureViewSlot(nn::gfx::DescriptorSlot* pOutValue,
        const nn::gfx::TextureView& textureView
        ) NN_NOEXCEPT;
    int RegisterSamplerSlot(nn::gfx::DescriptorSlot* pOutValue,
        const nn::gfx::Sampler& sampler) NN_NOEXCEPT;
    void UnregisterTextureViewSlot(nn::gfx::DescriptorSlot* pOutValue,
        const nn::gfx::TextureView& textureView
        ) const NN_NOEXCEPT;
    void UnregisterSamplerSlot(nn::gfx::DescriptorSlot* pOutValue,
        const nn::gfx::Sampler& sampler) const NN_NOEXCEPT;

    void InitializeDebugFont() NN_NOEXCEPT;
    void FinalizeDebugFont() NN_NOEXCEPT;

    void InitializeColorBuffer(
        nn::gfx::TextureView* pColorTextureView,
        nn::gfx::ColorTargetView* pColorTargetView,
        nn::gfx::Texture* pColorBuffer
    ) NN_NOEXCEPT;

    void FinalizeColorBuffer(
        nn::gfx::TextureView* pColorTextureView,
        nn::gfx::ColorTargetView* pColorTargetView,
        nn::gfx::Texture* pColorBuffer
    ) NN_NOEXCEPT;

    void InitializeSampler(nn::gfx::Sampler* pSampler) NN_NOEXCEPT;
    void FinalizeSampler(nn::gfx::Sampler* pSampler) NN_NOEXCEPT;

    nn::gfx::SwapChain* GetSwapChain() NN_NOEXCEPT;
    nn::gfx::MemoryPool* GetMemoryPoolInvisible() NN_NOEXCEPT;
    nn::gfx::MemoryPool* GetMemoryPool() NN_NOEXCEPT;
    void InitializeBlurBuffer() NN_NOEXCEPT;

private:
    static void OutOfCommandMemoryEventCallback(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT;
    static void OutOfControlMemoryEventCallback(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT;

    void InitializeDisplay() NN_NOEXCEPT;
    void InitializeDevice() NN_NOEXCEPT;
    void InitializeMemoryPool() NN_NOEXCEPT;
    void InitializeMemoryPoolInvisible() NN_NOEXCEPT;
    void InitializeSwapChain() NN_NOEXCEPT;
    void InitializeQueue() NN_NOEXCEPT;
    void InitializeFence() NN_NOEXCEPT;
    void InitializeDepthStencilTexture() NN_NOEXCEPT;
    void InitializeDepthStencilView() NN_NOEXCEPT;
    void InitializeBufferDescriptorPool() NN_NOEXCEPT;
    void InitializeTextureDescriptorPool() NN_NOEXCEPT;
    void InitializeSamplerDescriptorPool() NN_NOEXCEPT;
    void InitializeCommandBuffer() NN_NOEXCEPT;
    void InitializeViewportScissorState() NN_NOEXCEPT;
    void InitializeBlendState() NN_NOEXCEPT;
    void InitializeDepthStencilState() NN_NOEXCEPT;
    void InitializeRasterizerState() NN_NOEXCEPT;
    void FinalizeMemoryPool() NN_NOEXCEPT;
    void FinalizeMemoryPoolInvisible() NN_NOEXCEPT;
    void FinalizeViewportScissorState() NN_NOEXCEPT;
    void FinalizeBlendState() NN_NOEXCEPT;
    void ResetCommandBuffer() NN_NOEXCEPT;
    void AddCommandMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
        ) NN_NOEXCEPT;
    void AddControlMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
        ) NN_NOEXCEPT;

private:
    nn::mem::StandardAllocator* m_pApplicationHeap;

    nn::gfx::MemoryPool m_MemoryPool;
    void* m_pMemoryPool;
    ptrdiff_t m_MemoryPoolOffset;
    ptrdiff_t m_MemoryPoolCommandOffset;

    nn::gfx::MemoryPool m_MemoryPoolInvisible;
    void* m_pMemoryPoolInvisible;
    ptrdiff_t m_MemoryPoolInvisibleOffset;

    nn::gfx::DescriptorPool m_BufferDescriptorPool;

    nn::gfx::DescriptorPool m_TextureDescriptorPool;
    int m_TextureSlotCount;

    nn::gfx::DescriptorPool m_SamplerDescriptorPool;
    int m_SamplerSlotCount;

    nn::vi::Display* m_pDisplay;
    int m_DisplayWidth;
    int m_DisplayHeight;
    nn::vi::Layer* m_pLayer;
    nn::gfx::Device m_Device;
    nn::gfx::SwapChain m_SwapChain;
    nn::gfx::Queue m_Queue;
    nn::gfx::Fence m_Fence;
    nn::gfx::Texture m_ColorTargetTexture;
    nn::gfx::Texture m_DepthStencilTexture;
    nn::gfx::DepthStencilView m_DepthStencilView;
    nn::gfx::CommandBuffer m_CommandBuffer;
    nn::gfx::ViewportScissorState m_ViewportScissorState;
    nn::gfx::BlendState m_BlendState;
    nn::gfx::DepthStencilState m_DepthStencilState;
    nn::gfx::RasterizerState m_RasterizerState;

    nn::util::BytePtr m_DebugFontHeap;
    nn::gfx::util::DebugFontTextWriter m_DebugFont;

    std::vector<void*> m_CommandBufferControlMemoryChunks;

    nn::gfx::Texture m_ColorBuffer;
    nn::gfx::ColorTargetView m_ColorTargetView;
    nn::gfx::TextureView m_ColorTextureView;
    int m_ColorTextureDescriptorIndex;

    nn::gfx::Sampler m_Sampler;
    int m_BufferDescriptorIndex;
    int m_SamplerDescriptorIndex;

    nn::gfx::DescriptorSlot m_ColorDescriptorSlot;
    nn::gfx::DescriptorSlot m_SamplerSlot;

    nn::gfx::Texture m_BlurColorBuffer;
    nn::gfx::ColorTargetView m_BlurColorTargetView;
    nn::gfx::TextureView m_BlurColorTextureView;

    size_t m_VisiblePoolMemorySize;
    size_t m_InvisiblePoolMemorySize;
};
