﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <nn/nn_Assert.h>
#include <nn/nfp.h>
#include "Graphics.h"
#include "TagUtility.h"
#include "WriteState.h"

namespace nns { namespace nfp { namespace {

    char g_Buffer[64];
    const int LabeledValueOffset = 20;

    inline int ToElapsedFrame(const ApplicationData& data) NN_NOEXCEPT
    {
        return data.frame - data.transitFrame;
    }

    inline void WriteLabeledValue(
            int x, int y,
            const char* label,
            const char* value
        ) NN_NOEXCEPT
    {
        WriteText(x, y, label);
        WriteText(x + LabeledValueOffset, y, value);
    }

    void WriteTagInfo(
            int& x, int& y,
            const nn::nfp::TagInfo& tagInfo
        ) NN_NOEXCEPT
    {
        const nn::Bit8* uid = reinterpret_cast<const nn::Bit8*>(tagInfo.tagId.uid);
        char* buffer = g_Buffer;
        std::sprintf(buffer, ":");
        buffer += 1;
        for(int i = 0; i < tagInfo.tagId.length && i < sizeof(tagInfo.tagId.uid); i++)
        {
            std::sprintf(buffer, " %02X", uid[i]);
            buffer += 3;
        }
        WriteLabeledValue(x, y++, "UID", g_Buffer);

        std::sprintf(g_Buffer, ": %d", tagInfo.protocol);
        WriteLabeledValue(x, y++, "PROTOCOL", g_Buffer);

        std::sprintf(g_Buffer, ": %d", tagInfo.type);
        WriteLabeledValue(x, y++, "TYPE", g_Buffer);
    }

    void WriteModelInfo(
            int& x, int& y,
            const nn::nfp::ModelInfo& modelInfo
        ) NN_NOEXCEPT
    {
        std::sprintf(g_Buffer, ": %s", GetCharacterName(modelInfo.characterId));
        WriteLabeledValue(x, y++, "CHARACTER", g_Buffer);

        std::sprintf(g_Buffer, ": %d", modelInfo.numberingId);
        WriteLabeledValue(x, y++, "NUMBERING ID", g_Buffer);

        std::sprintf(g_Buffer, ": %d", modelInfo.seriesId);
        WriteLabeledValue(x, y++, "SERIES ID", g_Buffer);

        std::sprintf(g_Buffer, ": %d", modelInfo.nfpType);
        WriteLabeledValue(x, y++, "NFP TYPE", g_Buffer);
    }

    void WriteCommonInfo(
            int& x, int& y,
            const nn::nfp::CommonInfo& commonInfo
        ) NN_NOEXCEPT
    {
        std::sprintf(g_Buffer, ": %d", commonInfo.nfpVersion);
        WriteLabeledValue(x, y++, "NFP VERSION", g_Buffer);

        std::sprintf(g_Buffer, ": %d", commonInfo.applicationAreaSize);
        WriteLabeledValue(x, y++, "APP AREA SIZE", g_Buffer);

        std::sprintf(g_Buffer, ": %04d-%02d-%02d",
                     commonInfo.lastWriteDate.year,
                     commonInfo.lastWriteDate.month,
                     commonInfo.lastWriteDate.day);
        WriteLabeledValue(x, y++, "WRITE DATE", g_Buffer);

        std::sprintf(g_Buffer, ": %d", commonInfo.writeCounter);
        WriteLabeledValue(x, y++, "WRITE COUNTER", g_Buffer);
    }

    void WriteRegisterInfo(
            int& x, int& y,
            const nn::nfp::RegisterInfo& registerInfo
        ) NN_NOEXCEPT
    {
        std::sprintf(g_Buffer, ": %04d-%02d-%02d",
                     registerInfo.registerDate.year,
                     registerInfo.registerDate.month,
                     registerInfo.registerDate.day);
        WriteLabeledValue(x, y++, "REGISTER DATE", g_Buffer);

        std::sprintf(g_Buffer, ": %d", registerInfo.fontRegion);
        WriteLabeledValue(x, y++, "FONT REGION", g_Buffer);

        const nn::Bit8* nickname = reinterpret_cast<const nn::Bit8*>(registerInfo.nickname);
        std::sprintf(g_Buffer, ": %02X %02X %02X %02X %02X %02X %02X %02X",
                     nickname[0], nickname[1],
                     nickname[2], nickname[3],
                     nickname[4], nickname[5],
                     nickname[6], nickname[7]);
        WriteLabeledValue(x, y++, "NICKNAME", g_Buffer);
    }

    void WriteStateNone(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;
        WriteText(x, y++, "A: initialize nfp library");
        WriteText(x, y++, "Y: exit");
    }

    void WriteStateInitializing(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Initializing", ToElapsedFrame(data));
    }

    void WriteStateInitialized(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Connecting", ToElapsedFrame(data));

        ++y;
        WriteText(x, y++, "B: cancel");
    }

    void WriteStateDeviceListing(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Connecting", ToElapsedFrame(data));

        ++y;
        WriteText(x, y++, "B: cancel");
    }

    void WriteStateDeviceInitialize(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Connecting", ToElapsedFrame(data));
    }

    void WriteStateDeviceSelect(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteText(x, y++, "A: start detection");
        WriteText(x, y++, "B: finalize nfp library");

        ++y;
        for( int i = 0; i < data.deviceCount; i++)
        {
            std::sprintf(g_Buffer, "    NpadId 0x%02X", data.npadIds[i]);
            const bool selected = i == data.deviceIndex;
            WriteText(x, y++, g_Buffer, selected ? Color::Orange : Color::White);
        }
    }

    void WriteStateStartDetection(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Starting", ToElapsedFrame(data));
    }

    void WriteStateStopDetection(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Suspending", ToElapsedFrame(data));
    }

    void WriteStateTagRelease(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Finished", ToElapsedFrame(data));

        ++y;
        WriteText(x, y++, "Please remove the tag");
    }

    void WriteStateTagSearch(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Searching", ToElapsedFrame(data));

        ++y;
        std::sprintf(g_Buffer, "Please touch the tag to NpadId 0x%02X", data.npadIds[data.deviceIndex]);
        WriteText(x, y++, g_Buffer);
        WriteText(x, y++, "B: cancel");
    }

    void WriteStateDeviceCheck(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Checking", ToElapsedFrame(data));
    }

    void WriteStateTagRead(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Reading", ToElapsedFrame(data));

        ++y;
        WriteText(x, y++, "Do not remove the tag during access");
    }

    void WriteStateTagCheck(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Checking", ToElapsedFrame(data));

        ++y;
        WriteText(x, y++, "Do not remove the tag during access");
    }

    void WriteStateTagWrite(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Writing", ToElapsedFrame(data));

        ++y;
        WriteText(x, y++, "Do not remove the tag during access");
    }

    void WriteStateTagEnabled(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteText(x, y++, "A: update app counter");
        WriteText(x, y++, "B: cancel");

        ++y;
        WriteTagInfo(x, y, data.tagInfo);

        ++y;
        WriteModelInfo(x, y, data.modelInfo);

        ++y;
        WriteCommonInfo(x, y, data.commonInfo);

        ++y;
        WriteRegisterInfo(x, y, data.registerInfo);

        ++y;
        std::sprintf(g_Buffer, ": %u", data.counter);
        WriteLabeledValue(x, y++, "APP COUNTER", g_Buffer);
    }

    void WriteStateNeedCreate(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;
        WriteErrorText(x, y++,  "Application data not found.");

        ++y;
        WriteText(x, y++, "A: create");
        WriteText(x, y++, "B: cancel");
    }

    void WriteStateNeedRestore(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;
        WriteErrorText(x, y++, "Could not read tag data.");

        ++y;
        WriteText(x, y++, "A: restore");
        #if defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        WriteText(x, y++, "X: restore (amiibo settings)");
        #endif // defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        WriteText(x, y++, "B: cancel");
    }

    void WriteStateNeedFormat(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;
        WriteErrorText(x, y++,  "Could not read tag data.");
        ++y;
        WriteErrorText(x, y++,  "You should format this tag using amiibo settings");
        WriteErrorText(x, y++,  "found in system settings.");
        ++y;
        WriteErrorText(x, y++,  "Alternatively, if you have backup data stored on");
        WriteErrorText(x, y++,  "other console, you can use it to restore the");
        WriteErrorText(x, y++,  "original data.");
        ++y;
        WriteText(x, y++, "B: cancel");
    }

    void WriteStateNeedRegister(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;
        WriteErrorText(x, y++,  "Registration data not found.");

        ++y;
        #if defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        WriteText(x, y++, "X: register (amiibo settings)");
        #endif // defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        WriteText(x, y++, "B: cancel");
    }

    void WriteStateNeedDelete(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;
        WriteErrorText(x, y++,  "Already exists other application's data.");

        ++y;
        #if defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        WriteText(x, y++, "X: delete (amiibo settings)");
        #endif // defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        WriteText(x, y++, "B: cancel");
    }

    void WriteStateAmiiboNotSupported(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;
        WriteErrorText(x, y++,  "The amiibo can not be used with this application.");
        ++y;
        WriteText(x, y++, "B: cancel");
    }

    void WriteStateAmiiboSettings(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Running Amiibo Settings", ToElapsedFrame(data));
    }

    void WriteStateFinalize(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Finalizing", ToElapsedFrame(data));
    }

    void WriteStateError(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteErrorText(x, y, "ERROR: ");
        x += 7;

        nn::Result result = data.lastResult;
        if( result <= nn::nfp::ResultNfcDeviceNotFound() )
        {
            WriteErrorText(x, y++, "nfc device not found");
        }
        else if( result <= nn::nfp::ResultNfcDisabled() )
        {
            WriteErrorText(x, y++, "nfc disabled");
        }
        else if( result <= nn::nfp::ResultNeedRetry() )
        {
            WriteErrorText(x, y++, "please retry");
        }
        else if( result <= nn::nfp::ResultNeedRestart() ||
                 result <= nn::nfp::ResultAccessIdMisMatch() ||
                 result <= nn::nfp::ResultNotBroken() )
        {
            WriteErrorText(x, y++, "keep applying the same tag");
        }
        else if( result <= nn::nfp::ResultNeedRegister() )
        {
            WriteErrorText(x, y++, "need register");
        }
        else if( result <= nn::nfp::ResultNeedCreate() )
        {
            WriteErrorText(x, y++, "need create");
        }
        else if( result <= nn::nfp::ResultNeedRestore() )
        {
            WriteErrorText(x, y++, "need restore");
        }
        else if( result <= nn::nfp::ResultNeedFormat() )
        {
            WriteErrorText(x, y++, "need format");
        }
        else if( result <= nn::nfp::ResultAlreadyCreated() )
        {
            WriteErrorText(x, y++, "already created");
        }
        else if( result <= nn::nfp::ResultNotSupported() ||
                 result <= nn::nfp::ResultInvalidFormatVersion() )
        {
            WriteErrorText(x, y++, "this tag is not supported");
        }
        else if( result <= nn::nfp::ResultMaxNfcDeviceActivated() )
        {
            WriteErrorText(x, y++, "nfc device already activated");
        }
        else if( result <= nn::nfp::ResultConflictFunction() )
        {
            WriteErrorText(x, y++, "nfc device is conflicted");
        }
        else if( result <= nn::nfp::ResultNotUpdated() )
        {
            WriteErrorText(x, y++, "not been updated by amiibo setting");
        }
        else
        {
            WriteErrorText(x, y++, "unexpected");
        }

        x  = 1;
        y += 1;
        WriteText(x, y++, "A: restart");
        WriteText(x, y++, "B: finalize nfp library");
    }

    void WriteStateExit(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Exit", ToElapsedFrame(data));
    }

    typedef void (*WriteStateFunc)( const ApplicationData& data );
    struct WriteStateEntry
    {
        State           state;
        WriteStateFunc  func;
    };

    const WriteStateEntry WriteStateTable[] = {
        {   State_None,                 WriteStateNone              },
        {   State_Initializing,         WriteStateInitializing      },
        {   State_Initialized,          WriteStateInitialized       },
        {   State_DeviceListing,        WriteStateDeviceListing     },
        {   State_DeviceInitialize,     WriteStateDeviceInitialize  },
        {   State_DeviceSelect,         WriteStateDeviceSelect      },
        {   State_StartDetection,       WriteStateStartDetection    },
        {   State_StopDetection,        WriteStateStopDetection     },
        {   State_TagRelease,           WriteStateTagRelease        },
        {   State_TagSearch,            WriteStateTagSearch         },
        {   State_DeviceCheck,          WriteStateDeviceCheck       },
        {   State_TagRead,              WriteStateTagRead           },
        {   State_TagCheck,             WriteStateTagCheck          },
        {   State_TagWrite,             WriteStateTagWrite          },
        {   State_TagEnabled,           WriteStateTagEnabled        },
        {   State_NeedCreate,           WriteStateNeedCreate        },
        {   State_NeedRestore,          WriteStateNeedRestore       },
        {   State_NeedFormat,           WriteStateNeedFormat        },
        {   State_NeedRegister,         WriteStateNeedRegister      },
        {   State_NeedDelete,           WriteStateNeedDelete        },
        {   State_AmiiboNotSupported,   WriteStateAmiiboNotSupported },
        {   State_AmiiboSettings,       WriteStateAmiiboSettings    },
        {   State_Finalize,             WriteStateFinalize          },
        {   State_Error,                WriteStateError             },
        {   State_Exit,                 WriteStateExit              },
    };
    const int WriteStateCountMax = sizeof(WriteStateTable) / sizeof(WriteStateTable[0]);
    NN_STATIC_ASSERT( WriteStateCountMax == static_cast<int>(State_CountMax) );

    WriteStateFunc GetWriteStateFunc(State state)
    {
        const int index = static_cast<int>(state);
        NN_ASSERT( index >= 0 && index < WriteStateCountMax );
        NN_ASSERT( WriteStateTable[index].state == state );
        return WriteStateTable[index].func;
    }

}}} // end of namespace nns::nfp::unnamed

namespace nns { namespace nfp {

    void WriteState(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        // 状態に応じて画面に描画する内容を変更します。
        (*GetWriteStateFunc(data.state))(data);
    }

}} // end of namespace nns::nfp
