﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Log.h>
#include <nn/nfp.h>

#define NFPDEMO_ENABLE_LOG
//#define NFPDEMO_ENABLE_AMIIBO_SETTING

#if defined( NFPDEMO_ENABLE_LOG )
#define NFPDEMO_LOG(...)     NN_LOG("NFPDEMO: " __VA_ARGS__)
#else
#define NFPDEMO_LOG(...)
#endif // defined( NFPDEMO_ENABLE_LOG )

namespace nns { namespace nfp {

    /*!
        @brief      NFC デバイスの最大数です。
     */
    const int DeviceCountMax = nn::nfp::DeviceCountMax;

    /*!
        @brief      このアプリケーションのプレイヤーの最大数です。
     */
    const int PlayerCountMax = 4;

    /*!
        @brief      amiibo のアプリケーション専用領域にアクセスするために必要な ID です。
        @details    ID が異なるアプリケーション専用領域はオープンできません。
     */
    const nn::Bit32 AccessId = 0x0FF41E00;

    /*!
        @brief      アプリケーションの状態を表す定数です。
     */
    enum State
    {
        State_None,                     //!< NFP ライブラリを初期化していない状態です。
        State_Initializing,             //!< NFP ライブラリを初期中です。
        State_Initialized,              //!< NFP ライブラリが利用可能な状態です。
        State_DeviceListing,            //!< NFC デバイスリストを取得中です。
        State_Idle,                     //!< 次回の NFC デイバスリスト取得までの待機状態です。

        State_CountMax                  //!< アプリケーションの状態の総数です。
    };

    /*!
        @brief      プレイヤーの状態を表す定数です。
     */
    enum PlayerState
    {
        PlayerState_None,                     //!< NFC デバイスを初期化していない状態です。
        PlayerState_DeviceInitialize,         //!< NFC デバイスを初期化中です。
        PlayerState_AttachEvent,              //!< イベントを設定中です。
        PlayerState_Idle,                     //!< タグの検知を開始の指示を待っている状態です。
        PlayerState_StartDetection,           //!< タグの検知を開始しています。
        PlayerState_StopDetection,            //!< タグの検知を停止しています。
        PlayerState_TagRelease,               //!< タグの喪失待機中です。
        PlayerState_TagSearch,                //!< タグを検索中です。
        PlayerState_DeviceCheck,              //!< NFC デバイスの状態を確認中です。
        PlayerState_TagRead,                  //!< タグを読み込み中です。
        PlayerState_TagCheck,                 //!< タグを確認中です。
        PlayerState_TagWrite,                 //!< タグを書き込み中です。
        PlayerState_TagEnabled,               //!< 利用可能なタグの情報を取得できた状態です。
        PlayerState_NeedCreate,               //!< アプリケーション領域を作成する必要があります。
        PlayerState_NeedRestore,              //!< タグの復旧が必要です。
        PlayerState_NeedFormat,               //!< タグの初期化が必要です。
        PlayerState_NeedRegister,             //!< 初期登録情報を登録する必要があります。
        PlayerState_NeedDelete,               //!< アプリケーション領域を削除する必要があります。
        PlayerState_AmiiboNotSupported,       //!< アプリケーションが対応していない amiibo です。
        PlayerState_AmiiboSettings,           //!< amiibo 設定の起動中です。
        PlayerState_Error,                    //!< エラー状態です。

        PlayerState_CountMax                  //!< プレイヤーの状態の総数です。
    };

    /*!
        @brief      アプリケーションで扱うプレイヤーデータです。
     */
    struct PlayerData
    {
        bool                    isEnabled;      //!< 使用中かどうか
        bool                    isNfcDevice;    //!< NFC デバイスかどうか
        PlayerState             state;          //!< プレイヤーの状態です。
        nn::Result              lastResult;     //!< 最後に実行した NFP 関連の処理の結果です。
        uint32_t                transitFrame;   //!< 最後の状態遷移したときのフレーム番号です。
        nn::hid::NpadIdType     npadId;         //!< Npad ID です。
        nn::Bit8                ledPattern;     //!< プレイヤーランプの点灯パターンです。
        nn::nfp::TagInfo        tagInfo;        //!< タグの情報です。
        nn::nfp::ModelInfo      modelInfo;      //!< タグの NFP の型式情報です。
        nn::nfp::CommonInfo     commonInfo;     //!< タグの共有領域の情報です。
        nn::nfp::RegisterInfo   registerInfo;   //!< タグのオーナー登録情報です。
        uint32_t                counter;        //!< アプリケーション領域から読み込んだカウンタです。
    };

    /*!
        @brief      アプリケーションで扱うデータです。
     */
    struct ApplicationData
    {
        State                   state;          //!< アプリケーションの状態です。
        nn::Result              lastResult;     //!< 最後に実行した NFP 関連の処理の結果です。
        uint32_t                frame;          //!< フレーム番号です。
        uint32_t                transitFrame;   //!< 最後の状態遷移したときのフレーム番号です。
        PlayerData              playerData[ PlayerCountMax ]; //!< アプリケーションで扱うプレイヤーデータです。
    };

}} // end of namespace nns::nfp
