﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nfp.h>
#include <nn/nfp/nfp_PrivateTypes.h>
#include "TagAccessorBase.h"

namespace nns { namespace nfp {

    /*!
        @brief      スレッドに依頼する処理のコマンドです。
     */
    enum Command
    {
        Command_None = TagAccessorBase::Command_None,
        Command_InitializeSystem,
        Command_FinalizeSystem,
        Command_ListDevices,
        Command_AttachEvents,
        Command_StartDetection,
        Command_StopDetection,
        Command_Mount,
        Command_Unmount,
        Command_Format,
        Command_Flush,
        Command_Restore,
        Command_GetTagInfo,
        Command_GetCommonInfo,
        Command_GetModelInfo,
        Command_GetAdminInfo,
        Command_GetRegisterInfo,
        Command_SetRegisterInfo,
        Command_DeleteRegisterInfo,
        Command_DeleteApplicationArea,
        Command_ExistsApplicationArea,
        Command_GetDeviceState,
        Command_AttachAvailabilityChangeEvent,

        Command_CountMax
    };

    /*!
        @brief      NFP ライブラリの処理に非同期に行うクラスです。

        @details
                    NFP ライブラリで提供される API のほとんどは長時間処理をブロックしてしまいます。
                    このクラスは NFP ライブラリの API を別スレッドで実行する実装のサンプルです。

                    実行する処理に対応するコマンド識別子を @ref Enqueue() で登録し、
                    @ref Run() を呼びだすことで登録した処理をバックグラウンドで実行します。
                    登録できるコマンド識別子の最大数は @ref CommandCountMax です。

                    登録した処理が全て完了した場合、あるいは何らかのエラーが発生して中断された場合には
                    @ref Run() の引数に指定したイベントにシグナルが発行されます。
                    いくつの処理が実行されたかと最後の処理結果は、
                    @ref GetExecutedCount() と @ref GetLastResult() で取得することができます。

                    本クラスのメソッドはスレッドセーフではありませんので、一つのスレッドから呼び出すようにしてください。
     */
    class TagAccessor : public TagAccessorBase
    {
        NN_DISALLOW_COPY( TagAccessor );
        NN_DISALLOW_MOVE( TagAccessor );

    public:

        /*!
            @brief      TagAccessor のインスタンスを生成します。
            @return     TagAccessor のインスタンスです。
        */
        static TagAccessor& CreateInstance() NN_NOEXCEPT;

        /*!
            @brief      TagAccessor のインスタンスを取得します。
            @return     TagAccessor のインスタンスです。
        */
        static TagAccessor& GetInstance() NN_NOEXCEPT;

        /*!
            @brief      TagAccessor のインスタンスを破棄します。
        */
        static void DestroyInstance() NN_NOEXCEPT;

        /*!
            @brief      TagAccessor のインスタンスが存在するか判定します。
            @return     存在する場合は true を返します。
        */
        static bool HasInstance() NN_NOEXCEPT;

        /*!
            @brief      NFP ライブラリをシステムモードで初期化します。
            @return     登録された処理の位置です。
        */
        int InitializeSystem() NN_NOEXCEPT;

        /*!
            @brief      NFP ライブラリのシステムモードの使用を終了します。
            @return     登録された処理の位置です。

            @details    タグがマウントされている場合には自動的にアンマウントされ、検知も停止します。
                        NFC デバイスの状態が DeviceState_Active 状態または DeviceState_Mount 状態であれば、
                        @ref AttachEvents() で設定した Deactivate イベントが発生します。
        */
        int FinalizeSystem() NN_NOEXCEPT;

        /*!
            @brief      NFC デバイスのハンドルのリストを取得します。
            @param[out]     pOutBuffer      取得したハンドルのリストを格納するバッファへのポインタを指定します。
            @param[out]     pOutCount       実際に取得したハンドルの数を格納するバッファへのポインタを指定します。
            @param[in]      bufferCount     取得するハンドルの最大数（バッファの要素数）を指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int ListDevices(nn::nfp::DeviceHandle* pOutBuffer, int* pOutCount, int bufferCount) NN_NOEXCEPT;

        /*!
            @brief      タグの発見・喪失を通知するイベントを設定し、NFC デバイスの Npad ID を取得します。
            @param[out]     pOutNpadId          Npad ID です。
            @param[out]     pActivateEvent      タグ発見通知に使用するイベントを指定します。
            @param[out]     pDeactivateEvent    タグ喪失通知に使用するイベントを指定します。
            @param[in]      pDeviceHandle       NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。

                        引数として渡すイベントの初期化と解放は NFP ライブラリ側が担当します。
                        1 つのイベントで Activate と Deactivate の両方の通知を受け取ることはできません。
        */
        int AttachEvents(nn::hid::NpadIdType* pOutNpadId, nn::os::SystemEventType* pActivateEvent, nn::os::SystemEventType* pDeactivateEvent, const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグの検知を開始します。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int StartDetection(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグの検知を終了します。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。

                        タグがマウントされている場合には自動的にアンマウントされます。
                        DeviceState_Active 状態や DeviceState_Mount 状態であれば Deactivate イベントが発生します。
        */
        int StopDetection(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグをマウントします。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int Mount(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグをアンマウントします。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int Unmount(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグをフォーマットします。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int Format(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      マウント時に用意した内部バッファの内容をタグに書き込みます。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int Flush(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      データが破損したタグをフォーマットし、バックアップデータから復旧します。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int Restore(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグの情報を取得します。
            @param[out]     pOutInfo        TagInfo の取得先へのポインタです。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。

                        タグの情報はマウントする前であっても取得できます。
                        また、NFP 以外の NFC タグの情報を取得することも可能です。
        */
        int GetTagInfo(nn::nfp::TagInfo* pOutInfo, const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグの共用領域の情報を、マウント時に用意した内部バッファから取得します。
            @param[out]     pOutInfo        CommonInfo の取得先へのポインタです。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int GetCommonInfo(nn::nfp::CommonInfo* pOutInfo, const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグの ROM 領域の情報を、マウント時に用意した内部バッファから取得します。
            @param[out]     pOutInfo        ModelInfo の取得先へのポインタです。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int GetModelInfo(nn::nfp::ModelInfo* pOutInfo, const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグの管理情報を、マウント時に用意した内部バッファから取得します。
            @param[out]     pOutInfo        AdminInfo の取得先へのポインタです。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int GetAdminInfo(nn::nfp::AdminInfo* pOutInfo, const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグのオーナー登録情報を、マウント時に用意した内部バッファから取得します。
            @param[out]     pOutInfo        RegisterInfo の取得先へのポインタです。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int GetRegisterInfo(nn::nfp::RegisterInfo* pOutInfo, const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグのシステム領域の登録情報を、マウント時に用意した内部バッファに設定します。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @param[out]     pInfoPrivate    RegisterInfoPrivate の設定元へのポインタです。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int SetRegisterInfo(const nn::nfp::DeviceHandle* pDeviceHandle, const nn::nfp::RegisterInfoPrivate* pInfoPrivate) NN_NOEXCEPT;

        /*!
            @brief      タグの登録情報を削除します。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int DeleteRegisterInfo(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      タグのアプリケーション専用領域の情報を削除します。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int DeleteApplicationArea(const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      アプリケーション専用領域が存在するか判定します。
            @param[out]     pOutValue       存在するか有無の取得先へのポインタです。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int ExistsApplicationArea(bool* pOutValue, const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

         /*!
            @brief      デバイスの状態を取得します。
            @param[out]     pOutDeviceState DeviceState の取得先へのポインタです。
            @param[in]      pDeviceHandle   NFC デバイスのハンドルのポインタを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int GetDeviceState(nn::nfp::DeviceState* pOutDeviceState, const nn::nfp::DeviceHandle* pDeviceHandle) NN_NOEXCEPT;

        /*!
            @brief      デバイスの利用可/不可の変化を通知するイベントを設定します。
            @param[out]     pAvailabilityChangeEvent      通知に使用するイベントを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。

                        引数として渡すイベントの初期化と解放は NFP ライブラリ側が担当します。
        */
        int AttachAvailabilityChangeEvent(nn::os::SystemEventType* pAvailabilityChangeEvent) NN_NOEXCEPT;

        /*!
            @brief      指定番目の登録された処理の結果を取得します。
            @param[out]     pOutResult  結果を格納するバッファへのポインタを指定します。
            @param[in]      index       登録した処理の位置です。
            @return     指定番目の登録された処理の有無です。
                        指定番目の処理が実行されなかった場合には false を返します。
         */
        bool GetResult(nn::Result* pOutResult, int index) const NN_NOEXCEPT
        {
            return TagAccessorBase::GetNthResult(pOutResult, index);
        }

        /*!
            @brief      指定されたコマンド識別子の実行結果を取得します。
            @param[out]     pOutResult  結果を格納するバッファへのポインタを指定します。
            @param[in]      command     実行する処理を表すコマンド識別子です。
            @return     指定されたコマンド識別子の実行結果の有無です。
                        指定されたコマンド識別子が実行されなかった場合には false を返します。

            @details    複数回実行されたコマンド識別子を指定した場合には、先の実行結果を取得します。
         */
        bool GetResult(nn::Result* pOutResult, Command command) const NN_NOEXCEPT
        {
            return TagAccessorBase::GetCommandResult(pOutResult, static_cast<uint32_t>(command));
        }

    protected:

        /*!
            @brief      コマンド識別子で表される処理を実行します。
            @param[in]      command     実行する処理を表すコマンド識別子です。
            @param[in]      param       実行する処理に渡すパラメータです。
            @return     実行された処理の結果です。
         */
        virtual nn::Result Execute(uint32_t command, const ParameterBuffer& param) NN_NOEXCEPT NN_OVERRIDE;

    private:

        /*!
            @brief      NFP ライブラリの処理をバックグラウンドで実行するスレッドを開始します。
         */
        TagAccessor() NN_NOEXCEPT {}

        /*!
            @brief      スレッドが稼働中であれば終了します。
         */
        virtual ~TagAccessor() NN_NOEXCEPT NN_OVERRIDE {}

        //! 唯一存在する TagAccessor のインスタンスのポインタです。
        static TagAccessor* g_pInstance;

    };

}} // end of namespace nns::nfp
