﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{NfpCabinet.cpp,PageSampleNfpCabinet}

    @brief
    NFP ライブラリ amiibo 設定機能のサンプルプログラム
 */

/**
    @page PageSampleNfpCabinet NFP ライブラリ amiibo 設定機能のサンプル
    @tableofcontents

    @brief
    NFP ライブラリ amiibo 設定機能のサンプルプログラムの解説です。

    @section PageSampleNfpCabinet_SectionBrief 概要
    NFP ライブラリは、NFC（Near Field Communication）を介して、
    NFP タグ（NFP 用にフォーマットされた NFC タグ）の読み書きを行うライブラリです。
    NFP ライブラリを用いることで、アプリケーションと amiibo を連携させることが可能です。

    このサンプルプログラムは amiibo 設定機能を用いて、以下の amiibo 操作に対応しています。
    @n
    @li データが破損した amiibo の復旧
    @li amiibo のアプリケーション専用領域を削除
    @li amiibo のオーナー Mii とニックネームを登録
    @li amiibo の初期化

    サンプルプログラムでは固定のオーナー Mii とニックネームを登録します。

    @section PageSampleNfpCabinet_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/NfpCabinet
    Samples/Sources/Applications/NfpCabinet @endlink 以下にあります。

    @section PageSampleNfpCabinet_SectionNecessaryEnvironment 必要な環境
    事前に NoftWriter ツールを使用して NFP ライブラリで読み書き可能な NFP タグを用意してください。

    @section PageSampleNfpCabinet_SectionHowToOperate 操作方法
    サンプルプログラムを実行すると、NFP ライブラリの状態が表示されます。
    ゲームパッドの操作により、NFP ライブラリの状態を変化させることができます。

    @section PageSampleNfpCabinet_SectionPrecaution 注意事項
    サンプルプログラムを実行する開発機に、一度もタッチされたことがない amiibo のデータが破損している場合、
    バックアップデータが保存されていないため、データを復旧することができません。
    amiibo を初期化するか、バックアップデータが存在する開発機を使用して amiibo のデータを復旧してください。

    @section PageSampleNfpCabinet_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleNfpCabinet_SectionDetail 解説
    NfpCabinet では amiibo の状態によって異なる操作を実行します。
    特定の操作を狙って実行するためには、事前に以下の条件を満たす amiibo を用意しておく必要があります。

    @li アプリケーション専用領域の削除@n
    NfpSimple でアプリケーション専用領域を作成した amiibo を用意します。

    @li データが破損した amiibo の復旧@n
    データが破損した amiibo を用意します。

    @li オーナー Mii とニックネームの登録@n
    NoftWriter ツールで amiibo データリストを書き込んだばかりの amiibo を用意します。
    または、NfpCabinet から amiibo を初期化します。
 */

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/init.h>
#include <nn/la/la_Api.h>
#include "Graphics.h"
#include "HidController.h"

#include "ConfigSystem.h"
#include "UpdateStateSystem.h"
#include "WriteStateSystem.h"

namespace {

    // Graphics.cpp で malloc しているヒープの内訳
    //  VisiblePoolMemory       16 MB
    //  InvisiblePoolMemory     20 MB
    //  for CommandBuffer       32 MB
    //  nv::InitializeGraphics   8 MB
    //  for DebugFont            2 MB
    const size_t MemoryHeapSize         = 128 * 1024 * 1024;
    const size_t MallocHeapSize         =  96 * 1024 * 1024;

    // アプリケーションで扱うデータと状態です。
    nns::nfp::ApplicationData g_Data;

    // アプレットマネージャからの通知に使用するイベントオブジェクトです。
    nn::os::SystemEventType g_MessageEvent;

} // namespace

nn::applet::LibraryAppletHandle PrepareLibraryApplet() NN_NOEXCEPT
{
    NFPDEMO_LOG("Prepared LibraryApplet\n");

    nn::applet::LibraryAppletHandle appletHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateLibraryApplet(
            &appletHandle,
            nn::applet::AppletId_LibraryAppletCabinet,
            nn::applet::LibraryAppletMode_AllForeground
        ));

    // 共通パラメータ
    const uint16_t cMajorVersion = 0;
    const uint16_t cMinorVersion = 1;
    static nn::la::CommonArgumentsWriter s_CommonArg(cMajorVersion, cMinorVersion);
    s_CommonArg.PushToInChannel(appletHandle);

    // 固有パラメータ
    static nn::nfp::StartParamForAmiiboSettings s_Param;
    nn::la::PushToInChannel(appletHandle, &s_Param, sizeof(s_Param));

    return appletHandle;
}

void InitializeParameters()
{
    nn::applet::StorageHandle storageHandle[2];
    int inStorageSize[2];

    // 共通パラメータ
    nn::la::CommonArgumentsReader commonArg;
    NN_ABORT_UNLESS(nn::ae::TryPopFromInChannel(&storageHandle[0]));
    inStorageSize[0] = static_cast<int>(nn::applet::GetStorageSize(storageHandle[0]));
    NFPDEMO_LOG("Pop from InChannel[0] -> size = %d\n", inStorageSize[0]);
    NN_ABORT_UNLESS(commonArg.ReadFromStorage(storageHandle[0]));
    nn::applet::ReleaseStorage(storageHandle[0]);

    // 固有パラメータ
    NN_ABORT_UNLESS(nn::ae::TryPopFromInChannel(&storageHandle[1]));
    inStorageSize[1] = static_cast<int>(nn::applet::GetStorageSize(storageHandle[1]));
    NFPDEMO_LOG("Pop from InChannel[1] -> size = %d\n", inStorageSize[1]);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::ReadFromStorage(storageHandle[1], 0, &g_Data.startParam, sizeof(g_Data.startParam)));
    nn::applet::ReleaseStorage(storageHandle[1]);

    NFPDEMO_LOG("StartParam.mode %x\n", g_Data.startParam.mode);
    NFPDEMO_LOG("StartParam.infoFlags %x\n", g_Data.startParam.infoFlags);

    const auto flag = g_Data.startParam.infoFlags;
    const auto mask = nn::nfp::AmiiboSettingsInfoFlags_TagInfo;
    if( (flag & mask) == mask )
    {
        switch( g_Data.startParam.mode )
        {
        case nn::nfp::AmiiboSettingsMode_NicknameAndOwnerSettings:
            {
                g_Data.action = nns::nfp::Action_Register;
            }
            break;
        case nn::nfp::AmiiboSettingsMode_GameDataEraser:
            {
                g_Data.action = nns::nfp::Action_Delete;
            }
            break;
        case nn::nfp::AmiiboSettingsMode_Restorer:
            {
                g_Data.action = nns::nfp::Action_Restore;
            }
            break;
        case nn::nfp::AmiiboSettingsMode_Formatter:
            {
                g_Data.action = nns::nfp::Action_Format;
            }
            break;
        default:
            {
                g_Data.action = nns::nfp::Action_None;
            }
            break;
        }
    }
    else
    {
        g_Data.action = nns::nfp::Action_None;
    }
}

void FinalizeParameters()
{
    nn::applet::StorageHandle storageHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, sizeof(g_Data.returnValue)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, 0, &g_Data.returnValue, sizeof(g_Data.returnValue)));
    nn::ae::PushToOutChannel(storageHandle);
}

void InitializeMessageEvent() NN_NOEXCEPT
{
    NFPDEMO_LOG("Invoke InitializeNotificationMessageEvent()\n");
    nn::ae::InitializeNotificationMessageEvent(&g_MessageEvent);

    NFPDEMO_LOG("Wait Message_ChangeIntoForeground\n");
    nn::ae::Message message = nn::ae::WaitForNotificationMessage(&g_MessageEvent);
    nn::ae::Message expect = nn::ae::Message_ChangeIntoForeground;

    NN_ABORT_UNLESS( message == expect, "CABINET: Received unexpected expect=0x%08x message=0x%08x", expect, message );
    NFPDEMO_LOG("Received Message_ChangeIntoForeground\n");

    NFPDEMO_LOG("Invoke AcquireForegroundRights()\n");
    nn::ae::AcquireForegroundRights();
}

bool ProcessMessage() NN_NOEXCEPT
{
    bool background = false;
    auto message = nn::ae::GetNotificationMessage();
    do
    {
        switch( message )
        {
        case nn::ae::Message_None:
            break;

        // FG ⇒ BG 遷移
        case nn::ae::Message_ChangeIntoBackground:
            {
                NFPDEMO_LOG("Received Message_ChangeIntoBackground\n");
                nns::nfp::Cancel(g_Data);
                NFPDEMO_LOG("Invoke ReleaseForegroundRights()\n");
                background = true;
                nn::ae::ReleaseForegroundRights();
            }
            break;

        // BG ⇒ FG 遷移
        case nn::ae::Message_ChangeIntoForeground:
            {
                NFPDEMO_LOG("Received Message_ChangeIntoForeground\n");
                NFPDEMO_LOG("Invoke AcquireForegroundRights()\n");
                nn::ae::AcquireForegroundRights();
                background = false;
            }
            break;

        // 終了
        case nn::ae::Message_Exit:
            {
                NFPDEMO_LOG("Received Message_Exit\n");
                background = false;
                nns::nfp::Finalize(g_Data);
            }
            break;

        // 動作モード変化
        case nn::ae::Message_OperationModeChanged:
            {
                NFPDEMO_LOG("Received Message_OperationModeChanged\n");
            }
            break;

        // 性能モード変化
        case nn::ae::Message_PerformanceModeChanged:
            {
                NFPDEMO_LOG("Received Message_PerformanceModeChanged\n");
            }
            break;

        default:
            {
                NN_ABORT("CABINET: Received unknown message= 0x%08x", message);
            }
            break;
        }

        if( background )
        {
            NFPDEMO_LOG("Invoke WaitForNotificationMessage()\n");
            message = nn::ae::WaitForNotificationMessage(&g_MessageEvent);
        }
        else
        {
            break;
        }

    } while( NN_STATIC_CONDITION(true) );

    // 通知メッセージが終了依頼以外なら処理を継続
    return message != nn::ae::Message_Exit;
}   // NOLINT(impl/function_size)

// ライブラリアプレットのメモリ管理機構を初期化
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

// ライブラリアプレットのエントリポイント
void appletMain(const nn::ae::LibraryAppletSelfInfo& info)
{
    NFPDEMO_LOG("Launched LibraryApplet\n");

    InitializeParameters();
    InitializeGraphicSystem();
    InitializeMessageEvent();
    InitializeHidController();

    while( ProcessMessage() )
    {
        UpdateHidController();

        nns::nfp::UpdateState(g_Data);

        BeginText();
        nns::nfp::WriteState(g_Data);
        EndText();

        if( g_Data.state == nns::nfp::State_Exit )
        {
            break;
        }
    }

    FinalizeParameters();
    FinalizeHidController();
    FinalizeGraphicSystem();

    NFPDEMO_LOG("Invoke ExitLibraryApplet()\n");
    nn::ae::ExitLibraryApplet();
}

extern "C" void nnMain()
{
    NFPDEMO_LOG("Invoke SetCreateSelfLibraryAppletForDevelop()\n");
    nn::ae::SetCreateSelfLibraryAppletForDevelop(PrepareLibraryApplet);
    NFPDEMO_LOG("Invoke InvokeLibraryAppletMain()\n");
    nn::ae::InvokeLibraryAppletMain(appletMain);
}
