﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{NetworkServiceDiscoveryDownload.cpp,PageSampleNetworkServiceDiscovery}
 *
 * @brief
 *  ネットワークサービスディスカバリの設定データダウンロードのサンプルプログラム(ダウンロード部分)
 */

#include "./NetworkServiceDiscoveryDownload.h"

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/ssl/ssl_Context.h>
#include <curl/curl.h>

namespace networkServiceDiscoverySample
{

namespace
{
    struct CurlWriteFunctionUserData
    {
        uint8_t* pWriteBuffer;
        size_t writePosition;
        size_t buffreSize;
    };

    size_t CurlWriteFunction(char *pData, size_t blobsize, size_t blobcount, void *userdata) NN_NOEXCEPT;
    CURLcode CurlSslContextFunction(CURL* pCurl, void* pSslContext, void* pUserData) NN_NOEXCEPT;
}

CURLcode DownloadSettingData(
    size_t* pOutActualSize,
    void* pDownloadBuffer, size_t downloadBufferSize,
    CURL* curl,
    const nn::nsd::PassCode& passCode
    ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutActualSize);
    NN_ABORT_UNLESS_NOT_NULL(pDownloadBuffer);
    NN_ABORT_UNLESS_GREATER_EQUAL(downloadBufferSize, ResponseBufferSizeMin);
    NN_ABORT_UNLESS_NOT_NULL(curl);

    const char *url = "https://api.sect.srv.nintendo.net/v1/setting";

    CurlWriteFunctionUserData userData = { static_cast<uint8_t*>(pDownloadBuffer), 0, downloadBufferSize };

    curl_easy_setopt(curl, CURLOPT_URL, url);
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION,  CurlWriteFunction);
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, &userData);
    curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);

    curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 1L);
    curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 2L);
    curl_easy_setopt(curl, CURLOPT_SSL_VERIFYDATE, 0L);
    curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextFunction);

    // POSTデータの設定
    char postBuffer[256];
    int len = nn::util::SNPrintf(postBuffer, sizeof(postBuffer), "passcode=%s&device_id=0", passCode.value);
    NN_ABORT_UNLESS(len < static_cast<int>(sizeof(postBuffer)));

    curl_easy_setopt(curl, CURLOPT_POST, 1);
    curl_easy_setopt(curl, CURLOPT_POSTFIELDS, postBuffer);
    curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, len);

    // Perform the request
    CURLcode curlCode = curl_easy_perform(curl);

    if(curlCode == CURLE_OK)
    {
        *pOutActualSize = userData.writePosition;
    }

    return curlCode ;
}

namespace
{
    size_t CurlWriteFunction(char *pData, size_t blobsize, size_t blobcount, void *userdata) NN_NOEXCEPT
    {
        CurlWriteFunctionUserData* userData = static_cast<CurlWriteFunctionUserData*>(userdata);

        auto count = blobsize * blobcount;
        // NN_LOG("\n==== CurlWriteFunction count:%d userData->writePosition:%d\n", count, userData->writePosition);

        if(userData->writePosition + count <= userData->buffreSize)
        {
            memcpy(&userData->pWriteBuffer[userData->writePosition], pData, count);
            userData->writePosition += count;
        }
        else
        {
            NN_LOG("At CurlWriteFunction, download buffer was overflowed. bufferSize:%zu\n", userData->buffreSize);
            return 0; // 処理を中断します
        }

        return count;
    }

    CURLcode CurlSslContextFunction(CURL* pCurl, void* pSslContext, void* pUserData) NN_NOEXCEPT
    {
        NN_UNUSED(pCurl);
        NN_UNUSED(pUserData);

        auto& context = *reinterpret_cast<nn::ssl::Context*>(pSslContext);
        if (context.Create(nn::ssl::Context::SslVersion_Auto).IsFailure())
        {
            return CURLE_ABORTED_BY_CALLBACK;
        }
        return CURLE_OK;
    }
}

} // namespace networkServiceDiscoverySample
