﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{NetdiagGlobalIpAddress.cpp,PageSampleNetdiagGlobalIpAddress}
 *
 * @brief
 *  グローバル IP アドレス取得のサンプルプログラム
 */

/**
 * @page PageSampleNetdiagGlobalIpAddress NetdiagGlobalIpAddress
 * @tableofcontents
 *
 * @brief
 *  グローバル IP アドレス取得を行うサンプルの解説です。
 *
 * @section PageSampleNetdiagGlobalIpAddress_SectionBrief 概要
 *  ネットワーク接続後、HTTP 通信を行ってグローバル IP アドレス取得用サーバーに接続し、グローバル IP アドレスを取得します。
 *
 * @section PageSampleNetdiagGlobalIpAddress_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/NetdiagGlobalIpAddress Samples/Sources/Applications/NetdiagGlobalIpAddress @endlink 以下にあります。
 *
 * @section PageSampleNetdiagGlobalIpAddress_SectionNecessaryEnvironment 必要な環境
 *  事前に設定マネージャを使用してネットワーク接続設定をインポートする必要があります。
 *  詳細は @confluencelink{104465190,SettingsManager_network,ネットワーク接続設定の登録} をご覧ください。
 *
 * @section PageSampleNetdiagGlobalIpAddress_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleNetdiagGlobalIpAddress_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleNetdiagGlobalIpAddress_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleNetdiagGlobalIpAddress_SectionDetail 解説
 *
 * @subsection PageSampleNetdiagGlobalIpAddress_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  NetdiagGlobalIpAddress.cpp
 *  @includelineno NetdiagGlobalIpAddress.cpp
 *
 * @subsection PageSampleNetdiagGlobalIpAddress_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの全体像は以下の通りです。
 *
 * - NIFM ライブラリを使用してネットワーク接続の利用をシステムに要求します。
 * - socket ライブラリ、curl ライブラリを初期化します。
 * - nn::netdiag::GetGlobalIpAddress でグローバル IP アドレスを取得します。
 *   - 内部では、libcurl の easy interface を使用してグローバル IP アドレス取得用のサーバーに HTTP リクエストを送信します。
 *   - サーバーから、グローバル IP アドレスの情報が返ってきます。
 */

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/socket.h>
#include <curl/curl.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>

#include <nn/netdiag/netdiag_GlobalIpAddressTypes.h>
#include <nn/netdiag/netdiag_GlobalIpAddressApi.h>

//----------------------------------------------------------------
namespace
{
    NN_ALIGNAS(4096) uint8_t g_MallocBuffer[1 * 1024 * 1024];

    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;
}

extern "C" void nninitStartup()
{
    // 本サンプルはアプレット向け desc を利用しており、アプレット向けのリソース制限が適用されます。
    // ここでは、デフォルトの nninitStartup() のデフォルトメモリアロケータのサイズが
    // アプレットで利用できるサイズ上限を超えているため、
    // 自前で nninitStartup() を用意してデフォルトメモリアロケータのサイズを調整しています。

    // 本サンプルでは libcurl がデフォルトのメモリアロケータを利用するよう初期化しているので、
    // 適当に十分なサイズでデフォルトメモリアロケータを設定しています。
    // ここでのメモリアロケータ設定はグローバルな設定となるので、アプレットへの組み込み時には注意してください。
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

extern "C" void nnMain()
{
    // NIFM の初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    NN_LOG("Waiting for network interface availability...\n");
    nn::nifm::SubmitNetworkRequestAndWait();

    if( !nn::nifm::IsNetworkAvailable() )
    {
        NN_LOG("Network is not available.\n");
        return;
    }

    // SOCKET の初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::socket::Initialize(g_SocketConfigWithMemory) );

    // curl の初期化
    curl_global_init(CURL_GLOBAL_DEFAULT);

    // グローバルIPアドレス取得
    nn::netdiag::GlobalIpAddress addr;
    nn::Result result = nn::netdiag::GetGlobalIpAddress( &addr );
    if ( result.IsSuccess() )
    {
        NN_LOG("Global IP Address = [%s]\n", addr.value);
    }
    else
    {
        NN_LOG("Failed to get global IP address (error 0x%x)\n\n", result.GetDescription());
    }

    curl_global_cleanup();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Finalize());
}

