﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/socket.h>
#include <curl/curl.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>

#include <nn/netdiag/netdiag_BandWidthApi.h>
#include <nn/netdiag/netdiag_Result.h>

//----------------------------------------------------------------

/**
 * @examplesource{NetdiagBandWidthMeasuringMain.cpp,PageSampleNetdiagBandWidthMeasuring}
 *
 * @brief
 *  ネットワーク帯域測定のサンプルプログラム
 */

/**
 * @page PageSampleNetdiagBandWidthMeasuring NetdiagBandWidthMeasuring
 * @tableofcontents
 *
 * @brief
 *  ネットワーク帯域測定を行うサンプルの解説です。
 *
 * @section PageSampleNetdiagBandWidthMeasuring_SectionBrief 概要
 *  ネットワーク接続後、HTTP 通信を行ってアップロード・ダウンロードそれぞれの帯域測定を行います。
 *
 * @section PageSampleNetdiagBandWidthMeasuring_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/NetdiagBandWidthMeasuring Samples/Sources/Applications/NetdiagBandWidthMeasuring @endlink 以下にあります。
 *
 * @section PageSampleNetdiagBandWidthMeasuring_SectionNecessaryEnvironment 必要な環境
 *  事前に設定マネージャを使用してネットワーク接続設定をインポートする必要があります。
 *  詳細は @confluencelink{104465190,SettingsManager_network,ネットワーク接続設定の登録} をご覧ください。
 *
 * @section PageSampleNetdiagBandWidthMeasuring_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleNetdiagBandWidthMeasuring_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleNetdiagBandWidthMeasuring_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleNetdiagBandWidthMeasuring_SectionDetail 解説
 *
 * @subsection PageSampleNetdiagBandWidthMeasuring_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  NetdiagBandWidthMeasuringMain.cpp
 *  @includelineno NetdiagBandWidthMeasuringMain.cpp
 *
 * @subsection PageSampleNetdiagBandWidthMesuring_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの全体像は以下の通りです。
 *
 * - NIFM ライブラリを使用してネットワーク接続の利用をシステムに要求します。
 * - socket ライブラリ、curl ライブラリを初期化します。
 * - nn::netdiag::MeasureBandWidthForUpload と nn::netdiag::MeasureBandWidthForDownload で帯域測定を行います。
 * - nn::netdiag::MeasureBandWidthForUpload では、libcurl の easy interface を使用して帯域測定サーバーに HTTP リクエストを送信します。
 *   - 帯域測定サーバーへ 1MiB のデータを送信して、アップロード帯域測定を行います。
 *   - 同様に nn::netdiag::MeasureBandWidthForDownload も libcurl で帯域測定サーバーに HTTP リクエストを送信します。
 *   - 帯域測定サーバーから 30MiB のデータを受信して、ダウンロード帯域測定を行います。
 * - 各測定の API にはタイムアウト時間を指定しており、送受信が途中であった場合でもそこまでの進捗で結果を算定します。
 * - 各結果は、bytes/sec の値で表示します。
 */

namespace
{
    NN_ALIGNAS(4096) uint8_t g_MallocBuffer[1 * 1024 * 1024];

    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;
}

extern "C" void nninitStartup()
{
    // 本サンプルはアプレット向け desc を利用しており、アプレット向けのリソース制限が適用されます。
    // ここでは、デフォルトの nninitStartup() のデフォルトメモリアロケータのサイズが
    // アプレットで利用できるサイズ上限を超えているため、
    // 自前で nninitStartup() を用意してデフォルトメモリアロケータのサイズを調整しています。

    // 本サンプルでは libcurl がデフォルトのメモリアロケータを利用するよう初期化しているので、
    // 適当に十分なサイズでデフォルトメモリアロケータを設定しています。
    // ここでのメモリアロケータ設定はグローバルな設定となるので、アプレットへの組み込み時には注意してください。
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

extern "C" void nnMain()
{
    // NIFM の初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    NN_LOG("Waiting for network interface availability...\n");
    nn::nifm::SubmitNetworkRequestAndWait();

    if( !nn::nifm::IsNetworkAvailable() )
    {
        NN_LOG("Network is not available.\n");
        return;
    }

    // SOCKET の初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::socket::Initialize(g_SocketConfigWithMemory) );

    // curl の初期化
    curl_global_init(CURL_GLOBAL_DEFAULT);

    // アップロード帯域測定
    bool isTimedout;
    nn::netdiag::BandWidth uploadSpeed;
    nn::Result resultUpload = nn::netdiag::MeasureBandWidthForUpload( &uploadSpeed, nn::netdiag::TimeoutForUploadMeasuring, &isTimedout );
    if ( resultUpload.IsFailure() )
    {
        NN_LOG( "Upload speed : n/a (result=%08x, %03d-%04d)\n",
                resultUpload.GetInnerValueForDebug(),
                resultUpload.GetModule(),
                resultUpload.GetDescription());
    }
    else
    {
        if ( isTimedout )
        {
            NN_LOG( "Timeout.\n" );
        }
        NN_LOG( "Upload speed : %f (bytes/sec)\n", uploadSpeed );
    }

    // ダウンロード帯域測定
    nn::netdiag::BandWidth downloadSpeed;
    nn::Result resultDownload = nn::netdiag::MeasureBandWidthForDownload( &downloadSpeed, nn::netdiag::TimeoutForDownloadMeasuring, &isTimedout );
    if ( resultDownload.IsFailure() )
    {
        NN_LOG( "Download speed : n/a (result=%08x, %03d-%04d)\n",
                resultDownload.GetInnerValueForDebug(),
                resultDownload.GetModule(),
                resultDownload.GetDescription());
    }
    else
    {
        if ( isTimedout )
        {
            NN_LOG( "Timeout.\n" );
        }
        NN_LOG( "Download speed : %f (bytes/sec)\n", downloadSpeed );
    }

    curl_global_cleanup();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Finalize());
}
