﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
 /**
 * @examplesource{MoviePlayerSimple.cpp,PageSampleMoviePlayerSimple}
 *
 * @brief
 *  MoviePlayer API による動画再生サンプル
 */
 /**
 * @page PageSampleMoviePlayerSimple MoviePlayerSimple
 * @tableofcontents
 *
 * @brief
 * MoviePlayer API による動画再生サンプルプログラムです。
 *
 * @section PageSampleMoviePlayerSimple_SectionBrief 概要
 * NX 本体上で動画ファイルを入力し、再生を行うサンプルです。
 *
 * @section PageSampleMoviePlayerSimple_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/MoviePlayerSimple
 * Samples/Sources/Applications/MoviePlayerSimple @endlink 以下にあります。
 *
 * @section PageSampleMoviePlayerSimple_SectionNecessaryEnvironment 必要な環境
 * 追加で必要となるものはありません。
 *
 * @section PageSampleMoviePlayerSimple_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、自動的に指定した動画の再生が始まります。
 * 再生終了後は、自動的にプログラムは終了します。
 *
 * - A ボタン: 再生の一時停止／再生再開
 * - B ボタン: 再生の停止
 * - X ボタン; アプリケーションの終了
 * - L ボタン: 巻き戻し（デフォルトは250msごと）
 * - R ボタン: 先送り（デフォルトは250msごと）
 * - ZRボタン: 巻き戻しまたは先送りの単位時間を増やす（250ms ずつ最大 10sec まで）
 *
 * @section PageSampleMoviePlayerSimple_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleMoviePlayerSimple_SectionHowToExecute 実行手順
 *  @<MoviePlayerSimple executable@> @<movie path@> @<optional parameters@>
 *
 * @<movie path@> : サポート済みフォーマットの動画ファイル
 *
 * - コンテナ: mp4, m4a, webm, mkv
 * - video コーデック: h264, vp8, vp9
 * - audio コーデック: aac, vorbis
 *
 * @<optional parameters@>
 *
 * - -dm 1: movie::DecoderMode_Cpu モードを有効化する。
 * - -dm 2: movie::DecoderMode_NativeTexture モードを有効化する。
 *          アプリケーションは video デコーダーに NVN_FORMAT_RGBA8 フォーマットの NVN テクスチャを渡す。（デフォルト）
 *
 * 例（ホストPCから動画データを取得）
 * MoviePlayerSimple.nsp c:@\H264_AAC_1280x720_30sec_01.mp4
 *
 * @section PageSampleMoviePlayerSimple_SectionDetail 解説
 * 指定した動画ファイルを NX 本体上で再生します。
 */

#include "MoviePlayerSimple.h"
#include "MemoryHandler.h"

#include <nn/fs/fs_SdCardForDebug.h>

// Process startup setup
extern "C" void nninitStartup()
{
    FsInitHeap();
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
}

// MoviePlayerSimple Player Usage
static void Usage()
{
    NN_LOG( "\n MoviePlayerSimple ::Invalid options, refer Readme.MoviePlayerSimple.txt \n");
}

// Process main entry
extern "C" void nnMain()
{
    int argc = nn::os::GetHostArgc();
    char** argvs = nn::os::GetHostArgv();

    // Set movie allocator callback functions
    movie::SetAllocator(MovieAllocate, MovieDeallocate, MovieReallocate, NULL);

    // Donate memory for graphics driver to work in
    uint8_t *graphicsHeap = nullptr;
    size_t graphicsHeapSize = 0;
    GetGraphicsHeap(&graphicsHeap, &graphicsHeapSize);
    nv::InitializeGraphics(graphicsHeap, graphicsHeapSize);

    // Set graphics allocator callback functions
    nv::SetGraphicsAllocator(MovieAllocate, MovieDeallocate, MovieReallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(MovieAllocate, MovieDeallocate, MovieReallocate, NULL);

    if( argc <= 1 )
    {
        Usage();
        return;
    }
    char* mediaFile = argvs[ 1 ];
    movie::DecoderMode videoDecoderMode = movie::DecoderMode_NativeTexture;
    for( int i = 0; i < nn::os::GetHostArgc(); ++i )
    {
        if( strcmp("-dm", argvs[ i ]) == 0 )
        {
            if( i < nn::os::GetHostArgc() - 1 )
            {
                int mode = atoi(argvs[ i + 1 ]);
                if( mode == 1 )
                {
                    videoDecoderMode = movie::DecoderMode_Cpu;
                }
            }
        }
    }
    movie::Status movieStatus = movie::Status_UnknownError;
    MoviePlayerSimple *moviePlayerSimple = new MoviePlayerSimple { };
    if( moviePlayerSimple != nullptr )
    {
        moviePlayerSimple->Initialize();
        if( true == moviePlayerSimple->MountFileSystem(mediaFile) )
        {
            NN_LOG("\n Movie name [%s] \n", mediaFile);
            movieStatus = moviePlayerSimple->CreateMoviePlayer();
            if( movieStatus == movie::Status_Success )
            {
                moviePlayerSimple->CreateHidHandler();
                movieStatus = moviePlayerSimple->PlayMovie(mediaFile, videoDecoderMode);
                if( movieStatus == movie::Status_Success )
                {
                    moviePlayerSimple->HandleEvents();
                }
            }
            if( movieStatus != movie::Status_Success )
            {
                NN_LOG("\n MoviePlayer FAILED to play movie! Error = %s \n", movie::StatusToString(movieStatus));
            }
            moviePlayerSimple->DestroyHidHandler();
            moviePlayerSimple->DestroyMoviePlayer();
            moviePlayerSimple->UnMountFileSystem();
        }
        moviePlayerSimple->Finalize();
    }
    nv::FinalizeGraphics();
    if( moviePlayerSimple != nullptr )
    {
        delete moviePlayerSimple;
        moviePlayerSimple = nullptr;
    }
    NN_LOG("\n MoviePlayerSimple sample EXITED \n");
}

MoviePlayerSimple::MoviePlayerSimple()
    : m_NativeWindow(nullptr),
      m_MoviePlayer(nullptr),
      m_MediaFile(nullptr),
      m_EventsInitialized(false),
      m_SdcardMounted(false),
      m_HostFsMounted(false),
      m_VideoDecoderMode(movie::DecoderMode_Cpu),
      m_SeekSkipUnitUs(250000)
{
}

bool MoviePlayerSimple::Initialize()
{
    if( m_EventsInitialized == false )
    {
        nn::os::InitializeEvent(&m_PlayerStateChangedEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_PlayerErrorEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_HidEventButtonPressA, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_HidEventButtonPressB, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_HidEventButtonPressX, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_HidEventButtonPressZr, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_HidEventButtonPressL, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_HidEventButtonPressR, false, nn::os::EventClearMode_ManualClear);

        // Initialize multiWait for Player
        nn::os::InitializeMultiWait(&m_PlayerMultiWait);

        RegisterEvent(&m_PlayerStateChangedEvent, MoviePlayerSimpleEventType_StateChanged);
        RegisterEvent(&m_PlayerErrorEvent, MoviePlayerSimpleEventType_Error);

        RegisterEvent(&m_HidEventButtonPressA, MoviePlayerSimpleEventType_ButtonPressA);
        RegisterEvent(&m_HidEventButtonPressB, MoviePlayerSimpleEventType_ButtonPressB);
        RegisterEvent(&m_HidEventButtonPressX, MoviePlayerSimpleEventType_ButtonPressX);
        RegisterEvent(&m_HidEventButtonPressZr, MoviePlayerSimpleEventType_ButtonPressZr);
        RegisterEvent(&m_HidEventButtonPressL, MoviePlayerSimpleEventType_ButtonPressL);
        RegisterEvent(&m_HidEventButtonPressR, MoviePlayerSimpleEventType_ButtonPressR);

        m_EventsInitialized = true;
    }
    return true;
}

void MoviePlayerSimple::Finalize()
{
    // Finalize thread exit event holder.
    if( m_EventsInitialized == true )
    {
        nn::os::UnlinkMultiWaitHolder(&m_PlayerStateChangedEventHolder);
        nn::os::UnlinkMultiWaitHolder(&m_PlayerErrorEventHolder);
        nn::os::UnlinkMultiWaitHolder(&m_HidEventButtonPressAHolder);
        nn::os::UnlinkMultiWaitHolder(&m_HidEventButtonPressBHolder);
        nn::os::UnlinkMultiWaitHolder(&m_HidEventButtonPressXHolder);
        nn::os::UnlinkMultiWaitHolder(&m_HidEventButtonPressZrHolder);
        nn::os::UnlinkMultiWaitHolder(&m_HidEventButtonPressLHolder);
        nn::os::UnlinkMultiWaitHolder(&m_HidEventButtonPressRHolder);

        DeleteEventUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_PlayerStateChangedEventHolder));
        DeleteEventUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_PlayerErrorEventHolder));
        DeleteEventUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_HidEventButtonPressAHolder));
        DeleteEventUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_HidEventButtonPressBHolder));
        DeleteEventUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_HidEventButtonPressXHolder));
        DeleteEventUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_HidEventButtonPressZrHolder));
        DeleteEventUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_HidEventButtonPressLHolder));
        DeleteEventUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_HidEventButtonPressRHolder));

        // Finalize event holder.
        nn::os::FinalizeMultiWaitHolder(&m_PlayerStateChangedEventHolder);
        nn::os::FinalizeMultiWaitHolder(&m_PlayerErrorEventHolder);
        nn::os::FinalizeMultiWaitHolder(&m_HidEventButtonPressAHolder);
        nn::os::FinalizeMultiWaitHolder(&m_HidEventButtonPressBHolder);
        nn::os::FinalizeMultiWaitHolder(&m_HidEventButtonPressXHolder);
        nn::os::FinalizeMultiWaitHolder(&m_HidEventButtonPressZrHolder);
        nn::os::FinalizeMultiWaitHolder(&m_HidEventButtonPressLHolder);
        nn::os::FinalizeMultiWaitHolder(&m_HidEventButtonPressRHolder);

        // Finalize event holder.
        nn::os::FinalizeEvent(&m_PlayerStateChangedEvent);
        nn::os::FinalizeEvent(&m_PlayerErrorEvent);
        nn::os::FinalizeEvent(&m_HidEventButtonPressA);
        nn::os::FinalizeEvent(&m_HidEventButtonPressB);
        nn::os::FinalizeEvent(&m_HidEventButtonPressX);
        nn::os::FinalizeEvent(&m_HidEventButtonPressZr);
        nn::os::FinalizeEvent(&m_HidEventButtonPressL);
        nn::os::FinalizeEvent(&m_HidEventButtonPressR);

        // Finalize multiWait.
        nn::os::FinalizeMultiWait(&m_PlayerMultiWait);
        m_EventsInitialized = false;
    }
    m_NativeWindow = nullptr;
}

bool MoviePlayerSimple::CreateHidHandler()
{
    bool retStatus = false;
    m_HidHandler = new HidHandler();
    if( m_HidHandler != nullptr )
    {
        if( true == m_HidHandler->Initialize() )
        {
            m_HidHandler->RegisterEvent(&m_HidEventButtonPressA, HidHandler::HidHandlerKeyType_ButtonA);
            m_HidHandler->RegisterEvent(&m_HidEventButtonPressB, HidHandler::HidHandlerKeyType_ButtonB);
            m_HidHandler->RegisterEvent(&m_HidEventButtonPressX, HidHandler::HidHandlerKeyType_ButtonX);
            m_HidHandler->RegisterEvent(&m_HidEventButtonPressZr, HidHandler::HidHandlerKeyType_ButtonZr);
            m_HidHandler->RegisterEvent(&m_HidEventButtonPressL, HidHandler::HidHandlerKeyType_ButtonL);
            m_HidHandler->RegisterEvent(&m_HidEventButtonPressR, HidHandler::HidHandlerKeyType_ButtonR);

            retStatus = true;
        }
    }
    return retStatus;
}

movie::Status MoviePlayerSimple::CreateMoviePlayer()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    m_MoviePlayer = new MoviePlayer { };

    if( m_MoviePlayer != NULL )
    {
        movieStatus = m_MoviePlayer->Initialize();
        if( movieStatus != movie::Status_Success )
        {
            return movieStatus;
        }
        // Pass null NativeWindowHandle, renderer sets up VI
        m_MoviePlayer->SetViNativeWindowHandle(m_NativeWindow);
        movieStatus = m_MoviePlayer->RegisterEvent(MoviePlayer::EventType_StateChangedEvent, &m_PlayerStateChangedEvent);
        if( movieStatus != movie::Status_Success )
        {
            return movieStatus;
        }
        movieStatus = m_MoviePlayer->RegisterEvent(MoviePlayer::EventType_ErrorEvent, &m_PlayerErrorEvent);
        if( movieStatus != movie::Status_Success )
        {
            return movieStatus;
        }
    }
    return movieStatus;
}

void MoviePlayerSimple::DestroyMoviePlayer()
{
    if( m_MoviePlayer != NULL )
    {
        delete m_MoviePlayer;
        m_MoviePlayer = nullptr;
    }
}

void MoviePlayerSimple::DestroyHidHandler()
{
    if( m_HidHandler != nullptr )
    {
        m_HidHandler->Finalize();
        delete m_HidHandler;
        m_HidHandler = nullptr;
    }
}

movie::Status MoviePlayerSimple::PlayMovie(char* mediaFile, movie::DecoderMode videoDecoderMode)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    if( false == DoesMediaFileExists(mediaFile) )
    {
        NN_LOG("\n %s file not found !\n", mediaFile);
        return movie::Status_TrackNotFound;
    }
    m_MediaFile = mediaFile;
    m_VideoDecoderMode = videoDecoderMode;
    if( m_MoviePlayer != NULL )
    {
        int videoTrackNumber = -1;
        int audioTrackNumber = -1;

        movieStatus = m_MoviePlayer->SetDataSource(mediaFile);
        if( movieStatus != movie::Status_Success )
        {
            return movieStatus;
        }
        int64_t mediaDurationUs = 0;
        movieStatus = m_MoviePlayer->GetPlaybackDuration(&mediaDurationUs);
        if( movieStatus == movie::Status_Success )
        {
            NN_LOG("\n Movie playback duration = %lld Seconds \n", mediaDurationUs / 1000000);
        }
        movieStatus = m_MoviePlayer->SelectVideoTrack(&videoTrackNumber);
        if( movieStatus != movie::Status_Success )
        {
            NN_LOG("\n Valid VideoTrack not found ! \n");
        }
        movieStatus = m_MoviePlayer->SelectAudioTrack(&audioTrackNumber);
        if( movieStatus != movie::Status_Success )
        {
            NN_LOG("\n  Valid AudioTrack not found ! \n");
        }
        if( ( videoTrackNumber < 0 ) && ( audioTrackNumber < 0 ) )
        {
            return movie::Status_TrackNotFound;
        }
        movieStatus = m_MoviePlayer->SetVideoDecoderMode(m_VideoDecoderMode);
        if( movieStatus != movie::Status_Success )
        {
            return movieStatus;
        }
        if( m_VideoDecoderMode == movie::DecoderMode_NativeTexture )
        {
            NN_LOG("\n Using [movie::DecoderMode_NativeTexture] mode for video decoder \n");
        }
        else
        {
            NN_LOG("\n Using [movie::DecoderMode_Cpu] mode for video decoder \n");
        }
        movieStatus = m_MoviePlayer->Prepare();
        if( movieStatus != movie::Status_Success )
        {
            return movieStatus;
        }
        movieStatus = m_MoviePlayer->Start();
        if( movieStatus != movie::Status_Success )
        {
            return movieStatus;
        }
        NN_LOG("\n Movie playback STARTED \n");
    }
    return movieStatus;
}

movie::Status MoviePlayerSimple::SeekMovie(bool forward)
{
    movie::Status movieStatus = movie::Status_SeekNotAllowed;
    MoviePlayer::State state;
    movieStatus = m_MoviePlayer->GetState(&state);
    if( movieStatus != movie::Status_Success )
    {
        return movieStatus;
    }
    if( state == MoviePlayer::State_Started )
    {
        movieStatus = m_MoviePlayer->Pause();
    }
    int64_t playbackPositionUs = 0;
    m_MoviePlayer->GetPlaybackPosition(&playbackPositionUs);
    int64_t seekPositionUs = playbackPositionUs;
    if( forward == true )
    {
        seekPositionUs = playbackPositionUs + m_SeekSkipUnitUs;
    }
    else
    {
        seekPositionUs = playbackPositionUs - m_SeekSkipUnitUs;
    }
    if( seekPositionUs < 0 )
    {
        seekPositionUs = 0;
    }
    movieStatus = m_MoviePlayer->Seek(seekPositionUs);
    if( movieStatus == movie::Status_InvalidStateTransition )
    {
        NN_LOG("\n Movie Seek FAILED(Seek not allowed in this state) Error = %s \n", StatusToString(movieStatus));
    }
    if( state == MoviePlayer::State_Started )
    {
        movieStatus = m_MoviePlayer->Resume();
    }
    return movieStatus;
}

bool MoviePlayerSimple::HandleEvents()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    bool exitPlayer = false;
    // Wait for Player events, HID events
    for( ;; )
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&m_PlayerMultiWait);
        uintptr_t userData = GetMultiWaitHolderUserData(holder);
        EventUserData *eventUserData = ( EventUserData* ) userData;
        if( eventUserData == nullptr )
        {
            break;
        }
        MoviePlayerSimpleEventType eventype = eventUserData->m_EventType;
        nn::os::EventType *event = eventUserData->m_Event;
        nn::os::ClearEvent(event);
        switch( eventype )
        {
            case MoviePlayerSimpleEventType_StateChanged:
                m_MoviePlayer->GetState(&m_State);
                if( m_State == MoviePlayer::State::State_PlaybackCompleted )
                {
                    movieStatus = m_MoviePlayer->Stop();
                    NN_LOG("\n Movie playback STOPPED \n");
                    exitPlayer = true;
                }
                break;

            case MoviePlayerSimpleEventType_Error:
                NN_LOG("\n MoviePlayer: Async Error = %s \n", movie::StatusToString(m_MoviePlayer->GetLastError()));
                movieStatus = m_MoviePlayer->Stop();
                NN_LOG("\n Movie playback STOPPED \n");
                exitPlayer = true;
                break;

            case MoviePlayerSimpleEventType_ButtonPressA:
                m_MoviePlayer->GetState(&m_State);
                switch( m_State )
                {
                    case MoviePlayer::State::State_Paused:
                    case MoviePlayer::State::State_SeekCompleted:
                        movieStatus = m_MoviePlayer->Resume();
                        NN_LOG("\n Movie playback RESUMED \n");
                        break;

                    case MoviePlayer::State::State_Started:
                        movieStatus = m_MoviePlayer->Pause();
                        NN_LOG("\n Movie playback PAUSED \n");
                        break;

                    case MoviePlayer::State::State_Stopped:
                        PlayMovie(m_MediaFile, m_VideoDecoderMode);
                        break;
                    default:
                        break;
                }
                break;

            case MoviePlayerSimpleEventType_ButtonPressB:
                movieStatus = m_MoviePlayer->Stop();
                NN_LOG("\n Movie playback STOPPED \n");
                break;

            case MoviePlayerSimpleEventType_ButtonPressX:
                movieStatus = m_MoviePlayer->Stop();
                exitPlayer = true;
                NN_LOG("\n Movie playback STOPPED \n");
                break;

            case MoviePlayerSimpleEventType_ButtonPressZr:
            {
                m_SeekSkipUnitUs += 250000;
                if( m_SeekSkipUnitUs > 10000000 )
                {
                    m_SeekSkipUnitUs = 250000;
                }
                NN_LOG("\n Seek unit = %lld Us\n", m_SeekSkipUnitUs);
                break;
            }

            case MoviePlayerSimpleEventType_ButtonPressL:
            {
                NN_LOG("\n Movie SEEKING BACKWARD \n");
                bool forward = false;
                SeekMovie(forward);
                break;
            }

            case MoviePlayerSimpleEventType_ButtonPressR:
            {
                NN_LOG("\n Movie SEEKING FORWARD \n");
                bool forward = true;
                SeekMovie(forward);
                break;
            }

            default:
                break;
        }
        if( exitPlayer == true )
        {
            break;
        }
    }
    return true;
}

void MoviePlayerSimple::RegisterEvent(nn::os::EventType *event, MoviePlayerSimpleEventType eventType)
{
    nn::os::MultiWaitHolderType *eventHolder = CreateEventHolder(event, eventType);
    nn::os::LinkMultiWaitHolder(&m_PlayerMultiWait, eventHolder);
}

nn::os::MultiWaitHolderType* MoviePlayerSimple::CreateEventHolder(nn::os::EventType *event, MoviePlayerSimpleEventType eventType)
{
    switch( eventType )
    {
    case MoviePlayerSimpleEventType_StateChanged:
    {
        EventUserData *userData = new EventUserData(event, MoviePlayerSimpleEventType_StateChanged);
        nn::os::InitializeMultiWaitHolder(&m_PlayerStateChangedEventHolder, event);
        nn::os::SetMultiWaitHolderUserData(&m_PlayerStateChangedEventHolder, ( uintptr_t ) userData);
        return &m_PlayerStateChangedEventHolder;
    }

    case MoviePlayerSimpleEventType_Error:
    {
        EventUserData *userData = new EventUserData(event, MoviePlayerSimpleEventType_Error);
        nn::os::InitializeMultiWaitHolder(&m_PlayerErrorEventHolder, event);
        nn::os::SetMultiWaitHolderUserData(&m_PlayerErrorEventHolder, ( uintptr_t ) userData);
        return &m_PlayerErrorEventHolder;
    }
    case MoviePlayerSimpleEventType_ButtonPressA:
    {
        EventUserData *userData = new EventUserData(event, MoviePlayerSimpleEventType_ButtonPressA);
        nn::os::InitializeMultiWaitHolder(&m_HidEventButtonPressAHolder, event);
        nn::os::SetMultiWaitHolderUserData(&m_HidEventButtonPressAHolder, ( uintptr_t ) userData);
        return &m_HidEventButtonPressAHolder;
    }

    case MoviePlayerSimpleEventType_ButtonPressB:
    {
        EventUserData *userData = new EventUserData(event, MoviePlayerSimpleEventType_ButtonPressB);
        nn::os::InitializeMultiWaitHolder(&m_HidEventButtonPressBHolder, event);
        nn::os::SetMultiWaitHolderUserData(&m_HidEventButtonPressBHolder, ( uintptr_t ) userData);
        return &m_HidEventButtonPressBHolder;
    }

    case MoviePlayerSimpleEventType_ButtonPressX:
    {
        EventUserData *userData = new EventUserData(event, MoviePlayerSimpleEventType_ButtonPressX);
        nn::os::InitializeMultiWaitHolder(&m_HidEventButtonPressXHolder, event);
        nn::os::SetMultiWaitHolderUserData(&m_HidEventButtonPressXHolder, ( uintptr_t ) userData);
        return &m_HidEventButtonPressXHolder;
    }

    case MoviePlayerSimpleEventType_ButtonPressZr:
    {
        EventUserData *userData = new EventUserData(event, MoviePlayerSimpleEventType_ButtonPressZr);
        nn::os::InitializeMultiWaitHolder(&m_HidEventButtonPressZrHolder, event);
        nn::os::SetMultiWaitHolderUserData(&m_HidEventButtonPressZrHolder, ( uintptr_t ) userData);
        return &m_HidEventButtonPressZrHolder;
    }

    case MoviePlayerSimpleEventType_ButtonPressL:
    {
        EventUserData *userData = new EventUserData(event, MoviePlayerSimpleEventType_ButtonPressL);
        nn::os::InitializeMultiWaitHolder(&m_HidEventButtonPressLHolder, event);
        nn::os::SetMultiWaitHolderUserData(&m_HidEventButtonPressLHolder, ( uintptr_t ) userData);
        return &m_HidEventButtonPressLHolder;
    }

    case MoviePlayerSimpleEventType_ButtonPressR:
    {
        EventUserData *userData = new EventUserData(event, MoviePlayerSimpleEventType_ButtonPressR);
        nn::os::InitializeMultiWaitHolder(&m_HidEventButtonPressRHolder, event);
        nn::os::SetMultiWaitHolderUserData(&m_HidEventButtonPressRHolder, ( uintptr_t ) userData);
        return &m_HidEventButtonPressRHolder;
    }

    default:
        break;
    }
    return nullptr;
}

void MoviePlayerSimple::DeleteEventUserData(EventUserData* userData)
{
    if( userData != nullptr )
    {
        delete userData;
    }
}

bool MoviePlayerSimple::DoesMediaFileExists(char* mediaFileName)
{
    nn::fs::FileHandle fileHandle;
    fileHandle.handle = NULL;
    nn::Result result = nn::fs::OpenFile(&fileHandle, mediaFileName, nn::fs::OpenMode_Read);
    if( result.IsFailure() )
    {
        return false;
    }
    else
    {
        nn::fs::CloseFile(fileHandle);
    }
    return true;
}

bool MoviePlayerSimple::MountFileSystem(char* mediaFile)
{
    bool retValue = false;
    std::string path = mediaFile;
    std::string delimiter = ":";
    std::string token = path.substr(0, path.find(delimiter));
    if( token == "sdcard" )
    {
        nn::Result resultSdcardMount = nn::fs::MountSdCardForDebug("sdcard");
        if( resultSdcardMount.IsFailure() )
        {
            NN_LOG("\n nn::fs::SD card mount failure. Module:%d, Description:%d\n",
                resultSdcardMount.GetModule(),
                resultSdcardMount.GetDescription());
            return retValue;
        }
        m_SdcardMounted = true;
        retValue = true;
    }
    else if( token == "http" || token == "https" || token == "file" )
    {
        NN_LOG("\n MoviePlayer: Network playback not supported !\n");
    }
    else
    {
        nn::Result resultHostMount = nn::fs::MountHostRoot();
        if( resultHostMount.IsFailure() )
        {
            NN_LOG("\n nn::fs::Host root mount failure. Module:%d, Description:%d\n",
                resultHostMount.GetModule(),
                resultHostMount.GetDescription());
            return retValue;
        }
        m_HostFsMounted = true;
        retValue = true;
    }
    return retValue;
}

void MoviePlayerSimple::UnMountFileSystem()
{
    if( m_SdcardMounted == true )
    {
        nn::fs::Unmount("sdcard");
    }
    else if( m_HostFsMounted == true )
    {
        nn::fs::UnmountHostRoot();
    }
}
