﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @file
 * @brief MoviePlayer API's for media playback.
 */

#pragma once

#include <nns/mm.h>
#include <nn/oe.h>
#include "MovieDecoderEventHandler.h"

void MoviePlayerThreadFunction(void *arg);

/**
 * @brief MoviePlayer class for media playback
 */
class MoviePlayer
{

public:
    /**
     * @brief
     * Enumeration of MoviePlayer states.
     */
    enum State
    {
        State_NotInitialized    = 0,            //!< NotInitialized state.
        State_Initialized       = 1,            //!< Initialized state.
        State_Preparing         = 2,            //!< Preparing state.
        State_Prepared          = 3,            //!< Prepared state.
        State_Started           = 4,            //!< Started state.
        State_Stopped           = 5,            //!< Stopped state.
        State_Paused            = 6,            //!< Paused state.
        State_Seeking           = 7,            //!< Seeking state.
        State_SeekCompleted     = 8,            //!< SeekCompleted state.
        State_PlaybackCompleted = 9,            //!< PlaybackCompleted state.
        State_Error             = 10,           //!< Error state.
    };
    /**
     * @brief
     * Structure to hold audio track properties
     */
    struct TrackInfoAudio
    {
        std::string mime;                       //!< Mime type of the track.
        int sampleRate;                         //!< Audio sample rate in Hertz.
        int channels;                           //!< Number of audio channels.
        int bitRate;                            //!< Audio bitrate(bits per second).
        movie::DecoderType decoderType;         //!< Audio Decoder type.
    };
    /**
     * @brief
     * Structure to hold video track properties
     */
    struct TrackInfoVideo
    {
        std::string mime;                       //!< Mime type of the track.
        int width;                              //!< Video width (Number of distinct vertical lines).
        int height;                             //!< Video height (Number of distinct horizontal lines).
        int bitRate;                            //!< Video bit rate (bits per second).
        int frameRate;                          //!< Video presentation frame rate, frames per seconds.
        movie::DecoderType decoderType;         //!< Video decoder type.
    };

    /**
     * @brief
     * Structure to hold track properties
     */
    struct TrackInfo
    {
        TrackInfoAudio trackInfoAudio;          //!< Audio track information.
        TrackInfoVideo trackInfoVideo;          //!< Video track information.
    };

    /**
     * @brief
     * Enumeration of Track types.
     * @details
     * Track types recognized by MoviePlayer.
     */
    enum TrackType
    {
        TrackType_Unknown   = 0,                //!< Unknown track.
        TrackType_Audio     = 1,                //!< Audio track.
        TrackType_Video     = 2,                //!< Video track.
    };

    /**
     * @brief
     * Enumeration of EventType.
     */
    enum EventType
    {
        EventType_ErrorEvent        = 0,
        EventType_StateChangedEvent = 1,
    };

public:
    /**
     * @brief
     * Create MoviePlayer instance.
     *
     * @return                                  MoviePlayer instantiated.
     *
     * @post
     * MoviePlayer instance created.
     *
     * @details
     * Create an instance of MoviePlayer to play media files.
     *
     * MoviePlayer state transition on success:
     *  -> State_UnInitialized.
     */
    MoviePlayer() NN_NOEXCEPT;

    /**
     * @brief
     * MoviePlayer destructor.
     *
     * @return                                  MoviePlayer instance is destroyed.
     *
     * @pre
     * MoviePlayer instance is created.
     *
     * @post
     * MoviePlayer deleted.
     *
     * @details
     * MoviePlayer instance will be deleted. All resources will be released.
     */
    ~MoviePlayer();

    /**
     * @brief
     * Initialize MoviePlayer.
     *
     * @retval Status_Success                   MoviePlayer is initialized successfully.
     * @retval Status_FailedToInitialize        Failed to initialize the MoviePlayer.
     * @retval Status_OutOfMemory               Memory allocation failed.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is instantiated.
     *
     * @post
     * MoviePlayer is initialized.
     *
     * @details
     * MoviePlayer initialized. Non-track specific objects are created and initialized.
     *
     * MoviePlayer state transition on success:
     * State_UnInitialized -> State_Initialized.
     */
    movie::Status Initialize() NN_NOEXCEPT;

    /**
     * @brief
     * Set native window handle to  MoviePlayer.
     *
     * @param[in] NativeWindowHandle            NativeWindowHandle to set.
     *
     * @retval Status_Success                   MoviePlayer is initialized successfully.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is instantiated.
     *
     * @details
     * Native window handle is set to  MoviePlayer. This window handle will be used by renderer.
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status SetViNativeWindowHandle(nn::vi::NativeWindowHandle) NN_NOEXCEPT;

    /**
     * @brief
     * Register client events.
     *
     * @param[in] eventType                     Client event type to register.
     * @param[in] event                         Event to register.
     *
     * @retval Status_Success                   Client listener event is registered successfully.
     *
     * @pre
     * MoviePlayer is initialized.
     *
     * @post
     * Client event is registered with MoviePlayer.
     *
     * @details
     * The MoviePlayer signals an event for the registered event type, when the specific event trigger
     * conditions are met.
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status RegisterEvent(EventType eventType, nn::os::EventType *event) NN_NOEXCEPT;

    /**
     * @brief
     * Sets data source for playback.
     *
     * @param[in] mediaPath                     Media path for playback.
     *
     * @retval Status_Success                   Data source is set successfully.
     * @retval Status_OperationFailed           An unknown error occurred.
     * @retval Status_TrackNotFound             No valid tracks found.
     * @retval Status_UnknownTrackType          Track type not known to MoviePlayer.
     * @retval Status_OutOfMemory               Memory allocation failed.
     *
     * @pre
     * MoviePlayer is instantiated.
     * MoviePlayer is State_Initialized state.
     *
     * @details
     * Client can use this API to set data source for playback. The MoviePlayer will probe the media
     * using extractors. MoviePlayer will return error there are no valid or supported tracks to play.
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status SetDataSource(const char* mediaPath) NN_NOEXCEPT;

    /**
     * @brief
     * Sets data source for playback.
     *
     * @param[in] streamReader                  A pointer to the client reader interface
     * @param[in] mediaExtension                Media extension(.mp4, .m4a,.wbm, .mkv )
     *
     * @retval Status_Success                   Data source is set successfully.
     * @retval Status_OperationFailed           An unknown error occurred.
     * @retval Status_TrackNotFound             No valid tracks found.
     * @retval Status_UnknownTrackType          Track type not known to MoviePlayer.
     * @retval Status_OutOfMemory               Memory allocation failed.
     *
     * @pre
     * MoviePlayer is instantiated.
     * MoviePlayer is State_Initialized state.
     *
     * @details
     * Client can use this API to set data source for playback. The MoviePlayer will probe the media
     * using extractors. MoviePlayer will return error there are no valid or supported tracks to play.
     *
     * Media Extension:
     * MP4 files        : ".mp4"
     * M4A files        : ".m4a"
     * WebM files       : ".webm"
     * Matroska files   : ".mp4"
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status SetDataSource(movie::ClientStreamReader &streamReader, const char* mediaExtension) NN_NOEXCEPT;

    /**
     * @brief
     * Sets data source for playback.
     *
     * @param[in] buffer                        Memory containing media data
     * @param[in] length                        Media data size in bytes in buffer
     *
     * @retval Status_Success                   Data source is set successfully.
     * @retval Status_OperationFailed           An unknown error occurred.
     * @retval Status_TrackNotFound             No valid tracks found.
     * @retval Status_UnknownTrackType          Track type not known to MoviePlayer.
     * @retval Status_OutOfMemory               Memory allocation failed.
     *
     * @pre
     * MoviePlayer is instantiated.
     * MoviePlayer is State_Initialized state.
     *
     * @details
     * Client can use this API to set data source for playback. The MoviePlayer will probe the URI
     * using extractors. MoviePlayer will return error there are no valid or supported tracks to play.
     *
     * Media Extension:
     * MP4 files        : ".mp4"
     * M4A files        : ".m4a"
     * WebM files       : ".webm"
     * Matroska files   : ".mp4"
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status SetDataSource(const void* buffer, int length, const char* mediaExtension) NN_NOEXCEPT;

    /**
     * @brief
     * Prepare MoviePlayer for playback.
     *
     * @retval Status_Success                   MoviePlayer is prepared and ready.
     * @retval Status_FailedToPrepare           Failed to prepare.
     * @retval Status_OperationFailed           An unknown error occurred.
     * @retval Status_OutOfMemory               Memory allocation failed.
     *
     * @pre
     * MoviePlayer is State_Initialized state.
     * Valid data source is set.
     *
     * @post
     * MoviePlayer is prepared and ready for playback.
     * If client has registered event for state change, the event is signaled when prepare is complete.
     *
     * @details
     * MoviePlayer will create decoders for first audio and video track present. If audio track is
     * present, audio renderer will be created.  If video track is present video renderer will be created.
     *
     * MoviePlayer state transition on success:
     * State_Initialized -> State_Prepared.
     */
    movie::Status Prepare(bool syncMode = true) NN_NOEXCEPT;

    /**
     * @brief
     * Start playback of media.
     *
     * @retval Status_Success                   Playback is started.
     * @retval Status_OperationFailed           An unknown error occurred.
     * @retval Status_OutOfMemory               Memory allocation failed.
     *
     * @pre
     * MoviePlayer is State_Prepared state.
     *
     * @post
     * MoviePlayer is playing the media file.
     * If client has registered event for state change, the event is signaled when playback is started.
     *
     * @details
     * MoviePlayer will start rendering audio video data.
     *
     * MoviePlayer state transition on success:
     * State_Prepared -> State_Started.
     */
    movie::Status Start() NN_NOEXCEPT;

    /**
     * @brief
     * Stop playback of media.
     *
     * @retval Status_Success                   Playback is stopped.
     * @retval Status_OperationFailed           An unknown error occurred.
     * @retval Status_OutOfMemory               Memory allocation failed.
     *
     * @pre
     * MoviePlayer is State_Started or State_Stopped state
     *
     * @post
     * MoviePlayer stops playing the media file.
     * If client has registered event for state change, the event is signaled when playback is stopped.
     *
     * @details
     * MoviePlayer will stop rendering audio and video.
     *
     * MoviePlayer state transition on success:
     * State_Started -> State_Stopped.
     */
    movie::Status Stop() NN_NOEXCEPT;

    /**
     * @brief
     * Pause media playback.
     *
     * @retval Status_Success                   Pause is successful.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is State_Started or State_Paused state
     *
     * @post
     * MoviePlayer is paused.
     *
     * @details
     * MoviePlayer pauses rendering of audio and video data.
     *
     * MoviePlayer state transition on success:
     * State_Started -> State_Paused.
     * State_Paused  -> State_Paused.
     */
    movie::Status Pause();

    /**
     * @brief
     * Resumes media playback.
     *
     * @retval Status_Success                    Resume is successful.
     * @retval Status_OperationFailed            An unknown error occurred.
     *
     * @pre
     * MoviePlayer is State_Paused
     *
     * @post
     * MoviePlayer resumes rendering of audio and video data.
     *
     * @details
     * MoviePlayer resumes, rendering of audio and video data.
     *
     * MoviePlayer state transition on success:
     * State_Paused -> State_Started.
     */
    movie::Status Resume() NN_NOEXCEPT;

    /**
     * @brief
     * Return current playback position in microseconds.
     *
     * @param[out] playbackPositionUs           Current playback position in time, in microsecond units.
     *
     * @retval Status_Success                   Playback position is retrieved successfully.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is State_Initialized state.
     *
     * @details
     * The playback position of last frame (audio or video) in microseconds is retrieved and
     * returned to client
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status GetPlaybackPosition(int64_t *playbackPositionUs) const NN_NOEXCEPT;

    /**
     * @brief
     * Get playback duration of current media file in microseconds.
     *
     * @param[out] durationUs                   Playback duration.
     *
     * @retval Status_Success                   Duration is retrieved successfully.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is State_Initialized state.
     * Valid data source is set.
     *
     * @details
     * The playback duration (in microseconds) of the current media track returned to the client.
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status GetPlaybackDuration(int64_t *durationUs) const NN_NOEXCEPT;

    /**
     * @brief
     * Select default audio track for playback.
     *
     * @param[out] trackNumber                  Selected audio track number.
     *
     * @retval Status_Success                   Audio track is selected successfully.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is in State_Initialized state.
     * Valid data source is set.
     *
     * @details
     * Use this API to select audio track present in the media file.
     * MoviePlayer selects first audio track.
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status SelectAudioTrack(int *trackNumber) NN_NOEXCEPT;

    /**
     * @brief
     * Select default video track for playback.
     *
     * @param[out] trackNumber                  Selected video track number.
     *
     * @retval Status_Success                   video track is selected successfully.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is in State_Initialized state.
     * Valid data source is set.
     *
     * @details
     * Use this API to select video track present in the media file.
     * MoviePlayer selects first video track.
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status SelectVideoTrack(int *trackNumber) NN_NOEXCEPT;

    /**
     * @brief
     * Get number of tracks present in a media file.
     *
     * @param[out] trackCount                   Number of tracks present.
     *
     * @retval Status_Success                   Track count is retrieved successfully.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is State_Initialized state.
     * Valid data source is set.
     *
     * @details
     * The total tracks present in current media file is retrieved and returned to the client.
     * Valid track numbers are from 0 to (trackCount - 1)
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status GetTrackCount(int *trackCount) const NN_NOEXCEPT;

    /**
     * @brief
     * Select a particular media track.
     *
     * @param[in] trackNumber                   Track number to select.
     *
     * @retval Status_Success                   Requested track is selected successfully.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is in State_Initialized state.
     * Valid data source is set.
     *
     * @details
     * Use this API to select a particular track present in the media files.
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status SelectTrack(int trackNumber) NN_NOEXCEPT;

    /**
     * @brief
     * Un-select a particular media track.
     *
     * @param[in] trackNumber                   Track number to unselect.
     *
     * @retval Status_Success                   Requested track is deselected successfully.
     * @retval Status_ErrorBadValue             Invalid track number
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is in State_Initialized state.
     * Valid data source is set.
     *
     * @post
     * Data from unselected tracks are not extracted.
     *
     * @details
     * The requested track is unselected. Media data from unselected track is discarded.
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status UnSelectTrack(int trackNumber) NN_NOEXCEPT;

    /**
     * @brief
     * Get track type.
     *
     * @param[in] trackNumber                   Track number to retrieve track type.
     * @param[out] trackType                    Track type for the requested track.
     *
     * @retval Status_Success                   Track type is retrieved successfully.
     * @retval Status_ErrorBadValue             Invalid track number.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is in State_Initialized state.
     * Valid data source is set.
     *
     * @details
     * The track type for the requested track number is retrieved and returned to client.
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status GetTrackType(int trackNumber, TrackType *trackType) const NN_NOEXCEPT;

    /**
     * @brief
     * Reset internal states of MoviePlayer.
     *
     * @retval Status_Success                   MoviePlayer internal states are reset successfully.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is instantiated.
     *
     * @details
     * MoviePlayer is reset. This can be used when MoviePlayer returns async error.
     * A new media file can be set to MoviePlayer for playback.
     *
     * MoviePlayer state transition on success:
     * Any state -> State_Initialized.
     */
    movie::Status Reset() NN_NOEXCEPT;

    /**
     * @brief
     * Enable looping mode
     *
     * @param[in] loop                          Infinite looping mode.
     *
     * @retval Status_Success                   Loop flag is set to MoviePlayer.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is instantiated.
     *
     * @details
     * Use this API to set looping mode. Once set, the MoviePlayer will loop media file until
     * it is reset. This can be used to play a single media file indefinitely. When end of stream
     * is reached MoviePlayer will start playback from the beginning.
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status SetLooping(bool loop) NN_NOEXCEPT;

    /**
     * @brief
     * Get looping mode
     *
     * @param[out] loop                          Infinite looping mode.
     *
     * @retval Status_Success                    Return current playback loop mode.
     * @retval Status_OperationFailed            An unknown error occurred.
     *
     * @pre
     * MoviePlayer is instantiated.
     *
     * @details
     * Current playback loop mode is returned to client.
     *
     * MoviePlayer State:
     * No changes to MoviePlayer state.
     *
     */
    movie::Status GetLooping(bool *loop) const NN_NOEXCEPT;

    /**
     * @brief
     * Get current MoviePlayer state.
     *
     * @param[out] state                        MoviePlayer state.
     *
     * @retval Status_Success                   Current state of MoviePlayer is returned.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is instantiated.
     *
     * @details
     * Current state of MoviePlayer is returned.
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status GetState(State *state) NN_NOEXCEPT;

    /**
     * @brief
     * Get track information for a given track number.
     *
     * @param[in] trackNumber                   Track number to get track info
     * @param[out] trackInfo                    Track info to be filled.
     *
     * @retval Status_Success                   Track info is returned.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is in State_Initialized state.
     * Valid data source is set.
     *
     * @details
     * TrackInfo Properties are filled for the given track number.
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status GetTrackInfo(int trackNumber, TrackInfo* trackInfo) NN_NOEXCEPT;

    /**
     * @brief
     * Seek to specific time in the media file.
     *
     * @param[in] seekTimeUs                     Time in microseconds to seek.
     *
     * @retval Status_Success                    Seek is successful
     * @retval Status_EndOfStream                End of stream is reached during seek
     * @retval Status_OperationFailed            An unknown error occurred.
     *
     * @pre
     * MoviePlayer is State_Paused state.
     *
     * @post
     * If client has registered event for state change, the event is signaled
     * 1. when seek starts.
     * 2. when seek is completed.
     *
     * @details
     * MoviePlayer will seek the data source using the seek value (time) provided.
     * When Seek completed event is signaled to client. The actual seek time may be different than
     * the requested seek time. If the extractor is not able to find sync frame at requested seek time,
     * actual seek position will be different.
     * The actual seek position can be retrieved using GetLastSeekPosition() API.
     *
     * MoviePlayer state transition on success:
     * State_Paused  -> State_Seeking, when seek starts.
     * State_Seeking -> State_SeekCompleted, when seek completed.
     */
    movie::Status Seek(int64_t seekTimeUs, bool syncMode = true) NN_NOEXCEPT;

    /**
     * @brief
     * Return last seek position to client.
     *
     * @param[out] lastSeekTimeUs                Last seek position in microseconds.
     *
     * @pre
     * MoviePlayer is in State_Initialized state.
     *
     * @details
     * The last seek position is returned to client. The last seek time may be different than the requested seek time.
     * If the extractor is not able to find sync frame at requested seek time, actual seek position will be different.
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status GetLastSeekPosition(int64_t *lastSeekTimeUs) const NN_NOEXCEPT;

    /**
     * @brief
     * Return last movie error
     *
     * @return                                  Return last error.
     *
     * @pre
     * MoviePlayer is in State_Initialized state.
     *
     * @details
     * Client can use this API to get last error from player.
     * When client receives error event, they call this API to get the error codes.
     */
    movie::Status GetLastError() NN_NOEXCEPT;

    /**
     * @brief
     * Get string value of Player state enumeration.
     *
     * @return                                  Return string for state enum
     *
     * @pre
     * MoviePlayer is instantiated.
     *
     * @details
     * Client can use this API to get string value of Player state enumeration.
     */
    const char* StateToString(State state);

    /**
     * @brief
     * Select video decoder mode.
     *
     * @param[in] videoDecoderMode              Video decoder mode to use
     *
     * @retval Status_Success                   Requested mode is available and will be used by decoder.
     * @retval Status_OperationFailed           An unknown error occurred.
     *
     * @pre
     * MoviePlayer is in State_Initialized state.
     * This API need to be called before Prepare(), otherwise no impact
     *
     * @details
     * Use this API to select video decoder mode. By default player uses "DecoderMode_Cpu"
     *
     * MoviePlayer state transition on success:
     * No impact on MoviePlayer state.
     */
    movie::Status SetVideoDecoderMode(movie::DecoderMode videoDecoderMode = movie::DecoderMode_Cpu) NN_NOEXCEPT;

private:
    NN_DISALLOW_COPY(MoviePlayer);
    NN_DISALLOW_MOVE(MoviePlayer);

    friend void MoviePlayerThreadFunction(void *arg);

    // Player private API's.
    void OnPrepare();

    enum PlayerEventType
    {
        PlayerEventType_Unknown = 0,
        PlayerEventType_Prepare = 1,
        PlayerEventType_Seek = 2,
        PlayerEventType_AudioRenderComplete = 3,
        PlayerEventType_VideoRenderComplete = 4,
        PlayerEventType_ThreadExit = 5,
    };

    struct EventUserData
    {
        EventUserData(nn::os::EventType *event, PlayerEventType eventType)
            : m_Event(event),
              m_EventType(eventType)
        { }
        nn::os::EventType *m_Event;
        PlayerEventType m_EventType;
    };

    movie::Status GetContainerType(const char* mediaPath, movie::ContainerType *containertype);
    movie::Status ValidateStateTransition(State newState);
    movie::Status SetState(State state);
    void UpdateLastErrorAndSignalClient(movie::Status lastError);
    nn::os::MultiWaitHolderType* CreateEventHolder(nn::os::EventType *event, PlayerEventType eventType);
    movie::Status RegisterPlayerEvent(nn::os::EventType *event, PlayerEventType eventType);
    void SetAudioPlaybackComplete();
    void SetVideoPlaybackComplete();
    bool IsMediaPlaybackComplete();
    movie::Status ValidateTrackNumber(int trackNumber);
    void SetLastError(movie::Status movieStatus);
    void DeleteUserData(EventUserData* userData);
    movie::Status UpdateTrackPropertiesForPlayer();
    movie::Status CreateExtractorForPlayer(const char* mediaPath);
    movie::Status HandlePlaybackCompleteForPlayer();
    bool IsInRequestedState(State state);
    void DeletePlayerOwnedObjects();
    void FlushInputAndOutputHandlers();
    void FlushDecoders();
    // Last seek position.
    int64_t m_LastSeekPositionUs;

    // Current state of MoviePlayer
    State m_State;

    // Client registered events.
    nn::os::EventType *m_ErrorEvent;
    nn::os::EventType *m_StateChangedEvent;

    // VI Native window handle
    nn::vi::NativeWindowHandle  m_NativeWindow;

    // Track info of current media file
    TrackInfo m_TrackInfo;

    // MoviePlayer looping option.
    bool m_Loop;

    // Movie:: Extractor for reading media data.
    movie::Extractor *m_Extractor;

    // Movie::Decoder for decoding audio data.
    movie::Decoder *m_AudioDecoder;
    movie::DecoderType m_AudioDecoderType;

    // Movie::Decoder for decoding video data.
    movie::Decoder *m_VideoDecoder;
    movie::DecoderType m_VideoDecoderType;

    MovieVideoInputHandler *m_MovieVideoInputHandler;
    MovieVideoOutputHandler *m_MovieVideoOutputHandler;
    MovieAudioInputHandler *m_MovieAudioInputHandler;
    MovieAudioOutputHandler *m_MovieAudioOutputHandler;

    int m_NumberOfTracks;
    int m_AudioSampleRate;
    int m_NumberOfAudioChannels;
    int m_AudioTrackNumber;
    int m_VideoTrackNumber;
    int m_Width;
    int m_Height;

    movie::MediaData m_TrackFormat;

    nn::os::EventType m_AudioInputEvent;
    nn::os::EventType m_AudioOutputEvent;
    nn::os::EventType m_AudioFormatChangedEvent;
    nn::os::EventType m_AudioErrorEvent;

    nn::os::EventType m_VideoInputEvent;
    nn::os::EventType m_VideoOutputEvent;
    nn::os::EventType m_VideoFormatChangedEvent;
    nn::os::EventType m_VideoErrorEvent;

    movie::DecoderEvents m_AudioDecoderEvents;
    movie::DecoderEvents m_VideoDecoderEvents;

    MovieDecoderEventHandler *m_MovieDecoderEventHandler;
    MediaClock *m_MediaClock;

    uint64_t m_CoreMask;

    nn::os::ThreadType m_ThreadType;
    size_t m_ThreadStackSize;
    void* m_ThreadStack;

public:
    nn::os::MultiWaitType m_PlayerMultiWait;

private:
    nn::os::EventType m_PrepareEvent;
    nn::os::EventType m_SeekEvent;
    nn::os::EventType m_AudioRenderCompleteEvent;
    nn::os::EventType m_VideoRenderCompleteEvent;
    nn::os::EventType m_ThreadExitEvent;

    nn::os::MultiWaitHolderType m_PrepareEventHolder;
    nn::os::MultiWaitHolderType m_SeekEventHolder;
    nn::os::MultiWaitHolderType m_AudioRenderCompleteEventHolder;
    nn::os::MultiWaitHolderType m_VideoRenderCompleteEventHolder;
    nn::os::MultiWaitHolderType m_ThreadExitEventHolder;

    movie::Status m_LastError;

    bool m_AudioPlaybackComplete;
    bool m_VideoPlaybackComplete;
    bool m_EventsInitialized;
    bool m_ThreadCreated;
    bool m_ThreadStarted;

    Mutex m_StateLock;
    Mutex m_ApiLock;
    int64_t m_MediaDurationUs;
    movie::DecoderMode m_AudioDecoderMode;
    movie::DecoderMode m_VideoDecoderMode;
    movie::DecoderOutputFormat m_AudioDecoderOutputFormat;
    movie::DecoderOutputFormat m_VideoDecoderOutputFormat;
};
