﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "MovieDecoderEventHandler.h"
/**
 * @brief
 * MovieDecoderEventHandler class for handling audio and video decoder events
 *
 * @details
 *
 * Purpose:
 *     The purpose of @class MovieDecoderEventHandler is to respond to movie decoder events. @class MovieDecoderEventHandler
 * registers events with both audio and video decoder. When any of events are signaled from the decoder,
 * @class MovieDecoderEventHandler acquires information about the event and sends message to input or output handler message queues.
 *
 * Setup:
 *     MoviePlayer will create an instance of @class MovieDecoderEventHandler. The constructor is provided with input/output
 * handles. During initialization a thread is created for receiving events from decoder. The thread waits on a multi wait.
 * Other @class MovieDecoderEventHandler resources like input buffer list are initialized.
 *
 * Main Processing:
 *     @class MovieDecoderEventHandler thread multi-wait will be signaled by movie decoders when they have input, output
 * format change or any error occurs. @class MovieDecoderEventHandler identifies the events based on even user data and delegates
 * the event information to respective input/output handlers.
 *
 * Teardown:
 *    When @class MovieDecoderEventHandler receives thread exit event, the thread will exit and all resources are freed.
 *
 */
MovieDecoderEventHandler::MovieDecoderEventHandler(movie::Decoder* audioDecoder, movie::Decoder* videoDecoder,
    MovieVideoInputHandler *videoInputHandler, MovieVideoOutputHandler *videoOutputHandler,
    MovieAudioInputHandler *audioInputHandler, MovieAudioOutputHandler *audioOutputHandler)
    : m_VideoDecoder(videoDecoder),
    m_AudioDecoder(audioDecoder),
    m_MovieVideoInputHandler(videoInputHandler),
    m_MovieVideoOutputHandler(videoOutputHandler),
    m_MovieAudioInputHandler(audioInputHandler),
    m_MovieAudioOutputHandler(audioOutputHandler),
    m_ThreadStackSize(1024 * 128),
    m_ThreadStack(nullptr),
    m_AudioFormatChangeEventReceived(false),
    m_VideoFormatChangeEventReceived(false),
    m_AudioDecoderInFlush(false),
    m_VideoDecoderInFlush(false)
{
    if( m_AudioDecoder == nullptr )
    {
        m_AudioFormatChangeEventReceived = true;
    }
    if( m_VideoDecoder == nullptr )
    {
        m_VideoFormatChangeEventReceived = true;
    }
}

MovieDecoderEventHandler::~MovieDecoderEventHandler()
{
    nn::os::SignalEvent(&m_ThreadExitEvent);
    nn::os::WaitThread(&m_ThreadType);
    nn::os::DestroyThread(&m_ThreadType);

    // Finalize thread exit event holder.
    nn::os::UnlinkMultiWaitHolder(&m_ThreadExitEventHolder);
    nn::os::UnlinkMultiWaitHolder(&m_AudioInputEventHolder);
    nn::os::UnlinkMultiWaitHolder(&m_AudioOutputEventHolder);
    nn::os::UnlinkMultiWaitHolder(&m_AudioFormatChangedEventHolder);
    nn::os::UnlinkMultiWaitHolder(&m_AudioErrorEventHolder);
    nn::os::UnlinkMultiWaitHolder(&m_VideoInputEventHolder);
    nn::os::UnlinkMultiWaitHolder(&m_VideoOutputEventHolder);
    nn::os::UnlinkMultiWaitHolder(&m_VideoFormatChangedEventHolder);
    nn::os::UnlinkMultiWaitHolder(&m_VideoErrorEventHolder);

    DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_ThreadExitEventHolder));
    DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_AudioInputEventHolder));
    DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_AudioOutputEventHolder));
    DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_AudioFormatChangedEventHolder));
    DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_AudioErrorEventHolder));
    DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_VideoInputEventHolder));
    DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_VideoOutputEventHolder));
    DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_VideoFormatChangedEventHolder));
    DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_VideoErrorEventHolder));

    // Finalize thread exit event holder.
    nn::os::FinalizeMultiWaitHolder(&m_ThreadExitEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_AudioInputEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_AudioOutputEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_AudioFormatChangedEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_AudioErrorEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_VideoInputEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_VideoOutputEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_VideoFormatChangedEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_VideoErrorEventHolder);

    // Finalize thread exit event.
    nn::os::FinalizeEvent(&m_ThreadExitEvent);

    nn::os::FinalizeEvent(&m_FormatChangedPlayerEvent);

    // Finalize decoder multiwait.
    nn::os::FinalizeMultiWait(&m_DecoderMultiWait);

    // Delete thread stack memory.
    if( m_ThreadStack != NULL )
    {
        free(m_ThreadStack);
        m_ThreadStack = nullptr;
    }
}

movie::Status MovieDecoderEventHandler::Initialize(uint64_t coreMask)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::Result nnResult = nn::ResultSuccess();

    if( coreMask )
    {
        movie::SetCoreMask(coreMask);
    }

    // Initialize multiWait for receivng decoder events.
    nn::os::InitializeMultiWait(&m_DecoderMultiWait);

    // Initialize Event for MovieDecoderEventHandler thread exit.
    nn::os::InitializeEvent(&m_ThreadExitEvent, false, nn::os::EventClearMode_ManualClear);

    // Initialize Events for Player to wait for format changes.
    nn::os::InitializeEvent(&m_FormatChangedPlayerEvent, false, nn::os::EventClearMode_ManualClear);

    // Register thread exit event holder.
    RegisterEvent(&m_ThreadExitEvent, DecoderEventType_ThreadExit);

    // Create movie decoder event handler thread.
    m_ThreadStack = aligned_alloc(nn::os::ThreadStackAlignment, m_ThreadStackSize);
    if( m_ThreadStack == NULL )
    {
        movieStatus = movie::Status_OutOfMemory;
    }
    else
    {
        nnResult = nn::os::CreateThread(&m_ThreadType,
            &MovieDecoderEventHandlerThreadFunction,
            ( void* )this,
            m_ThreadStack,
            m_ThreadStackSize,
            nn::os::DefaultThreadPriority);
        if( nnResult.IsSuccess() )
        {
            nn::os::SetThreadName(&m_ThreadType, "MovieDecoderEventHandler");
            nn::os::StartThread(&m_ThreadType);
        }
        else
        {
            movieStatus = movie::Status_FailedToCreateThread;
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderEventHandler::RegisterEvent(nn::os::EventType *event, DecoderEventType eventType)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::MultiWaitHolderType *eventHolder = CreateEventHolder(event, eventType);
    nn::os::LinkMultiWaitHolder(&m_DecoderMultiWait, eventHolder);
    return movieStatus;
}

nn::os::MultiWaitHolderType* MovieDecoderEventHandler::CreateEventHolder(nn::os::EventType *event, DecoderEventType eventType)
{
    switch( eventType )
    {
        case DecoderEventType_VideoInputBuffer:
        {
            EventUserData *userData = new EventUserData(event, DecoderEventType_VideoInputBuffer);
            nn::os::InitializeMultiWaitHolder(&m_VideoInputEventHolder, event);
            nn::os::SetMultiWaitHolderUserData(&m_VideoInputEventHolder, ( uintptr_t ) userData);
            return &m_VideoInputEventHolder;
        }
        case DecoderEventType_VideoOutputBuffer:
        {
            EventUserData *userData = new EventUserData(event, DecoderEventType_VideoOutputBuffer);
            nn::os::InitializeMultiWaitHolder(&m_VideoOutputEventHolder, event);
            nn::os::SetMultiWaitHolderUserData(&m_VideoOutputEventHolder, ( uintptr_t ) userData);
            return &m_VideoOutputEventHolder;
        }
        case DecoderEventType_VideoFormatChanged:
        {
            EventUserData *userData = new EventUserData(event, DecoderEventType_VideoFormatChanged);
            nn::os::InitializeMultiWaitHolder(&m_VideoFormatChangedEventHolder, event);
            nn::os::SetMultiWaitHolderUserData(&m_VideoFormatChangedEventHolder, ( uintptr_t ) userData);
            return &m_VideoFormatChangedEventHolder;
        }
        case DecoderEventType_VideoError:
        {
            EventUserData *userData = new EventUserData(event, DecoderEventType_VideoError);
            nn::os::InitializeMultiWaitHolder(&m_VideoErrorEventHolder, event);
            nn::os::SetMultiWaitHolderUserData(&m_VideoErrorEventHolder, ( uintptr_t ) userData);
            return &m_VideoErrorEventHolder;
        }
        case DecoderEventType_AudioInputBuffer:
        {
            EventUserData *userData = new EventUserData(event, DecoderEventType_AudioInputBuffer);
            nn::os::InitializeMultiWaitHolder(&m_AudioInputEventHolder, event);
            nn::os::SetMultiWaitHolderUserData(&m_AudioInputEventHolder, ( uintptr_t ) userData);
            return &m_AudioInputEventHolder;
        }
        case DecoderEventType_AudioOutputBuffer:
        {
            EventUserData *userData = new EventUserData(event, DecoderEventType_AudioOutputBuffer);
            nn::os::InitializeMultiWaitHolder(&m_AudioOutputEventHolder, event);
            nn::os::SetMultiWaitHolderUserData(&m_AudioOutputEventHolder, ( uintptr_t ) userData);
            return &m_AudioOutputEventHolder;
        }
        case DecoderEventType_AudioFormatChanged:
        {
            EventUserData *userData = new EventUserData(event, DecoderEventType_AudioFormatChanged);
            nn::os::InitializeMultiWaitHolder(&m_AudioFormatChangedEventHolder, event);
            nn::os::SetMultiWaitHolderUserData(&m_AudioFormatChangedEventHolder, ( uintptr_t ) userData);
            return &m_AudioFormatChangedEventHolder;
        }
        case DecoderEventType_AudioError:
        {
            EventUserData *userData = new EventUserData(event, DecoderEventType_AudioError);
            nn::os::InitializeMultiWaitHolder(&m_AudioErrorEventHolder, event);
            nn::os::SetMultiWaitHolderUserData(&m_AudioErrorEventHolder, ( uintptr_t ) userData);
            return &m_AudioErrorEventHolder;
        }
        case DecoderEventType_ThreadExit:
        {
            EventUserData *userData = new EventUserData(event, DecoderEventType_ThreadExit);
            nn::os::InitializeMultiWaitHolder(&m_ThreadExitEventHolder, event);
            nn::os::SetMultiWaitHolderUserData(&m_ThreadExitEventHolder, ( uintptr_t ) userData);
            return &m_ThreadExitEventHolder;
        }
        default:
            break;
    }
    return nullptr;
}

movie::Status MovieDecoderEventHandler::VideoInputBufferAvailableEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    int32_t index = -1;
    movie::Buffer buffer;
    int32_t numberOfBuffersAvailable = 0;
    if( ( m_VideoDecoder != nullptr ) && ( m_MovieVideoInputHandler != nullptr ) )
    {
        movieStatus = m_VideoDecoder->AcquireInputBufferIndex(&index, &numberOfBuffersAvailable);
        if( movieStatus == movie::Status_Success )
        {
            m_MovieVideoInputHandler->AddBufferToVideoIndexList(index);
            m_MovieVideoInputHandler->SignalVideoInputBufferAvailable();
            for( int32_t i = 0; i < numberOfBuffersAvailable; i++ )
            {
                int32_t remainingBuffers = 0;
                movieStatus = m_VideoDecoder->AcquireInputBufferIndex(&index, &remainingBuffers);
                if( movieStatus == movie::Status_Success )
                {
                    m_MovieVideoInputHandler->AddBufferToVideoIndexList(index);
                    m_MovieVideoInputHandler->SignalVideoInputBufferAvailable();
                }
            }
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderEventHandler::VideoOutputBufferAvailableEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    int32_t index = -1;
    int64_t presentationTimeUs = 0ll;
    uint32_t flags = 0;
    int32_t numberOfBuffersAvailable = 0;

    movie::Buffer buffer;
    if( ( m_VideoDecoder != nullptr ) && ( m_MovieVideoOutputHandler != nullptr ) )
    {
        movieStatus = m_VideoDecoder->AcquireOutputBufferIndex(&index, &presentationTimeUs, &flags, &numberOfBuffersAvailable);
        if( ( movieStatus == movie::Status_Success ) || ( movieStatus == movie::Status_EndOfStream ) )
        {
            m_MovieVideoOutputHandler->OnOutputAvailable(index, presentationTimeUs, flags);
            for( int32_t i = 0; i < numberOfBuffersAvailable; i++ )
            {
                int32_t remainingBuffers = 0;
                movieStatus = m_VideoDecoder->AcquireOutputBufferIndex(&index, &presentationTimeUs, &flags, &remainingBuffers);
                if( ( movieStatus == movie::Status_Success ) || ( movieStatus == movie::Status_EndOfStream ) )
                {
                    m_MovieVideoOutputHandler->OnOutputAvailable(index, presentationTimeUs, flags);
                }
            }
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderEventHandler::VideoFormatChangedEvent()
{
    m_VideoFormatChangeEventReceived = true;
    CheckAndSignalFormatChangeWaiter();
    return movie::Status_Success;
}

movie::Status MovieDecoderEventHandler::VideoErrorEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    return movieStatus;
}

movie::Status MovieDecoderEventHandler::AudioInputBufferAvailableEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    int32_t index = -1;
    movie::Buffer buffer;
    int32_t numberOfBuffersAvailable = 0;
    if( ( m_AudioDecoder != nullptr ) && ( m_MovieAudioInputHandler != nullptr ))
    {
        movieStatus = m_AudioDecoder->AcquireInputBufferIndex(&index, &numberOfBuffersAvailable);
        if( movieStatus == movie::Status_Success )
        {
            m_MovieAudioInputHandler->AddBufferToAudioIndexList(index);
            m_MovieAudioInputHandler->SignalAudioInputBufferAvailable();
            for( int32_t i = 0; i < numberOfBuffersAvailable; i++ )
            {
                int32_t remainingBuffers = 0;
                movieStatus = m_AudioDecoder->AcquireInputBufferIndex(&index, &remainingBuffers);
                if( movieStatus == movie::Status_Success )
                {
                    m_MovieAudioInputHandler->AddBufferToAudioIndexList(index);
                    m_MovieAudioInputHandler->SignalAudioInputBufferAvailable();
                }
            }
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderEventHandler::AudioOutputBufferAvailableEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    int32_t index = -1;
    int64_t presentationTimeUs= 0ll;
    uint32_t flags = 0;
    int32_t numberOfBuffersAvailable = 0;

    if( ( m_AudioDecoder != nullptr ) && ( m_MovieAudioOutputHandler != nullptr ) )
    {
        movieStatus = m_AudioDecoder->AcquireOutputBufferIndex(&index, &presentationTimeUs, &flags, &numberOfBuffersAvailable);
        if( ( movieStatus == movie::Status_Success ) || ( movieStatus == movie::Status_EndOfStream ) )
        {
            m_MovieAudioOutputHandler->OnOutputAvailable(index, presentationTimeUs, flags);
            for( int32_t i = 0; i < numberOfBuffersAvailable; i++ )
            {
                int32_t remainingBuffers = 0;
                movieStatus = m_AudioDecoder->AcquireOutputBufferIndex(&index, &presentationTimeUs, &flags, &remainingBuffers);
                if( ( movieStatus == movie::Status_Success ) || ( movieStatus == movie::Status_EndOfStream ) )
                {
                    m_MovieAudioOutputHandler->OnOutputAvailable(index, presentationTimeUs, flags);
                }
            }
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderEventHandler::AudioFormatChangedEvent()
{
    m_AudioFormatChangeEventReceived = true;
    CheckAndSignalFormatChangeWaiter();
    return movie::Status_Success;
}

movie::Status MovieDecoderEventHandler::AudioErrorEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    return movieStatus;
}

void MovieDecoderEventHandler::DeleteUserData(EventUserData* userData)
{
    if( userData != nullptr )
    {
        delete userData;
    }
}

void MovieDecoderEventHandler::CheckAndSignalFormatChangeWaiter()
{
    if( ( m_AudioFormatChangeEventReceived == true ) && ( m_VideoFormatChangeEventReceived == true ) )
    {
        nn::os::SignalEvent(&m_FormatChangedPlayerEvent);
        m_AudioFormatChangeEventReceived = false;
        m_VideoFormatChangeEventReceived = false;
        if( m_AudioDecoder == nullptr )
        {
            m_AudioFormatChangeEventReceived = true;
        }
        if( m_VideoDecoder == nullptr )
        {
            m_VideoFormatChangeEventReceived = true;
        }
    }
}

void MovieDecoderEventHandler::WaitForFormatChange()
{
    nn::os::TimedWaitEvent(&m_FormatChangedPlayerEvent, nn::TimeSpan::FromMilliSeconds(5000));
    nn::os::ClearEvent(&m_FormatChangedPlayerEvent);
}

void MovieDecoderEventHandlerThreadFunction(void *arg)
{
    MovieDecoderEventHandler *movieDecoderEventHandler = ( MovieDecoderEventHandler*) arg;
    bool exitThread = false;
    if( movieDecoderEventHandler != nullptr )
    {
        // Wait for events from movie::decoder and process them.
        for( ;; )
        {
            nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&movieDecoderEventHandler->m_DecoderMultiWait);
            uintptr_t userData = GetMultiWaitHolderUserData(holder);
            MovieDecoderEventHandler::EventUserData *eventUserData = ( MovieDecoderEventHandler::EventUserData* ) userData;
            if( eventUserData == nullptr )
            {
                break;
            }
            MovieDecoderEventHandler::DecoderEventType eventype = eventUserData->m_EventType;
            nn::os::EventType *event = eventUserData->m_Event;
            nn::os::ClearEvent(event);
            switch( eventype )
            {
            case MovieDecoderEventHandler::DecoderEventType_VideoInputBuffer:
                if( false == movieDecoderEventHandler->GetVideoDecoderFlushState() )
                {
                    movieDecoderEventHandler->VideoInputBufferAvailableEvent();
                }
                break;
            case MovieDecoderEventHandler::DecoderEventType_VideoOutputBuffer:
                if( false == movieDecoderEventHandler->GetVideoDecoderFlushState() )
                {
                    movieDecoderEventHandler->VideoOutputBufferAvailableEvent();
                }
                break;
            case MovieDecoderEventHandler::DecoderEventType_VideoFormatChanged:
                movieDecoderEventHandler->VideoFormatChangedEvent();
                break;
            case MovieDecoderEventHandler::DecoderEventType_VideoError:
                movieDecoderEventHandler->VideoErrorEvent();
                break;
            case MovieDecoderEventHandler::DecoderEventType_AudioInputBuffer:
                if( false == movieDecoderEventHandler->GetAudioDecoderFlushState() )
                {
                    movieDecoderEventHandler->AudioInputBufferAvailableEvent();
                }
                break;
            case MovieDecoderEventHandler::DecoderEventType_AudioOutputBuffer:
                if( false == movieDecoderEventHandler->GetAudioDecoderFlushState() )
                {
                    movieDecoderEventHandler->AudioOutputBufferAvailableEvent();
                }
                break;
            case MovieDecoderEventHandler::DecoderEventType_AudioFormatChanged:
                movieDecoderEventHandler->AudioFormatChangedEvent();
                break;
            case MovieDecoderEventHandler::DecoderEventType_AudioError:
                movieDecoderEventHandler->AudioErrorEvent();
                break;
            case MovieDecoderEventHandler::DecoderEventType_ThreadExit:
                exitThread = true;
                break;
            default:
                break;
            }

            if( exitThread == true )
            {
                break;
            }
        }
    }
}
