﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @brief
 * MovieVideoOutputHandler class for processing video data from video decoders
 *
 * @details
 * Purpose:
 *     The main purpose of @class MovieVideoOutputHandler is to get decoded video data from movie::decoder and
 * render it to an output device capable of displaying the video frames. This class receives callbacks from
 * @class MovieDecoderEventHandler when video output buffer is available. The buffer index is stored in a list.
 * @class MovieVideoOutputHandler creates a thread for each video decoder. Each thread performs video sync to maintain
 * video frame rate. A separate thread is used to render the video. This thread composites the video from all decoders
 * and submits to NVN for display. MovieVideoOutputHandler supports usage of CPU buffers and Native texture (NVN Texture).
 *
 * Setup:
 *     @class MovieDecoderPlayer will create an instance of @class MovieVideoOutputHandler. Based on decoder mode,
 * NvnVideoRenderer will be set up to allocate either NVN buffers or NVN textures. When the decoder is set up for
 * NVN texture, the decoder output will be in ABGR format. Renderer does not need to do any color conversion. If
 * decoder uses CPU buffers, the decoder output will be NV12YUV. In this case, renderer need to convert decoder output
 * from NV12YUV to ABGR format. When client calls Start() API, one thread is created to submit video frames to renderer.
 * This thread calls Draw() API to present video frames to NvnVideoRenderer. This thread paces video frames at
 * 60 frames per second. Separate video sync threads are created for each video decoder. These threads add video frames
 * to renderer list paced at their frame rate.
 *
 * Main Processing:
 *     When client calls Start() API, main rendering thread and all video sync threads are started. Video sync thread calls
 * PrepareFrameForRendering() API. If video frame is available it will be added to NvnVideoRenderer list. This process is
 * repeated for each sync thread. The main renderer thread calls Draw() API to present video frames to NvnVideoRenderer.
 * This API will check for any latest video frames. If no frame is available for a decoder, previous frame is submitted.
 * This approach is used to maintain frame rate of individual video files. Video files with different frame rate are supported.
 * When last video frame is received (checked using EOS flag), and if an event is registered it will be signaled.
 *
 * Teardown:
 *    Client needs to call Stop() API to stop MovieVideoOutputHandler. Any video decoder buffers remaining will be released
 * back to the decoder. When Finalize() API is called, the main render thread is stopped and resources are released. It also
 * stops all video sync threads.
 *
 */

#include "MovieVideoOutputHandler.h"
MovieVideoOutputHandler::MovieVideoOutputHandler(int numberOfVideos, movie::DecoderMode decoderMode)
    : m_NvnVideoRenderer(nullptr),
      m_NumberOfVideos(numberOfVideos),
      m_DecoderMode(decoderMode)
{
    NvnVideoRenderer::NvnOutputFormat nvnOutputFormat = NvnVideoRenderer::NvnOutputFormat_AbgrNvnTexture;
    if( m_DecoderMode == movie::DecoderMode_Cpu )
        nvnOutputFormat = NvnVideoRenderer::NvnOutputFormat_YuvNv12NvnBuffer;
    m_NvnVideoRenderer = new NvnVideoRenderer(m_NativeWindow, nvnOutputFormat);
    m_nvnOutputFormat = nvnOutputFormat;
    if( m_NvnVideoRenderer != nullptr )
        m_NvnVideoRenderer->Initialize(m_NumberOfVideos);
}

MovieVideoOutputHandler::~MovieVideoOutputHandler()
{
    if( m_NvnVideoRenderer != nullptr )
    {
        m_NvnVideoRenderer->Finalize();
        delete m_NvnVideoRenderer;
        m_NvnVideoRenderer = nullptr;
    }
}

movie::Status MovieVideoOutputHandler::Initialize()
{
    nn::os::InitializeMutex(&m_VideoMutexRender, false, 0);
    if( m_NvnVideoRenderer == nullptr )
        return movie::Status_OutOfMemory;
    return movie::Status_Success;
}

movie::Status MovieVideoOutputHandler::RegisterDecoder(movie::Decoder* decoder, int width, int height, movie::DecoderOutputFormat outputFormat, MediaClock *mediaClock, nn::os::EventType *renderComplete)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    if( decoder == nullptr || mediaClock == nullptr )
        return movie::Status_ErrorBadValue;
    RenderContext *renderContext = new RenderContext(decoder, width, height, outputFormat, m_DecoderMode, mediaClock, this, renderComplete);
    if( renderContext != nullptr )
    {
        nn::os::InitializeMutex(&renderContext->m_VideoMutex, false, 0);
        renderContext->m_VideoBuffers.reserve(64);
        m_RenderContextMap.insert(std::pair<movie::Decoder *, RenderContext *>(decoder, renderContext));
        m_NumberOfVideos++;
        movieStatus = movie::Status_Success;
    }
    return movieStatus;
}

movie::Status MovieVideoOutputHandler::UpdateDecoderProperties(movie::Decoder* decoder, int width, int height)
{
    if( decoder == nullptr )
        return movie::Status_ErrorBadValue;
    if( m_RenderContextMap.find(decoder) != m_RenderContextMap.end() )
    {
        RenderContext* &renderContext = m_RenderContextMap.find(decoder)->second;
        renderContext->m_Width = width;
        renderContext->m_Height = height;
        m_NvnVideoRenderer->RegisterDecoder(decoder, width, height, m_nvnOutputFormat);
    }
    return movie::Status_Success;
}

movie::Status MovieVideoOutputHandler::Finalize()
{
    if( m_ThreadCreated == true )
    {
        if( m_ThreadStarted == true )
        {
            m_ThreadDone = true;
            nn::os::WaitThread(&m_ThreadType);
            m_ThreadStarted = false;
            m_ThreadDone = false;
        }
        nn::os::DestroyThread(&m_ThreadType);
        m_ThreadCreated = false;
        if( m_Threadstack != nullptr )
        {
            free(m_Threadstack);
            m_Threadstack = nullptr;
        }
    }
    for( auto element : m_RenderContextMap )
    {
        if( element.second->m_ThreadCreated == true )
        {
            if( element.second->m_ThreadStarted == true )
            {
                element.second->m_ThreadDone = true;
                nn::os::WaitThread(&element.second->m_ThreadType);
                element.second->m_ThreadStarted = false;
                element.second->m_ThreadDone = false;
            }
            nn::os::DestroyThread(&element.second->m_ThreadType);
            element.second->m_ThreadCreated = false;
            if( element.second->m_Threadstack != nullptr )
            {
                free(element.second->m_Threadstack);
                element.second->m_Threadstack = nullptr;
            }
        }
        nn::os::FinalizeMutex(&element.second->m_VideoMutex);
        element.second->m_VideoBuffers.clear();
        delete element.second;
    }
    nn::os::FinalizeMutex(&m_VideoMutexRender);
    return movie::Status_Success;
}

movie::Status MovieVideoOutputHandler::Open()
{
    nn::os::LockMutex(&m_VideoMutexRender);
    movie::Status movieStatus = movie::Status_UnknownError;
    nn::Result nnResult = nn::ResultSuccess();
    m_ThreadstackSize = ( 1024 * 512 );
    m_Threadstack = aligned_alloc(nn::os::ThreadStackAlignment, m_ThreadstackSize);
    if( m_Threadstack == nullptr )
        return movie::Status_OutOfMemory;
    nnResult = nn::os::CreateThread(
        &m_ThreadType,
        &MovieVideoOutputHandlerThreadFunctionRender,
        ( void* ) this,
        m_Threadstack,
        m_ThreadstackSize,
        nn::os::DefaultThreadPriority);
    m_ThreadCreated = true;
    if( nnResult.IsSuccess() )
    {
        nn::os::SetThreadName(&m_ThreadType, "ThreadFunctionRender ");
        movieStatus = movie::Status_Success;
    }
    nn::os::ThreadFunction function = &MovieVideoOutputHandlerThreadFunctionAvSync;
    for( auto element : m_RenderContextMap )
    {
        nn::Result nnResult = nn::ResultSuccess();
        element.second->m_Threadstack = aligned_alloc(nn::os::ThreadStackAlignment, element.second->m_ThreadstackSize);
        if( element.second->m_Threadstack == nullptr )
            return movie::Status_OutOfMemory;
        nnResult = nn::os::CreateThread(
            &element.second->m_ThreadType,
            function,
            ( void* ) element.second,
            element.second->m_Threadstack,
            element.second->m_ThreadstackSize,
            nn::os::DefaultThreadPriority);
        element.second->m_ThreadCreated = true;
        if( nnResult.IsSuccess() )
        {
            nn::os::SetThreadName(&element.second->m_ThreadType, "MovieVideoOutputHandlerThread ");
            movieStatus = movie::Status_Success;
        }
    }
    nn::os::UnlockMutex(&m_VideoMutexRender);
    return movieStatus;
}

movie::Status MovieVideoOutputHandler::Start()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex(&m_VideoMutexRender);
    if( m_ThreadStarted == false )
    {
        nn::os::StartThread(&m_ThreadType);
        m_ThreadStarted = true;
        for( auto element : m_RenderContextMap )
        {
            m_NvnVideoRenderer->CreateVideoBufferAndTextures(element.second->m_Decoder, element.second->m_Width, element.second->m_Height);
            nn::os::StartThread(&element.second->m_ThreadType);
            element.second->m_ThreadStarted = true;
        }
    }
    nn::os::UnlockMutex(&m_VideoMutexRender);
    return movieStatus;
}

bool MovieVideoOutputHandler::AllFramesRendered()
{
    bool allVideosRendered = true;
    for( auto element : m_RenderContextMap )
    {
        if( element.second->m_AllFramesRendered == false )
            allVideosRendered = false;
    }
    return allVideosRendered;
}

movie::Status MovieVideoOutputHandler::Render(int64_t *timeToRenderUs)
{
    if( timeToRenderUs == nullptr )
        return movie::Status_ErrorBadValue;
    m_NvnVideoRenderer->Draw(timeToRenderUs);
    return movie::Status_Success;
}

void MovieVideoOutputHandler::CheckEndOfStreamAndSignalPlayer(RenderContext *renderContext, uint32_t flags)
{
    if( flags == movie::BufferFlags_EndOfStream )
    {
        renderContext->m_AllFramesRendered = true;
        if( renderContext->m_RenderComplete != nullptr )
            nn::os::SignalEvent(renderContext->m_RenderComplete);
    }
}

movie::Status MovieVideoOutputHandler::PrepareFrameForRendering(RenderContext *context)
{
    if( context == nullptr )
        return movie::Status_ErrorBadValue;
    if( m_RenderContextMap.find(context->m_Decoder) != m_RenderContextMap.end() )
    {
        RenderContext* &renderContext = m_RenderContextMap.find(context->m_Decoder)->second;
        VideoData* videoData = nullptr;
        MediaClock *mediaClock = nullptr;
        movie::Buffer videoBufferData;
        if( renderContext->m_VideoBuffers.size() > 0 )
            videoData = &renderContext->m_VideoBuffers.front();
        else
            return movie::Status_Success;
        mediaClock = renderContext->m_MediaClock;
        movie::Decoder* videoDecoder = renderContext->m_Decoder;
        movie::DecoderMode videoDecoderMode = renderContext->m_DecoderMode;
        movie::DecoderOutputFormat videoDecoderOutputFormat = renderContext->m_OutputFormat;
        int width = renderContext->m_Width;
        int height = renderContext->m_Height;
        int yStride = 0;
        int uvOffset = 0;
        int yOffset = 0;
        int colorSpace = 0;
        int64_t presentationTimeUs = videoData->presentationTimeUs;
        int64_t lastPresentationTimeUs = renderContext->m_LastPresentationTimeUs;
        uint32_t flags = videoData->flags;
        uint32_t index = videoData->index;
        renderContext->m_PlaybackPositionUs = presentationTimeUs;
        bool frameAvailableForRendering = false;
        void *handle = nullptr;
        int  bufferIndex = 0;
        size_t yuvBufferSize = 0;
        size_t rgbTextureHandleSize = 0;
        if( ( videoDecoderMode == movie::DecoderMode_Cpu ) &&
            ( videoDecoderOutputFormat == movie::DecoderOutputFormat_VideoColorNv12 ) )
        {
            m_NvnVideoRenderer->GetNvnVideoBuffer(videoDecoder, &bufferIndex, &handle, &yuvBufferSize);
            if( handle == nullptr )
            {
                CheckEndOfStreamAndSignalPlayer(renderContext, flags);
                return movie::Status_OutputBufferNotAvailable;
            }
            videoBufferData.SetDataAndCapacity(handle, yuvBufferSize);
            videoBufferData.SetRange(0, yuvBufferSize);
        }
        else if( ( videoDecoderMode == movie::DecoderMode_NativeTexture ) &&
            ( videoDecoderOutputFormat == movie::DecoderOutputFormat_VideoColorAbgr ) )
        {
            void *NvnDevice = nullptr;
            void *NvnSync = nullptr;
            m_NvnVideoRenderer->GetNvnSyncInfo(&NvnDevice, &NvnSync);
            videoBufferData.SetNvnDev(NvnDevice);
            videoBufferData.SetNvnSync(NvnSync);
            m_NvnVideoRenderer->GetNvnVideoTexture(videoDecoder, &bufferIndex, &handle, &rgbTextureHandleSize);
            if( handle == nullptr )
            {
                CheckEndOfStreamAndSignalPlayer(renderContext, flags);
                return movie::Status_OutputBufferNotAvailable;
            }
            videoBufferData.SetDataAndCapacity(handle, rgbTextureHandleSize);
            videoBufferData.SetRange(0, rgbTextureHandleSize);
        }
        else
        {
            return movie::Status_UnknownError;
        }
        movie::MediaData format;
        movie::Status status = videoDecoder->GetOutputBufferFormat(index, &format);
        if( status == movie::Status_Success )
        {
            format.FindInt32("nv12-y-stride", &yStride);
            format.FindInt32("nv12-uv-offset", &uvOffset);
            format.FindInt32("nv12-y-offset", &yOffset);
            format.Clear();
            status = videoDecoder->GetOutputFormat(&format);
            if( status == movie::Status_Success )
                format.FindInt32("nv12-colorspace", &colorSpace);
            movie::Status status = videoDecoder->GetOutputBuffer(index, &videoBufferData);
            if( status == movie::Status_Success )
            {
                videoDecoder->ReleaseOutputBufferIndex(index);
                if( renderContext->m_VideoBuffers.size() > 0 )
                    renderContext->m_VideoBuffers.erase(renderContext->m_VideoBuffers.begin());
                frameAvailableForRendering = true;
            }
            else
            {
                CheckEndOfStreamAndSignalPlayer(renderContext, flags);
                videoDecoder->ReleaseOutputBufferIndex(index);
                if( renderContext->m_VideoBuffers.size() > 0 )
                    renderContext->m_VideoBuffers.erase(renderContext->m_VideoBuffers.begin());
                m_NvnVideoRenderer->ReturnBufferToRenderer(videoDecoder, handle, bufferIndex);
                return status;
            }
        }
        bool dropFrame = AVSync(renderContext, mediaClock, presentationTimeUs);
        if( dropFrame == true )
        {
            CheckEndOfStreamAndSignalPlayer(renderContext, flags);
            m_NvnVideoRenderer->ReturnBufferToRenderer(videoDecoder, handle, bufferIndex);
            return movie::Status_Success;
        }
        if( presentationTimeUs <= lastPresentationTimeUs )
            frameAvailableForRendering = false;
        if( frameAvailableForRendering == true )
        {
            if( ( videoDecoderMode == movie::DecoderMode_Cpu ) && ( videoDecoderOutputFormat == movie::DecoderOutputFormat_VideoColorNv12 ) )
                m_NvnVideoRenderer->AddVideoFrameToRender(videoDecoder, handle, bufferIndex, width, height, yOffset, yStride, uvOffset, colorSpace, yuvBufferSize);
            else
                m_NvnVideoRenderer->AddTextureToRender(videoDecoder, handle, bufferIndex, width, height);
        }
        else
        {
            m_NvnVideoRenderer->ReturnBufferToRenderer(videoDecoder, handle, bufferIndex);
        }
        CheckEndOfStreamAndSignalPlayer(renderContext, flags);
    }
    return movie::Status_Success;
}//NOLINT(impl/function_size)

void MovieVideoOutputHandler::Stop(movie::Decoder* decoder)
{
    nn::os::LockMutex(&m_VideoMutexRender);
    if( m_RenderContextMap.find(decoder) != m_RenderContextMap.end() )
    {
        RenderContext* &renderContext = m_RenderContextMap.find(decoder)->second;
        nn::os::LockMutex(&renderContext->m_VideoMutex);
        if( renderContext->m_VideoBuffers.size() > 0 )
        {
            for( auto videoData : renderContext->m_VideoBuffers )
                decoder->ReleaseOutputBufferIndex(videoData.index);
        }
        renderContext->m_VideoBuffers.clear();
        renderContext->m_VideoBuffers.reserve(64);
        nn::os::UnlockMutex(&renderContext->m_VideoMutex);
        renderContext->m_VideoOutputBuffersReceived = 0;
        renderContext->m_AllFramesRendered = false;
        renderContext->m_MediaClock->ClearAnchorTime();
    }
    nn::os::UnlockMutex(&m_VideoMutexRender);
}

void MovieVideoOutputHandler::Stop()
{
    nn::os::LockMutex( &m_VideoMutexRender);
    for( auto element : m_RenderContextMap )
    {
        nn::os::LockMutex(&element.second->m_VideoMutex);
        if( element.second->m_VideoBuffers.size() > 0 )
        {
            for( auto videoData : element.second->m_VideoBuffers )
                element.second->m_Decoder->ReleaseOutputBufferIndex(videoData.index);
        }
        element.second->m_VideoBuffers.clear();
        element.second->m_VideoBuffers.reserve(64);
        nn::os::UnlockMutex(&element.second->m_VideoMutex);
        element.second->m_VideoOutputBuffersReceived = 0;
        element.second->m_AllFramesRendered = false;
        element.second->m_MediaClock->ClearAnchorTime();
    }
    nn::os::UnlockMutex( &m_VideoMutexRender);
}

void MovieVideoOutputHandler::Pause()
{
    nn::os::LockMutex(&m_VideoMutexRender);
    for( auto &element : m_RenderContextMap )
    {
        element.second->m_VideoOutputBuffersReceived = 0;
        element.second->m_IsPaused = true;
        element.second->m_MediaClock->ClearAnchorTime();
    }
    nn::os::UnlockMutex(&m_VideoMutexRender);
}

bool MovieVideoOutputHandler::IsPaused(RenderContext *renderContext)
{
    bool isPaused = true;
    for( auto &element : m_RenderContextMap )
    {
        if( element.second->m_IsPaused == false )
        {
            isPaused = false;
            break;
        }
    }
    return isPaused;
}

void MovieVideoOutputHandler::Flush()
{
    nn::os::LockMutex(&m_VideoMutexRender);
    for( auto element : m_RenderContextMap )
    {
        nn::os::LockMutex(&element.second->m_VideoMutex);
        element.second->m_VideoBuffers.clear();
        nn::os::UnlockMutex(&element.second->m_VideoMutex);
    }
    nn::os::UnlockMutex(&m_VideoMutexRender);
}

void MovieVideoOutputHandler::Resume()
{
    nn::os::LockMutex(&m_VideoMutexRender);
    for( auto &element : m_RenderContextMap )
        element.second->m_IsPaused = false;
    nn::os::UnlockMutex(&m_VideoMutexRender);
}

bool MovieVideoOutputHandler::DoneProcessing(RenderContext *renderContext)
{
    if( renderContext == nullptr )
        return true;
    return renderContext->m_ThreadDone;
}

bool MovieVideoOutputHandler::DoneRendering()
{
    return m_ThreadDone;
}

void MovieVideoOutputHandler::Close()
{
    nn::os::LockMutex(&m_VideoMutexRender);
    for( auto element : m_RenderContextMap )
    {
        element.second->m_ThreadStarted = true;
        if( element.second->m_DroppedVideoFrames > 0 )
            NN_LOG("AVSync: Dropped Frame count =  %d Video Decoder = 0x%x \n", element.second->m_DroppedVideoFrames, element.second->m_Decoder);
    }
    nn::os::UnlockMutex(&m_VideoMutexRender);
}

void MovieVideoOutputHandler::OnOutputAvailable(movie::Decoder* decoder, int32_t index, int64_t presentationTimeUs, uint32_t flags)
{
    if( m_RenderContextMap.find(decoder) != m_RenderContextMap.end() )
    {
        RenderContext* &renderContext = m_RenderContextMap.find(decoder)->second;
        VideoData videoData;
        videoData.index = index;
        videoData.presentationTimeUs = presentationTimeUs;
        videoData.flags = flags;
        renderContext->m_VideoOutputBuffersReceived ++;
        nn::os::LockMutex(&renderContext->m_VideoMutex);
        renderContext->m_VideoBuffers.push_back(videoData);
        nn::os::UnlockMutex(&renderContext->m_VideoMutex);
    }
}

int64_t MovieVideoOutputHandler::GetRealTimeUs(RenderContext *renderContext, MediaClock *mediaClock, int64_t mediaTimeUs, int64_t nowUs)
{
    int64_t realUs = 0;
    if( ( renderContext != nullptr ) && ( mediaClock != nullptr ) )
    {
        if( mediaClock->GetRealTimeForMediaTime(mediaTimeUs, &realUs) != movie::Status_Success )
        {
            realUs = nowUs + ( mediaTimeUs - renderContext->m_LastPresentationTimeUs );
            return realUs;
        }
    }
    return realUs;
}

bool MovieVideoOutputHandler::AVSync(RenderContext *renderContext, MediaClock *mediaClock, int64_t presentationTimeUs)
{
    if( renderContext == nullptr || mediaClock == nullptr )
        return true;
    bool frameDropped = false;
    renderContext->m_PresentationTimeUs = presentationTimeUs;
    if( renderContext->m_UpdateVideoStartTime == true )
    {
        renderContext->m_UpdateVideoStartTime = false;
        renderContext->m_VideoStartTime = Clock::GetNowUs();
        mediaClock->UpdateAnchorTime(renderContext->m_PresentationTimeUs, renderContext->m_VideoStartTime, INT64_MAX);
    }
    int64_t nowUs = Clock::GetNowUs();
    int64_t realTimeUs = GetRealTimeUs(renderContext, mediaClock, renderContext->m_PresentationTimeUs, nowUs);
    int64_t videoLatebyUs = nowUs - realTimeUs;
    renderContext->m_LastPresentationTimeUs = renderContext->m_PresentationTimeUs;
    renderContext->m_VideoNeedDelayTimeUs = videoLatebyUs * -1;
    if( videoLatebyUs > 40000 )
    {
        frameDropped = true;
        renderContext->m_DroppedVideoFrames += 1;
    }
    int64_t sleepTimeUs = Clock::GetNowUs() - renderContext->m_LastAvSyncTimeUs;
    int64_t overSleptTimeUs = 0;
    int64_t underSleptTimeUs = 0;
    if( sleepTimeUs > renderContext->m_LastSleepTimeUs )
        overSleptTimeUs = sleepTimeUs - renderContext->m_LastSleepTimeUs;
    else
        underSleptTimeUs = renderContext->m_LastSleepTimeUs - sleepTimeUs;

    if( overSleptTimeUs > 0)
        renderContext->m_VideoNeedDelayTimeUs = renderContext->m_VideoNeedDelayTimeUs - overSleptTimeUs;
    else if ( underSleptTimeUs > 0 )
        renderContext->m_VideoNeedDelayTimeUs = renderContext->m_VideoNeedDelayTimeUs + overSleptTimeUs;

    if( renderContext->m_VideoNeedDelayTimeUs < 0 )
        renderContext->m_VideoNeedDelayTimeUs = 0;

    renderContext->m_LastAvSyncTimeUs = Clock::GetNowUs();
    return frameDropped;
}

int64_t MovieVideoOutputHandler::GetVideoPostDelayUs(RenderContext *renderContext)
{
    int64_t videoNeedDelayTimeUs = 0;
    if( renderContext != nullptr )
    {
        if( renderContext->m_VideoNeedDelayTimeUs < 2000 )
            renderContext->m_VideoNeedDelayTimeUs = 2000;
        else if( renderContext->m_VideoNeedDelayTimeUs > 300000 )
            renderContext->m_VideoNeedDelayTimeUs = 300000;
        int64_t timeSinceAvSync = Clock::GetNowUs() - renderContext->m_LastAvSyncTimeUs;
        if( timeSinceAvSync > 0 && timeSinceAvSync < renderContext->m_VideoNeedDelayTimeUs )
            renderContext->m_VideoNeedDelayTimeUs = renderContext->m_VideoNeedDelayTimeUs - timeSinceAvSync;
        videoNeedDelayTimeUs = renderContext->m_VideoNeedDelayTimeUs;
        renderContext->m_LastSleepTimeUs = videoNeedDelayTimeUs;
    }
    return videoNeedDelayTimeUs;
}

movie::Status MovieVideoOutputHandler::GetPlaybackPosition(movie::Decoder* decoder, int64_t *playbackPositionUs)
{
    if( playbackPositionUs == nullptr )
        return movie::Status_ErrorBadValue;
    *playbackPositionUs = -1;
    if( m_RenderContextMap.find(decoder) != m_RenderContextMap.end() )
    {
        RenderContext* &renderContext = m_RenderContextMap.find(decoder)->second;
        *playbackPositionUs = renderContext->m_PlaybackPositionUs;
    }
    return movie::Status_Success;
}

bool MovieVideoOutputHandler::CanStartRendering()
{
    bool canStart = true;
    for( auto element : m_RenderContextMap )
    {
        if( element.second->m_VideoOutputBuffersReceived < 3 )
        {
            canStart = false;
            break;
        }
    }
    return canStart;
}

void MovieVideoOutputHandlerThreadFunctionAvSync(void *arg)
{
    MovieVideoOutputHandler::RenderContext *renderContext = ( MovieVideoOutputHandler::RenderContext* ) arg;
    if( renderContext != nullptr )
    {
        for( ;; )
        {
            if( true == renderContext->m_MovieVideoOutputHandler->DoneProcessing(renderContext) )
                break;
            nn::os::LockMutex(&renderContext->m_VideoMutex);
            if( false == renderContext->m_MovieVideoOutputHandler->IsPaused(renderContext) )
                renderContext->m_MovieVideoOutputHandler->PrepareFrameForRendering(renderContext);
            nn::os::UnlockMutex(&renderContext->m_VideoMutex);
            int64_t renderSleepTimeUs = renderContext->m_MovieVideoOutputHandler->GetVideoPostDelayUs(renderContext);
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(renderSleepTimeUs));
        }
    }
}

void MovieVideoOutputHandlerThreadFunctionRender(void *arg)
{
    MovieVideoOutputHandler*  videoOutputHandler = ( MovieVideoOutputHandler* ) arg;
    if( videoOutputHandler != nullptr )
    {
        int64_t renderSleepTimeUs60fps = 16666;
        int64_t renderSleepTimeUs = renderSleepTimeUs60fps;
        for( ;; )
        {
            if( true == videoOutputHandler->DoneRendering() )
                break;
            int64_t timeToRenderUs = 0;
            nn::os::LockMutex(&videoOutputHandler->m_VideoMutexRender);
            if( false == videoOutputHandler->IsPaused(nullptr) )
                videoOutputHandler->Render(&timeToRenderUs);
            nn::os::UnlockMutex(&videoOutputHandler->m_VideoMutexRender);
            if( timeToRenderUs > 0 )
                renderSleepTimeUs = renderSleepTimeUs60fps - timeToRenderUs;
            if( renderSleepTimeUs < 0 )
                renderSleepTimeUs = 1000;
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(renderSleepTimeUs));
        }
    }
}

