﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @examplesource{MovieDecoderMultiInstance.cpp,PageSampleMovieDecoderMultiInstance}
 *
 * @brief
 *  movie API による動画再生サンプル
 */
 /**
 * @page PageSampleMovieDecoderMultiInstance MovieDecoderMultiInstance
 * @tableofcontents
 *
 * @brief
 * movie API による動画再生を並列で実行するサンプルプログラムです。
 *
 * @section PageSampleMovieDecoderMultiInstance_SectionBrief 概要
 * NX 本体上で複数の mp4 ファイルを入力し、再生を行うサンプルです。
 *
 * @section PageSampleMovieDecoderMultiInstance_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/MovieDecoderMultiInstance
 * Samples/Sources/Applications/MovieDecoderMultiInstance @endlink 以下にあります。
 *
 * @section PageSampleMovieDecoderMultiInstance_SectionNecessaryEnvironment 必要な環境
 * 追加で必要となるものはありません。
 *
 * @section PageSampleMovieDecoderMultiInstance_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、自動的に指定した動画の再生が始まります。
 * 再生終了後は、自動的にプログラムは終了します。
 *
 * @section PageSampleMovieDecoderMultiInstance_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleMovieDecoderMultiInstance_SectionHowToExecute 実行手順
 *  @<MovieDecoderMultiInstance executable@> -n @<Number of media files@> @<Media path@> @<Media path@> ...
 *
 * 例（ホストPCから動画データを取得）
 * MovieDecoderPlayer.nca c:@\H264_AAC_1280x720_30sec_01.mp4 -me mp4
 *
 * @section PageSampleMovieDecoderMultiInstance_SectionDetail 解説
 * 指定した動画ファイルを NX 本体上で再生します。
 */

#include "MovieDecoderMultiInstance.h"
#include "MemoryHandler.h"
#include <movie/Utils.h>
#include <movie/Common.h>
#include <movie/Status.h>
#include <nn/init.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <string>
#include <cinttypes>
#include <memory>
#include <nn/mem/mem_StandardAllocator.h>
#include <nv/nv_MemoryManagement.h>

// Process startup setup
extern "C" void nninitStartup()
{
    FsInitHeap();
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
}

// MovieDecoderMultiInstance Player Usage
static void Usage()
{
    NN_LOG( "\n ERROR:: Invalid options \n");
}

// Process main entry
extern "C" void nnMain()
{
    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();
    // Set movie allocator callback functions
    movie::SetAllocator(MovieAllocate, MovieDeallocate, MovieReallocate, &MovieMemoryTracker());
    // Donate memory for graphics driver to work in
    uint8_t *graphicsHeap = nullptr;
    size_t graphicsHeapSize = 0;
    GetGraphicsHeap(&graphicsHeap, &graphicsHeapSize);
    nv::InitializeGraphics(graphicsHeap, graphicsHeapSize);
    // Set graphics allocator callback functions
    nv::SetGraphicsAllocator(MovieAllocate, MovieDeallocate, MovieReallocate, &CoreMemoryTracker());
    nv::SetGraphicsDevtoolsAllocator(MovieAllocate, MovieDeallocate, MovieReallocate, &CoreMemoryTracker());
    if( argc <= 1 )
    {
       Usage();
       return;
    }
    char* inputFileName[4];
    movie::DecoderMode videoDecoderMode = movie::DecoderMode_NativeTexture;
    int numberOfVideosToDecode = 0;
    bool invalidOptions = false;
    bool trackMemoryAllocation = false;
    bool setLooping = false;
    for(int options = 0; options < argc; options ++)
    {
        if( !strcmp(argv[options], "-n" ) )
        {
            options ++;
            invalidOptions = true;
            if( options < argc )
            {
                if( ( !strcmp(argv[options], "1") ) ||  ( !strcmp(argv[options], "2") ) ||
                    ( !strcmp(argv[options], "3") ) ||  ( !strcmp(argv[options], "4") ) )

                {
                    numberOfVideosToDecode = atoi(argv[options]);
                    for( int i = 0; i < numberOfVideosToDecode; i++ )
                        inputFileName[ i ] = argv[ options + i + 1  ];
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }
        if( !strcmp(argv[ options ], "-h") )
        {
            options++;
            invalidOptions = true;
            if( options < argc )
            {
                if( ( !strcmp(argv[ options ], "0") ) || ( !strcmp(argv[ options ], "1") ) )
                {
                    trackMemoryAllocation = atoi(argv[ options ]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }
        if( !strcmp(argv[ options ], "-dm") )
        {
            options++;
            invalidOptions = true;
            if( options < argc )
            {
                if( ( !strcmp(argv[ options ], "1") ) || ( !strcmp(argv[ options ], "2") ) )
                {
                    int mode = atoi(argv[ options ]);
                    if( mode == 1 )
                        videoDecoderMode = movie::DecoderMode_Cpu;
                    invalidOptions = false;
                    continue;
                }
            }
        }
        if( !strcmp(argv[ options ], "-l") )
        {
            options++;
            invalidOptions = true;
            if( options < argc )
            {
                if( ( !strcmp(argv[ options ], "0") ) || ( !strcmp(argv[ options ], "1") ) )
                {
                    setLooping = atoi(argv[ options ]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }
    }
    if( invalidOptions == true || (numberOfVideosToDecode == 0) )
    {
        Usage();
        return;
    }
    g_TrackMemoryAllocation = trackMemoryAllocation;
    for(int i = 0; i < numberOfVideosToDecode; i ++)
        NN_LOG( "\n MovieDecoderMultiInstance:: Input File [%d] Name: %s\n", i + 1, inputFileName[i]);
    if( videoDecoderMode == movie::DecoderMode_NativeTexture )
        NN_LOG("\n Using [movie::DecoderMode_NativeTexture] mode for video decoder \n");
    else
        NN_LOG("\n Using [movie::DecoderMode_Cpu] mode for video decoder \n");
    char* mediaFile = inputFileName[ 0 ];
    movie::Status movieStatus = movie::Status_UnknownError;
    MovieDecoderMultiInstance *movieDecoderMultiInstance = new MovieDecoderMultiInstance(numberOfVideosToDecode);
    if( movieDecoderMultiInstance != nullptr )
    {
        movieDecoderMultiInstance->Initialize();
        if( true == movieDecoderMultiInstance->MountFileSystem(mediaFile) )
        {
            movieDecoderMultiInstance->CreateHidHandler();
            movieStatus = movieDecoderMultiInstance->CreateMovieDecoderPlayer();
            if( movieStatus == movie::Status_Success )
            {
                if( setLooping == true )
                    movieDecoderMultiInstance->EnableLooping();
                for( int i = 0; i < numberOfVideosToDecode; i++ )
                    movieDecoderMultiInstance->AddMovie(inputFileName[ i ], videoDecoderMode);
                movieStatus = movieDecoderMultiInstance->PlayMovies();
                if( movieStatus == movie::Status_Success )
                    movieDecoderMultiInstance->HandleEvents();
            }
            if( movieStatus != movie::Status_Success )
                NN_LOG("\n MovieDecoderMultiInstance FAILED to play movie! Error = %s \n", movie::StatusToString(movieStatus));
            movieDecoderMultiInstance->DestroyMovieDecoderPlayer();
            movieDecoderMultiInstance->DestroyHidHandler();
            movieDecoderMultiInstance->UnMountFileSystem();
        }
        movieDecoderMultiInstance->Finalize();
    }
    nv::FinalizeGraphics();
    if( movieDecoderMultiInstance != nullptr )
        delete movieDecoderMultiInstance;
    NN_LOG("\n MovieDecoderMultiInstance Sample EXITED \n");
    if( g_TrackMemoryAllocation == true )
    {
        NN_LOG("\n [Movie] memory allocator statistics \n");
        MovieMemoryTracker().OutputUsage();
        NN_LOG("\n [Core] memory allocator statistics \n");
        CoreMemoryTracker().OutputUsage();
        NN_LOG("\n [Malloc] memory allocator statistics \n");
        MallocMemoryTracker().OutputUsage();
        NN_LOG("\n [New] memory allocator statistics \n");
        NewMemoryTracker().OutputUsage();
    }
}//NOLINT(impl/function_size)

MovieDecoderMultiInstance::MovieDecoderMultiInstance(int numberOfVideosToDecode)
    : m_NumberOfVideosToDecode(numberOfVideosToDecode)
{
    m_MovieDecoderPlayer = nullptr;
    m_HidHandler = nullptr;
    m_EventsInitialized = false;
    m_SdcardMounted = false;
    m_HostFsMounted = false;
    m_State = MovieDecoderPlayer::State::State_NotInitialized;
    m_NativeWindow = nullptr;
}

void MovieDecoderMultiInstance::Initialize()
{
    // Initialize multiWait for Player
    nn::os::InitializeMultiWait(&m_PlayerDecoderMultiWait);
    RegisterEvent(&m_PlayerStateChangedEvent, MultiInstanceEventType_StateChanged, &m_PlayerStateChangedEventHolder);
    RegisterEvent(&m_PlayerErrorEvent, MultiInstanceEventType_Error, &m_PlayerErrorEventHolder);
    RegisterEvent(&m_HidEventButtonPressA, MultiInstanceEventType_ButtonPressA,&m_HidEventButtonPressAHolder);
    RegisterEvent(&m_HidEventButtonPressB, MultiInstanceEventType_ButtonPressB, &m_HidEventButtonPressBHolder);
    RegisterEvent(&m_HidEventButtonPressX, MultiInstanceEventType_ButtonPressX, &m_HidEventButtonPressXHolder);
    m_EventsInitialized = true;
}

void MovieDecoderMultiInstance::Finalize()
{
    if( m_EventsInitialized == true )
    {
        nn::os::UnlinkMultiWaitHolder(&m_PlayerStateChangedEventHolder);
        nn::os::UnlinkMultiWaitHolder(&m_PlayerErrorEventHolder);
        nn::os::UnlinkMultiWaitHolder(&m_HidEventButtonPressAHolder);
        nn::os::UnlinkMultiWaitHolder(&m_HidEventButtonPressBHolder);
        nn::os::UnlinkMultiWaitHolder(&m_HidEventButtonPressXHolder);
        DeleteEventUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_PlayerStateChangedEventHolder));
        DeleteEventUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_PlayerErrorEventHolder));
        DeleteEventUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_HidEventButtonPressAHolder));
        DeleteEventUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_HidEventButtonPressBHolder));
        DeleteEventUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_HidEventButtonPressXHolder));
        nn::os::FinalizeMultiWaitHolder(&m_PlayerStateChangedEventHolder);
        nn::os::FinalizeMultiWaitHolder(&m_PlayerErrorEventHolder);
        nn::os::FinalizeMultiWaitHolder(&m_HidEventButtonPressAHolder);
        nn::os::FinalizeMultiWaitHolder(&m_HidEventButtonPressBHolder);
        nn::os::FinalizeMultiWaitHolder(&m_HidEventButtonPressXHolder);
        nn::os::FinalizeEvent(&m_PlayerStateChangedEvent);
        nn::os::FinalizeEvent(&m_PlayerErrorEvent);
        nn::os::FinalizeEvent(&m_HidEventButtonPressA);
        nn::os::FinalizeEvent(&m_HidEventButtonPressB);
        nn::os::FinalizeEvent(&m_HidEventButtonPressX);
        nn::os::FinalizeMultiWait(&m_PlayerDecoderMultiWait);
        m_EventsInitialized = false;
    }
    m_NativeWindow = nullptr;
}

bool MovieDecoderMultiInstance::CreateHidHandler()
{
    bool retStatus = false;
    m_HidHandler = new HidHandler();
    if( m_HidHandler != nullptr )
    {
        if( true == m_HidHandler->Initialize() )
        {
            m_HidHandler->RegisterEvent(&m_HidEventButtonPressA, HidHandler::HidHandlerKeyType_ButtonA);
            m_HidHandler->RegisterEvent(&m_HidEventButtonPressB, HidHandler::HidHandlerKeyType_ButtonB);
            m_HidHandler->RegisterEvent(&m_HidEventButtonPressX, HidHandler::HidHandlerKeyType_ButtonX);
            retStatus = true;
        }
    }
    return retStatus;
}
void MovieDecoderMultiInstance::DestroyHidHandler()
{
    if( m_HidHandler != nullptr )
    {
        m_HidHandler->Finalize();
        delete m_HidHandler;
        m_HidHandler = nullptr;
    }
}

void MovieDecoderMultiInstance::RegisterEvent(nn::os::EventType *event, MultiInstanceEventType eventType, nn::os::MultiWaitHolderType *holder)
{
    nn::os::InitializeEvent(event, false, nn::os::EventClearMode_ManualClear);
    EventUserData *userData = new EventUserData(event, eventType);
    nn::os::InitializeMultiWaitHolder(holder, event);
    nn::os::SetMultiWaitHolderUserData(holder, ( uintptr_t ) userData);
    nn::os::LinkMultiWaitHolder(&m_PlayerDecoderMultiWait, holder);
}

void MovieDecoderMultiInstance::DeleteEventUserData(EventUserData* userData)
{
    if( userData != nullptr )
        delete userData;
}

bool MovieDecoderMultiInstance::DoesMediaFileExists(char* mediaFileName)
{
    nn::fs::FileHandle fileHandle;
    fileHandle.handle = NULL;
    nn::Result result = nn::fs::OpenFile(&fileHandle, mediaFileName, nn::fs::OpenMode_Read);
    if( result.IsFailure() )
        return false;
    else
        nn::fs::CloseFile(fileHandle);
    return true;
}

movie::Status MovieDecoderMultiInstance::CreateMovieDecoderPlayer()
{
    movie::Status movieStatus = movie::Status_OutOfMemory;
    m_MovieDecoderPlayer = new MovieDecoderPlayer { };
    if( m_MovieDecoderPlayer != nullptr )
    {
        movieStatus = m_MovieDecoderPlayer->Initialize();
        if( movieStatus != movie::Status_Success )
            return movieStatus;
        m_MovieDecoderPlayer->SetViNativeWindowHandle(m_NativeWindow);
        movieStatus = m_MovieDecoderPlayer->RegisterEvent(MovieDecoderPlayer::EventType_StateChangedEvent, &m_PlayerStateChangedEvent);
        if( movieStatus != movie::Status_Success )
            return movieStatus;
        movieStatus = m_MovieDecoderPlayer->RegisterEvent(MovieDecoderPlayer::EventType_ErrorEvent, &m_PlayerErrorEvent);
        if( movieStatus != movie::Status_Success )
            return movieStatus;
    }
    return movieStatus;
}

void MovieDecoderMultiInstance::DestroyMovieDecoderPlayer()
{
    if( m_MovieDecoderPlayer != NULL )
    {
        delete m_MovieDecoderPlayer;
        m_MovieDecoderPlayer = nullptr;
    }
}

movie::Status MovieDecoderMultiInstance::AddMovie(std::string path, movie::DecoderMode videoDecoderMode)
{
    m_MovieInfoList.push_back(MovieInfo(path, videoDecoderMode));
    return movie::Status_Success;
}

movie::Status MovieDecoderMultiInstance::PlayMovies()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    for( auto&& element : m_MovieInfoList )
    {
        if( false == DoesMediaFileExists((char*)element.moviePath.c_str()) )
            return movie::Status_TrackNotFound;
    }

    if( m_MovieDecoderPlayer != NULL )
    {
        for( auto&& element : m_MovieInfoList )
        {
            int videoTrackNumber = -1;
            movieStatus = m_MovieDecoderPlayer->SetDataSource(&element.extractor, element.moviePath, &videoTrackNumber, element.decoderMode);
            if( movieStatus != movie::Status_Success )
                return movieStatus;
            movieStatus = m_MovieDecoderPlayer->SelectVideoTrack(element.extractor, &videoTrackNumber);
            if( movieStatus != movie::Status_Success )
                return movieStatus;
        }
        movieStatus = m_MovieDecoderPlayer->Prepare();
        if( movieStatus != movie::Status_Success )
            return movieStatus;
        movieStatus = m_MovieDecoderPlayer->Start();
        if( movieStatus != movie::Status_Success )
            return movieStatus;
        NN_LOG("\n Movie playback STARTED \n");
    }
    return movieStatus;
}

bool MovieDecoderMultiInstance::HandleEvents()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    bool exitPlayer = false;
    // Wait for Player events, HID events
    for( ;; )
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&m_PlayerDecoderMultiWait);
        uintptr_t userData = GetMultiWaitHolderUserData(holder);
        EventUserData *eventUserData = ( EventUserData* ) userData;
        if( eventUserData == nullptr )
            break;
        MultiInstanceEventType eventype = eventUserData->m_EventType;
        nn::os::EventType *event = eventUserData->m_Event;
        nn::os::ClearEvent(event);
        switch( eventype )
        {
        case MultiInstanceEventType_StateChanged:
            m_MovieDecoderPlayer->GetState(&m_State);
            if( m_State == MovieDecoderPlayer::State::State_PlaybackCompleted )
            {
                movieStatus = m_MovieDecoderPlayer->Stop();
                NN_LOG("\n Movie playback STOPPED \n");
                exitPlayer = true;
            }
            break;

        case MultiInstanceEventType_Error:
            NN_LOG("\n MoviePlayer: Async Error = %s \n", movie::StatusToString(m_MovieDecoderPlayer->GetLastError()));
            movieStatus = m_MovieDecoderPlayer->Stop();
            NN_LOG("\n Movie playback STOPPED \n");
            exitPlayer = true;
            break;

        case MultiInstanceEventType_ButtonPressA:
            m_MovieDecoderPlayer->GetState(&m_State);
            switch( m_State )
            {
            case MovieDecoderPlayer::State::State_Paused:
                movieStatus = m_MovieDecoderPlayer->Resume();
                NN_LOG("\n Movie playback RESUMED \n");
                break;

            case MovieDecoderPlayer::State::State_Started:
                movieStatus = m_MovieDecoderPlayer->Pause();
                NN_LOG("\n Movie playback PAUSED \n");
                break;
            default:
                break;
            }
            break;

        case MultiInstanceEventType_ButtonPressB:
            movieStatus = m_MovieDecoderPlayer->Stop();
            NN_LOG("\n Movie playback STOPPED \n");
            break;

        case MultiInstanceEventType_ButtonPressX:
            movieStatus = m_MovieDecoderPlayer->Stop();
            exitPlayer = true;
            NN_LOG("\n Movie playback STOPPED \n");
            break;
        default:
            break;
        }
        if( exitPlayer == true )
            break;
    }
    return true;
}

bool MovieDecoderMultiInstance::MountFileSystem(char* mediaFile)
{
    bool retValue = false;
    std::string path = mediaFile;
    std::string delimiter = ":";
    std::string token = path.substr(0, path.find(delimiter));
    if( token == "sdcard" )
    {
        nn::Result resultSdcardMount = nn::fs::MountSdCardForDebug("sdcard");
        if( resultSdcardMount.IsFailure() )
        {
            NN_LOG("\n nn::fs::SD card mount failure. Module:%d, Description:%d\n",
                resultSdcardMount.GetModule(),
                resultSdcardMount.GetDescription());
            return retValue;
        }
        m_SdcardMounted = true;
        retValue = true;
    }
    else if( token == "http" || token == "https" || token == "file" )
    {
        NN_LOG("\n MoviePlayer: Network playback not supported !\n");
    }
    else
    {
        nn::Result resultHostMount = nn::fs::MountHostRoot();
        if( resultHostMount.IsFailure() )
        {
            NN_LOG("\n nn::fs::Host root mount failure. Module:%d, Description:%d\n",
                resultHostMount.GetModule(),
                resultHostMount.GetDescription());
            return retValue;
        }
        m_HostFsMounted = true;
        retValue = true;
    }
    return retValue;
}

void MovieDecoderMultiInstance::UnMountFileSystem()
{
    if( m_SdcardMounted == true )
        nn::fs::Unmount("sdcard");
    if( m_HostFsMounted == true )
        nn::fs::UnmountHostRoot();
}

void MovieDecoderMultiInstance::EnableLooping()
{
    if( m_MovieDecoderPlayer != nullptr )
        m_MovieDecoderPlayer->SetLooping(true);
}
