﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{MifareSimple.cpp,PageSampleMifareSimple}

    @brief
    NFC ライブラリによる MIFARE タグアクセス機能のサンプルプログラム
 */

/**
    @page PageSampleMifareSimple NFC ライブラリによる MIFARE タグアクセス機能のサンプル
    @tableofcontents

    @brief
    NFC ライブラリによる MIFARE タグアクセス機能のサンプルプログラムの解説です。

    @section PageSampleMifareSimple_SectionBrief 概要
    NFC ライブラリによる MIFARE タグアクセス機能は、NFC（Near Field Communication）を介して、
    MIFARE タグ（MIFARE 用にフォーマットされた NFC タグ）の読み書きを行う機能です。
    NFC ライブラリによる MIFARE タグアクセス機能を用いることで、アプリケーションと MIFARE タグを連携させることが可能です。

    このサンプルプログラムは、以下の MIFARE 操作に対応しています。
    @n
    @li 読み書き

    サンプルプログラムではブロックアドレス 1 の先頭 1 バイトとブロックアドレス 6 の先頭 1 バイトに カウンタを書き込みます。(先頭以外の 15 バイトは 0x00)
    書き込みを行うたびに、ブロックアドレス 1 のカウンタを 1 つずつ、ブロックアドレス 6 のカウンタを 2 つずつインクリメントしていきます。

    @section PageSampleMifareSimple_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/MifareSimple
    Samples/Sources/Applications/MifareSimple @endlink 以下にあります。

    @section PageSampleMifareSimple_SectionNecessaryEnvironment 必要な環境
    MIFARE タグを用意してください。

    @section PageSampleMifareSimple_SectionHowToOperate 操作方法
    サンプルプログラムを実行すると、NFC ライブラリの状態が表示されます。
    ゲームパッドの操作により、NFC ライブラリの状態を変化させることができます。

    @section PageSampleMifareSimple_SectionPrecaution 注意事項
    サンプルでは 全 6 バイトが全て 0xFF である Key A でブロックアドレス 1, 6 の読み書きが可能な MIFARE タグを想定しています。
    想定している MIFARE タグと異なる場合は、プログラム内で指定している鍵やブロックアドレスを適切に書き変えてください。

    @section PageSampleMifareSimple_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleMifareSimple_SectionDetail 解説
    MifareSimple では MIFARE タグの状態に関わらずカウンタのインクリメント処理を実行します。

 */

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/init.h>
#include "Graphics.h"
#include "HidController.h"

#include "Config.h"
#include "UpdateState.h"
#include "WriteState.h"

namespace {

    // Graphics.cpp で malloc しているヒープの内訳
    //  VisiblePoolMemory       16 MB
    //  InvisiblePoolMemory     20 MB
    //  for CommandBuffer       32 MB
    //  nv::InitializeGraphics   8 MB
    //  for DebugFont            2 MB
    const size_t MemoryHeapSize         = 128 * 1024 * 1024;
    const size_t MallocHeapSize         =  96 * 1024 * 1024;

    // アプリケーションで扱うデータと状態です。
    nns::nfc::mifare::ApplicationData g_Data;

} // namespace

// アプリケーションのメモリ管理機構を初期化
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

// アプリケーションのエントリポイント
extern "C" void nnMain()
{
    InitializeGraphicSystem();
    InitializeHidController();

    while( NN_STATIC_CONDITION(true) )
    {
        UpdateHidController();

        nns::nfc::mifare::UpdateState(g_Data);

        BeginText();
        nns::nfc::mifare::WriteState(g_Data);
        EndText();

        if( g_Data.state == nns::nfc::mifare::State_Exit )
        {
            break;
        }
    }

    FinalizeHidController();
    FinalizeGraphicSystem();
}
