﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @examplesource{MediaPlayerSample.cpp,PageSampleMediaPlayerSample}
 *
 * @brief
 *  movie player API による動画再生サンプル
 */
 /**
 * @page PageSampleMediaPlayerSample MediaPlayerSample
 * @tableofcontents
 *
 * @brief
 * movie player API による動画再生サンプルプログラムです。
 *
 * @section PageSampleMediaPlayerSample_SectionBrief 概要
 * NX 本体上で mp4 ファイルを入力し、再生を行うサンプルです。
 *
 * @section PageSampleMediaPlayerSample_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/MediaPlayerSample
 * Samples/Sources/Applications/MediaPlayerSample @endlink 以下にあります。
 *
 * @section PageSampleMediaPlayerSample_SectionNecessaryEnvironment 必要な環境
 * 追加で必要となるものはありません。
 *
 * @section PageSampleMediaPlayerSample_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、自動的に指定した動画の再生が始まります。
 * 再生終了後は、自動的にプログラムは終了します。
 *
 *  - A ボタン: 一時停止／再生
 *  - B ボタン: 停止
 *  - X ボタン: 再生倍率のリセット
 *  - Y ボタン: 現在の再生倍率を出力
 *  - 右ボタン: 再生位置の移動（早送り）
 *  - 左ボタン: 再生位置の移動（巻き戻し）
 *  - 上ボタン: 音のボリュームを 1.0 に設定（ミュートの解除）
 *  - 下ボタン: 音のボリュームを 0.0 に設定（ミュート）
 *  - R ボタン: 10秒先送り
 *  - L ボタン: 15秒巻き戻し
 *  - ZRボタン: あらかじめ定義された再生倍率の適用
 *  - Lスティック: 再生位置の移動（高速）
 *  - Rスティック: 再生倍率の連続的な調節（左入力により 0.05 倍まで減少、右入力により 4倍まで増加）
 *
 * @section PageSampleMediaPlayerSample_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleMediaPlayerSample_SectionHowToExecute 実行手順
 *  @<mediaplayer sample executable@> @<media path@> @<optional parameters@>
 *
 * @<optional parameters@>
 *
 * - -a 1: オーディオトラックを再生する。
 * - -a 0: オーディオトラックを再生しない。
 *
 * - -v 1: ビデオトラックを再生する。
 * - -v 0: ビデオトラックを再生しない。
 *
 * - -r @<rate@>: 再生レート（例: -r 2.0）
 *
 * - -l 0: ループ再生をしない。
 * - -l 1: ループ再生をする。
 *
 * - -h 0: ヒープトラッキングを無効にする。
 * - -h 1: ヒープトラッキングを有効にする。
 *
 * - -dm 1: movie::VideoDecoderMode_Cpu モードの適用
 * - -dm 2: movie::VideoDecoderMode_NativeTexture モードの適用
 *
 * - -ur 0: HLS 解像度を 720p に制限（デフォルト）
 * - -ur 1: HLS 解像度を制限しない
 *
 * - -c コンフィグファイルのパスを指定する。
 *
 * 例（ホストPCから動画データを取得）
 * MediaPlayerSample.nsp c:@\H264_AAC_1280x720_30sec_01.mp4
 *
 * @section PageSampleMediaPlayerSample_SectionDetail 解説
 * 指定した動画ファイルを NX 本体上で再生します。
 */
#include <getopt.h>
#include "MediaPlayerSample.h"
#include "MediaPlayerObserver.h"
#include "MediaPlayerUtilities.h"
#include "HidHandler.h"
#include "HeapTracker.h"
#include <movie/Utils.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_ScopeExit.h>

#include <nv/nv_MemoryManagement.h>
#include <sys/time.h>

#include <nn/nifm.h>
#include <nn/socket.h>
#include <curl/curl.h>
#include <string>
#include <cinttypes>
#include <limits>
#include <cmath>

#ifdef MOUNT_SDCARD
    #include <nn/fs/fs_SdCardForDebug.h>
#endif

#ifdef ENABLE_PROFILER
    #include <nn/profiler.h>
#endif

#include <nvn/nvn_FuncPtrInline.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>

// Instantiate Globals
uint8_t g_GraphicsHeap[g_GraphicsSystemMemorySize] __attribute__(( aligned(4096) ));
uint8_t g_FsHeapBuffer[FsHeapSize];
nn::lmem::HeapHandle g_FsHeap;
bool USE_PROXY = false;
std::string PROXY_SETTING_HOST = "proxy-host.com";
int PROXY_SETTING_PORT = 8080;
std::string PROXY_SETTING_USERPWD = "UserName:Password";
unsigned long PROXY_SETTING_AUTH = CURLAUTH_BASIC;
bool USE_HEAP_TRACKING = false;
nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

void FsInitHeap()
{
    g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, FsHeapSize,
        nn::lmem::CreationOption_DebugFill);
}

void* FsAllocate(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
}

void FsDeallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    return nn::lmem::FreeToExpHeap(g_FsHeap, p);
}

/* MediaPlayer Sample Usage */
static void Usage()
{
    NN_SDK_LOG( "\n ERROR:: Invalid options See Readme.MediaPlayerSample.txt \n");
}

/* Process startup setup */
extern "C" void nninitStartup()
{
    const size_t heapSize = 256 << 20;
    nn::Result result;
    result = nn::os::SetMemoryHeapSize(heapSize); // This specifies the amount of memory made available to
                                                  // nn::os::AllocateMemoryBlock(), which is only used
                                                  // in this sample by the HeapTracker.
    FsInitHeap();
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
}

extern "C" int* __errno()
{
    static int i = 0;
    return &i;
}

/* Process main entry */
extern "C" void nnMain()
{
    { // We use an embedded scope to make sure local destructors have been called
      // before we print our final heap tracking statistics at the end of nnMain()

#ifdef ENABLE_PROFILER
    std::vector<char> profiler_buffer__(nn::profiler::MinimumBufferSize);
    nn::profiler::Initialize(profiler_buffer__.data(), profiler_buffer__.size());
    NN_UTIL_SCOPE_EXIT { nn::profiler::Finalize(); };
#endif

    bool isNetworkFile = false;
    nn::hid::InitializeDebugPad();

    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();
    // Set allocator callback functions

    movie::SetAllocator(mmAllocate, mmDeallocate, mmReallocate, &MMHeap());
    // Donate memory for graphics driver to work in
    nv::InitializeGraphics(g_GraphicsHeap, g_GraphicsSystemMemorySize);
    nv::SetGraphicsAllocator(mmAllocate, mmDeallocate, mmReallocate, &CoreHeap());
    nv::SetGraphicsDevtoolsAllocator(mmAllocate, mmDeallocate, mmReallocate, &CoreHeap());

    if( argc <= 1 )
    {
       Usage();
       return;
    }

    bool playAudio = true;
    bool playVideo = true;
    float playRate = 1.0; //Normal;
    bool setLooping = false;
    bool invalidOptions = false;
    bool isVideoTrack = false;
    bool unrestrictedResolution = false;
    const char *configFile = NULL;
    const char *urlFile = NULL;
    movie::PlayerResMax maxPlayerRes = movie::PlayerResMax_720;
    movie::VideoDecodeMode videoDecodeMode = movie::VideoDecodeMode_NativeTexture;
    movie::OutputFormat videoOutputFormat = movie::OutputFormat_VideoColorAbgr;
    std::string urlString;
    float volume = 1.0; //Normal
    //    while ((option = getopt(argc, argv, "avp:")) != -1)
    for(int options = 0; options < argc; options ++)
    {
        if( !strcmp(argv[options], "-a" ) )
        {
            options ++;
            invalidOptions = true;
            if( options < argc )
            {
                if( ( !strcmp(argv[options], "0") ) ||  ( !strcmp(argv[options], "1") ) )
                {
                    playAudio = atoi(argv[options]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }

        if( !strcmp(argv[options], "-v") )
        {
            options ++;
            invalidOptions = true;
            if( options < argc )
            {
                if( ( !strcmp(argv[options], "0") ) ||  ( !strcmp(argv[options], "1") ) )
                {
                    playVideo = atoi(argv[options]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }

        if( !strcmp(argv[options], "-ur") )
        {
            options ++;
            invalidOptions = true;
            if( options < argc )
            {
                if( ( !strcmp(argv[options], "0") ) ||  ( !strcmp(argv[options], "1") ) )
                {
                    invalidOptions = false;
                    unrestrictedResolution = atoi(argv[options]);
                    continue;
                }
            }
            break;
        }

        if( !strcmp(argv[options], "-r"))
        {
            options ++;
            invalidOptions = true;
            if( options < argc )
            {
                playRate = atof(argv[options]);
                if (playRate < 0.0001f) {   // This limit is just to rule out negative and close-to-zero values, as well as values that
                    invalidOptions = true;  // atof() could not recognize as floats (in which case it returns 0.0)
                } else {
                    invalidOptions = false;
                }
                continue;
            }
            break;
        }

        if( !strcmp(argv[ options ], "-vol") )
        {
            options++;
            invalidOptions = true;
            if( options < argc )
            {
                volume = atof(argv[ options ]);
                if( ( volume < 0.0 ) || ( volume > 4.0 ) )
                    invalidOptions = true;
                else
                    invalidOptions = false;
                continue;
            }
            break;
        }

        if( !strcmp(argv[options], "-l"))
        {
            options ++;
            invalidOptions = true;
            if( options < argc )
            {
                if( ( !strcmp(argv[options], "0") ) ||  ( !strcmp(argv[options], "1") ) )
                {
                    setLooping = atoi(argv[options]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }

        if (!strcmp(argv[options], "-h"))
        {
            options++;
            invalidOptions = true;
            if (options < argc)
            {
                if ((!strcmp(argv[options], "0")) || (!strcmp(argv[options], "1")))
                {
                    USE_HEAP_TRACKING = atoi(argv[options]);
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }

        if( !strcmp(argv[options], "-c"))
        {
            options ++;
            invalidOptions = true;
            if( options < argc )
            {
                configFile = argv[options];
                invalidOptions = false;
                continue;
            }
            break;
        }

        if (!strcmp(argv[options], "-u"))
        {
            options++;
            invalidOptions = true;
            if (options < argc) {
                urlFile = argv[options];
                invalidOptions = false;
                continue;
            }
            break;
        }

        if (!strcmp(argv[options], "-dm"))
        {
            options++;
            invalidOptions = true;
            if (options < argc)
            {
                if ((!strcmp(argv[options], "1")) || (!strcmp(argv[options], "2")) )
                {
                    int videoDecodeModeSelection = atoi(argv[options]);
                    switch( videoDecodeModeSelection )
                    {
                    case 1:
                        videoDecodeMode = movie::VideoDecodeMode_Cpu;
                        videoOutputFormat = movie::OutputFormat_VideoColorNv12;
                        break;
                    case 2:
                        videoDecodeMode = movie::VideoDecodeMode_NativeTexture;
                        videoOutputFormat = movie::OutputFormat_VideoColorAbgr;
                        break;
                    default:
                        break;
                    }
                    invalidOptions = false;
                    continue;
                }
            }
            break;
        }
    }

    if( invalidOptions == true )
    {
        Usage();
        return;
    }

    switch( videoDecodeMode )
    {
    case movie::VideoDecodeMode_Cpu:
        NN_SDK_LOG("\n MediaPlayerSample:: Using [movie::VideoDecodeMode_Cpu] video decode mode \n");
        break;
    case movie::VideoDecodeMode_NativeTexture:
        NN_SDK_LOG("\n MediaPlayerSample:: Using [movie::VideoDecodeMode_NativeTexture] video decode mode \n");
        break;
    default:
        NN_SDK_LOG("\n MediaPlayerSample:: Using [movie::VideoDecodeMode_NativeTexture] video decode mode \n");
        break;
    }

    if( unrestrictedResolution )
    {
        maxPlayerRes = movie::PlayerResMax_None;
        NN_SDK_LOG("\n MediaPlayerSample:: Unrestricted maximum resolution\n");
    }

    timeval playerTimeVal;
    int64_t startTime = 0ll;
    int64_t endTime = 0ll;

    const char* inputFileName = inputFileName = argv[1];
    std::string path = inputFileName;
    std::string delimiter = ":";
    std::string token = path.substr(0, path.find(delimiter));
    bool sdcardMounted = false;
    if( token == "sdcard" )
    {
#ifdef MOUNT_SDCARD
        nn::Result resultSdcardMount = nn::fs::MountSdCardForDebug("sdcard");
        if( resultSdcardMount.IsFailure() )
        {
            NN_SDK_LOG( "\n nn::fs::SD card mount failure. Module:%d, Description:%d\n" ,
                    resultSdcardMount.GetModule(),
                    resultSdcardMount.GetDescription());
            return;
        }
        sdcardMounted = true;
#else
        NN_SDK_LOG( "\n SD card Not supported \n");
        return;
#endif
    }
    else if ( token == "http" || token == "https" || urlFile )
    {
        isNetworkFile = true;
        nn::nifm::Initialize();
        nn::nifm::SubmitNetworkRequest();
        while (nn::nifm::IsNetworkRequestOnHold())
        {
            NN_SDK_LOG("Network request on hold\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        if (!nn::nifm::IsNetworkAvailable())
        {
            NN_SDK_LOG("Network initialization failed\n");
            nn::nifm::CancelNetworkRequest();
            return;
        }
        else
        {
            nn::Result res = nn::socket::Initialize(g_SocketConfigWithMemory);
            if (res.IsFailure())
            {
                nn::nifm::CancelNetworkRequest();
                NN_SDK_LOG("nn::socket::Initialize failed!\n");
                return;
            }
        }

        CURLcode res = CURLE_OK;

        /* initialize using system malloc for libcurl for testing */
        res = curl_global_init(CURL_GLOBAL_ALL);
        if (res != CURLE_OK)
        {
            NN_LOG("\nError! curl_global_init failed. Err: %d\n\n", res);
            return;
        }
    }
    nn::Result resultHostMount = nn::fs::MountHostRoot();
    if( resultHostMount.IsFailure() )
    {
        NN_SDK_LOG( "\n nn::fs::Host root mount failure. Module:%d, Description:%d\n",
            resultHostMount.GetModule(),
            resultHostMount.GetDescription() );
        return;
    }
    if(configFile && !ProcessConfigFile(configFile))
        return;
    /* Check whether media exists if it is a local file */
    if (!isNetworkFile && token != "file")
    {
        nn::fs::FileHandle fileHandle;
        fileHandle.handle = NULL;
        nn::Result result = nn::fs::OpenFile(&fileHandle, inputFileName, nn::fs::OpenMode_Read);
        if (result.IsFailure())
        {
            NN_SDK_LOG("\n Failed to open %s \n \n Terminating PlayerSample MediaPlayerSample\n", inputFileName);
            return;
        }
        else
        {
            nn::fs::CloseFile(fileHandle);
        }
    }

    if (urlFile)
    {
        if (ProcessUrlFile(urlFile, &urlString))
            {
            inputFileName = urlString.c_str();
        }
        else
        {
            return;
        }
    }

    NN_SDK_LOG( "\n MediaPlayerSample:: Input File Name : %s\n", inputFileName );

    movie::PlayerConfig config;
    config.returnAudioDataToClient = false;
    config.returnVideoDataToClient = true;
    // Usage of "useNativeBuffers" is being deprecated, use movie::videoDecodeMode
    // config.useNativeBuffers = useNativeBuffers;
    config.videoDecodeMode = videoDecodeMode;
    config.returnSyncedData = true;
    config.useClientMemoryForOutput = true;
    config.autoReleaseOutputBuffers = true;
    config.audioFormat = movie::OutputFormat_AudioPcm16;
    config.videoFormat = videoOutputFormat;
    config.maxResolution = maxPlayerRes;
    movie::MediaData bufferAlignment;
    int nv12YAlignment = 32;
    int nv12UvAlignment = 32;
    int pcmAlignment = 32;
    bufferAlignment.SetInt32("nv12-y-alignment", nv12YAlignment);
    bufferAlignment.SetInt32("nv12-uv-alignment", nv12UvAlignment);
    bufferAlignment.SetInt32("pcm-alignment", pcmAlignment);

    /*Set browser config, below are some sample values set to test this feature*/
    movie::BrowserConfig browserConfig;
    browserConfig.coreMask = 0x00;
    browserConfig.sfThreadPriority = 16;
    browserConfig.audioRendererThreadPriority = 16;
    browserConfig.videoRendererThreadPriority = 16;


    MediaPlayerObserver* mediaPlayerObserver = new MediaPlayerObserver;
    if( mediaPlayerObserver == NULL )
    {
        NN_SDK_LOG( "\n Failed to Create MediaPlayerSample - Terminating PlayerSample \n" );
        return;
    }
    mediaPlayerObserver->Initialize(&config);

    NN_SDK_LOG( "\n MediaPlayerSample:: Create \n" );
    movie::BrowserPlayer* player = NULL;
    movie::BrowserPlayer::Create(&player, &config, &browserConfig);
    if( player == NULL )
    {
        NN_SDK_LOG( "\n MediaPlayerSample:: Failed to Create Player  - Terminating PlayerSample \n" );
        mediaPlayerObserver->Finalize(&config);
        if( isNetworkFile == true )
        {
            nn::socket::Finalize();
            nn::nifm::CancelNetworkRequest();
        }
        else if( sdcardMounted == true )
        {
    #ifdef MOUNT_SDCARD
            nn::fs::Unmount("sdcard");
    #endif
        }
        else
        {
            nn::fs::UnmountHostRoot();
        }
        delete mediaPlayerObserver;
        return;
    }
    mediaPlayerObserver->SetMoviePlayer(player);

    HidHandler hidHandler(player, mediaPlayerObserver, setLooping);
    hidHandler.Initialize();

    movie::Status movieStatus = movie::Status_Success;
    bool errorCleanUp = false;

    if( player != NULL )
    {
        NN_SDK_LOG( "\n MediaPlayerSample:: Set SetObserver \n" );
        movieStatus = player->SetObserver(mediaPlayerObserver);
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG( "\n MediaPlayerSample:: Failed to Set SetObserver: Error : 0x%x  - Terminating PlayerSample \n", movieStatus );
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }

        NN_SDK_LOG("\n MediaPlayerSample :: Set Output Buffer Alignment \n");
        movieStatus = player->SetOutputBufferAlignment(&bufferAlignment);
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG("\n MediaPlayerSample:: Failed to Set Output Buffer Alignment Error : 0x%x \n", movieStatus);
        }

        NN_SDK_LOG( "\n MediaPlayerSample :: Set data Source \n" );
        movieStatus = player->SetDataSource(inputFileName);
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG( "\n MediaPlayerSample:: Failed to Set data Source: Error : 0x%x  - Terminating PlayerSample \n", movieStatus );
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }

        NN_SDK_LOG( "\n MediaPlayerSample:: Prepare \n" );
        movieStatus = player->Prepare();
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG( "\n MediaPlayerSample:: Failed to Prepare: Error : 0x%x  - Terminating PlayerSample \n", movieStatus );
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }

        NN_SDK_LOG( "\n MediaPlayerSample:: Test API: GetPlaybackDuration() + \n");
        int64_t trackDuration = 0;
        movieStatus = player->GetPlaybackDuration(&trackDuration);
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG( "\n MediaPlayerSample:: Failed to GetPlaybackDuration: Error : 0x%x \n", movieStatus );
        }

        trackDuration = trackDuration / 1000;
        NN_SDK_LOG( "   GetPlaybackDuration (In msec) : %" PRId64 "\n", trackDuration);
        NN_SDK_LOG( " MediaPlayerSample:: Test API: GetPlaybackDuration() - \n");

        NN_SDK_LOG( "\n MediaPlayerSample:: Test API: GetTrackCount() + \n");
        int32_t trackCount = 0;
        movieStatus = player->GetTrackCount(&trackCount);
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG( "\n MediaPlayerSample:: Failed to GetTrackCount: Error : 0x%x  - Terminating PlayerSample \n", movieStatus );
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }
        NN_SDK_LOG( "   GetTrackCount: %d \n" , trackCount);
        NN_SDK_LOG( " MediaPlayerSample:: Test API: GetTrackCount() - \n");

        NN_SDK_LOG( "\n MediaPlayerSample:: Test API: GetVideoDimensions() + \n");
        int32_t width = 0;
        int32_t height = 0;
        movieStatus = player->GetVideoDimensions(&width, &height);
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG( "\n MediaPlayerSample:: Failed to GetVideoDimensions: Error : 0x%x, Detail: %s\n", movieStatus );
        }
        NN_SDK_LOG( "   GetVideoDimensions: width: %d height: %d \n", width, height);
        NN_SDK_LOG( " MediaPlayerSample:: Test API: GetVideoDimensions() - \n");

        NN_SDK_LOG( "\n MediaPlayerSample:: Test API: GetTrackType(), GetTrackInfo() + \n");

        for( int32_t track = 0; track < trackCount; track ++)
        {
            movie::TrackType trackType = movie::TrackType_Unknown;
            movie::TrackInfo trackInfo;
            movieStatus = player->GetTrackType(track, &trackType);
            if( movieStatus != movie::Status_Success )
            {
                NN_SDK_LOG( "\n MediaPlayerSample:: Failed to GetTrackType: Error : 0x%x \n", movieStatus );
            }

            switch (trackType )
            {
            case movie::TrackType_Audio:
                NN_SDK_LOG( "   GetTrackType() Track Index : %d Track Type: TrackType_Audio \n" , track);
                movieStatus = player->GetTrackInfo(track, &trackInfo);
                if( movieStatus != movie::Status_Success )
                {
                    NN_SDK_LOG( "\n MediaPlayerSample:: Failed to GetTrackInfo: Error : 0x%x \n", movieStatus );
                }
                else
                {
                    NN_SDK_LOG( "\n GetTrackInfo() Track Index : %d Track Type: TrackType_Audio \n" , track);
                    NN_SDK_LOG( "                  mime        : %s \n", trackInfo.audioTrackInfo.mime);
                    NN_SDK_LOG( "                  sampleRate  : %d \n", trackInfo.audioTrackInfo.sampleRate);
                    NN_SDK_LOG( "                  channels    : %d \n", trackInfo.audioTrackInfo.channels);
                    NN_SDK_LOG( "                  bitRate     : %d \n", trackInfo.audioTrackInfo.bitRate);
                    NN_SDK_LOG( "                  codec       : 0x%x \n", trackInfo.audioTrackInfo.codecType);
                    NN_SDK_LOG( "                  container   : 0x%x \n", trackInfo.containerType);
                }
                break;

            case movie::TrackType_Video:
                if (playVideo)
                    isVideoTrack = true;
                NN_SDK_LOG( "   GetTrackType() Track Index : %d Track Type: TrackType_Video \n" , track);
                movieStatus = player->GetTrackInfo(track, &trackInfo);

                {
                    auto& fps = trackInfo.videoTrackInfo.frameRate;
                    if (fps == 60 || fps == 30 || fps == 20 || fps == 15) {
                        nvnWindowSetPresentInterval(&mediaPlayerObserver->GetNvnVideoRenderer()->GetDevice()->GetWindow(), 60 / fps);
                        NN_SDK_LOG("\n MediaPlayerSample:: presentation interval set for %d fps\n",fps);
                    }
                }

                if( movieStatus != movie::Status_Success )
                {
                    NN_SDK_LOG( "\n MediaPlayerSample:: Failed to GetTrackInfo: Error : 0x%x \n", movieStatus );
                }
                else
                {
                    NN_SDK_LOG( "\n GetTrackInfo() Track Index : %d Track Type: TrackType_Video \n" , track);
                    NN_SDK_LOG( "                  mime        : %s \n", trackInfo.videoTrackInfo.mime);
                    NN_SDK_LOG( "                  frameRate   : %d \n", trackInfo.videoTrackInfo.frameRate);
                    NN_SDK_LOG( "                  height      : %d \n", trackInfo.videoTrackInfo.height);
                    NN_SDK_LOG( "                  width       : %d \n", trackInfo.videoTrackInfo.width);
                    NN_SDK_LOG( "                  bitRate     : %d \n", trackInfo.videoTrackInfo.bitRate);
                    NN_SDK_LOG( "                  codec       : 0x%x \n", trackInfo.videoTrackInfo.codecType);
                    NN_SDK_LOG( "                  container   : 0x%x \n", trackInfo.containerType);
                }
                break;

            case movie::TrackType_Subtitle:
                NN_SDK_LOG( "   GetTrackType() Track Index : %d Track Type: TrackType_Subtitle \n" , track);
                movieStatus = player->GetTrackInfo(track, &trackInfo);
                if( movieStatus != movie::Status_Success )
                {
                    NN_SDK_LOG( "\n MediaPlayerSample:: Failed to GetTrackInfo: Error : 0x%x \n", movieStatus );
                }
                else
                {
                    NN_SDK_LOG( "\n GetTrackInfo() Track Index : %d Track Type: TrackType_Subtitle \n" , track);
                    NN_SDK_LOG( "                  mime        : %s \n", trackInfo.subtitleTrackInfo.mime);
                }
                break;

            case movie::TrackType_ClosedCaption:
                NN_SDK_LOG( "   GetTrackType() Track Index : %d Track Type: TrackType_ClosedCaption \n" , track);
                movieStatus = player->GetTrackInfo(track, &trackInfo);
                if( movieStatus != movie::Status_Success )
                {
                    NN_SDK_LOG( "\n MediaPlayerSample:: Failed to GetTrackInfo: Error : 0x%x \n", movieStatus );
                }
                else
                {
                    NN_SDK_LOG( "\n GetTrackInfo() Track Index : %d Track Type: TrackType_ClosedCaption \n" , track);
                    NN_SDK_LOG( "                  mime        : %s \n", trackInfo.closedCaptionTrackInfo.mime);
                }
                break;

            default:
                NN_SDK_LOG( "   GetTrackType() Track Index : %d Track Type: TrackType_Unknown \n" , track);
                break;
            }
        }
        NN_SDK_LOG( " MediaPlayerSample:: Test API: GetTrackType(), GetTrackInfo() - \n");

        /* Play only selected track */
        for( int32_t track = 0; track < trackCount; track ++)
        {
            movie::TrackType trackType = movie::TrackType_Unknown;
            movieStatus = player->GetTrackType(track, &trackType);
            if( movieStatus != movie::Status_Success )
            {
                NN_SDK_LOG( "\n MediaPlayerSample:: Failed to GetTrackType: Error : 0x%x \n", movieStatus );
            }
            switch (trackType )
            {
            case movie::TrackType_Audio:
                if( playAudio == false )
                {
                    movieStatus = player->DeSelectTrack(track);
                    if( movieStatus != movie::Status_Success )
                    {
                        NN_SDK_LOG( "\n MediaPlayerSample:: Failed to DeSelectTrack: Error : 0x%x \n", movieStatus );
                    }
                }
                mediaPlayerObserver->SetAudioTrackNumber(track);
                break;

            case movie::TrackType_Video:
                if( playVideo == false )
                {
                    movieStatus = player->DeSelectTrack(track);
                    if( movieStatus != movie::Status_Success )
                    {
                        NN_SDK_LOG( "\n MediaPlayerSample:: Failed to DeSelectTrack: Error : 0x%x \n", movieStatus );
                    }
                }
                mediaPlayerObserver->SetVideoTrackNumber(track);
                break;

            default:
                break;
            }
        }

        /* Set loop mode */
        if( setLooping == true )
        {
            movieStatus = player->SetLooping(setLooping);
            if( movieStatus != movie::Status_Success )
            {
                NN_SDK_LOG( "\n MediaPlayerSample:: Failed to SetLooping: Error : 0x%x \n", movieStatus );
            }
        }

        NN_SDK_LOG( "\n MediaPlayerSample:: setting playback rate to %f\n", playRate);
        player->SetPlaybackRate(playRate);

        movieStatus = player->SetVolume(volume);
        if( movieStatus != movie::Status_Success )
            NN_SDK_LOG("\n MediaPlayerSample::SetVolume Error = 0x%x \n", movieStatus);

        float vol = 1.0;
        movieStatus = player->GetVolume(&vol);
        if( movieStatus != movie::Status_Success )
            NN_SDK_LOG("\n MediaPlayerSample::GetVolume Error = 0x%x \n", movieStatus);
        NN_SDK_LOG("\n MediaPlayerSample:: Volume = %f \n", vol);

        NN_SDK_LOG( "\n MediaPlayerSample:: start \n" );
        movieStatus = player->Start();
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG( "\n MediaPlayerSample:: Failed to Start: Error : 0x%x  - Terminating PlayerSample \n", movieStatus );
            errorCleanUp = true;
            goto Exit_PlayerSample;
        }
        if( isVideoTrack )
        {
            nn::os::WaitEvent(&mediaPlayerObserver->m_StartPlaybackEvent);
            nn::os::ClearEvent(&mediaPlayerObserver->m_StartPlaybackEvent);
            player->GetVideoDimensions(&width, &height);
            if( ( config.returnVideoDataToClient == true ) && ( config.useClientMemoryForOutput == true ) )
            {
                size_t videoOutputBufferSize = width * height;
                size_t extraForAlignment = 1024 * 4;
                if( config.videoFormat == movie::OutputFormat_VideoColorAbgr )
                {
                    videoOutputBufferSize = videoOutputBufferSize * 4 + extraForAlignment;
                }
                else if( config.videoFormat == movie::OutputFormat_VideoColorNv12 )
                {
                    videoOutputBufferSize = videoOutputBufferSize + ( 2 * ( ( width / 2 ) * ( height / 2 ) ) ) + extraForAlignment;
                }

                mediaPlayerObserver->m_VideoOutputBufferSize = videoOutputBufferSize;
            }

        }

        gettimeofday(&playerTimeVal, NULL);
        startTime = (int64_t)playerTimeVal.tv_sec * 1000 + (int64_t)playerTimeVal.tv_usec / 1000;

        if( playAudio == false && playVideo == false )
        {
            mediaPlayerObserver->SetPlaybackComplete();
        }

        int64_t lastPlaybackPosition = 0;
        int64_t getPlaybackPosition = 0;

        while ( !mediaPlayerObserver->IsPlaybackComplete() )
        {

#ifdef ENABLE_PROFILER
            nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_Main);
#endif
            getPlaybackPosition = 0;
            movieStatus = player->GetCurrentPlaybackPosition(&getPlaybackPosition);
            if( movieStatus != movie::Status_Success )
            {
                NN_SDK_LOG( "\n MediaPlayerSample:: Failed to GetCurrentPlaybackPosition: Error : 0x%x \n", movieStatus );
            }
            getPlaybackPosition = getPlaybackPosition / (1000 * 1000);
            if( ( (getPlaybackPosition % 5 ) == 0ll) && ( lastPlaybackPosition != getPlaybackPosition))
            {
                NN_SDK_LOG( "Track Duration : %" PRId64 " Seconds ", trackDuration / 1000);
                NN_SDK_LOG( " Playback Position : %" PRId64 " Seconds\n", getPlaybackPosition);
                lastPlaybackPosition = getPlaybackPosition;
            }

            hidHandler.Update();

            if( isVideoTrack )
            {
                int yOffset = 0;
                int uvOffset = 0;
                int ystride = 0;
                int nativeWidth = 0;
                int nativeHeight = 0;
                int cropWidth = 0;
                int cropHeight = 0;
                int colorSpace = 0;
                int32_t bufferIndex = -1;
                int32_t trackNumber = -1;
                int64_t presentationTimeUs = 0;
                uint32_t flags = 0;
                int32_t bufferIndexForProperties = -1;

                if( ( config.returnVideoDataToClient == true ) && ( config.useClientMemoryForOutput == true ) )
                {
                    nn::os::LockMutex( &mediaPlayerObserver->m_VideoOutputBuffersListMutex );
                    uint32_t videoListSize = mediaPlayerObserver->m_VideoOutputBuffersList.size();

                    if( videoListSize > 0 )
                    {
                        bool videoAvailableForRendering = false;
                        MediaPlayerObserver::OutputBufferInfo videoBuffer;

                        videoBuffer = mediaPlayerObserver->m_VideoOutputBuffersList.front();
                        mediaPlayerObserver->m_VideoOutputBuffersList.erase( mediaPlayerObserver->m_VideoOutputBuffersList.begin());

                        nn::os::UnlockMutex( &mediaPlayerObserver->m_VideoOutputBuffersListMutex );

                        bufferIndex = videoBuffer.bufferIndex;
                        trackNumber = videoBuffer.trackNumber;

                        if( config.videoDecodeMode == movie::VideoDecodeMode_Cpu )
                        {
                            bufferIndexForProperties = bufferIndex;
                        }
                        else
                        {
                            //make sure the BufferIndex is one of the valid ones
                            for( int i = 0; i < mediaPlayerObserver->m_videoBufferCount; ++i )
                            {
                                if( bufferIndex == mediaPlayerObserver->m_videoBufferIndexArray[ i ] )
                                {
                                    bufferIndexForProperties = bufferIndex;
                                    mediaPlayerObserver->m_PresentationIndex = bufferIndex;
                                    videoAvailableForRendering = true;
                                    break;
                                }
                            }
                        }

                        player->GetOutputBufferProperties(bufferIndexForProperties, &mediaPlayerObserver->m_BufferProperty);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("nv12-y-stride", &ystride);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("nv12-y-offset", &yOffset);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("nv12-uv-offset", &uvOffset);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("width", &nativeWidth);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("height", &nativeHeight);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("crop-width", &cropWidth);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("crop-height", &cropHeight);
                        mediaPlayerObserver->m_BufferProperty.FindInt32("nv12-colorspace", &colorSpace);
                        if( mediaPlayerObserver->m_CropWidth != cropWidth || mediaPlayerObserver->m_CropHeight != cropHeight )
                        {
                            mediaPlayerObserver->m_CropWidth = cropWidth;
                            mediaPlayerObserver->m_CropHeight = cropHeight;
                            if( config.videoFormat == movie::OutputFormat_VideoColorNv12 )
                            {
                                NN_SDK_LOG("Video resolution changed, ResizeTextures : WxH %dx%d ystride %d\n", cropWidth, cropHeight, ystride);
                                if( ( cropWidth > 0 ) && ( cropHeight > 0 ) )
                                {
                                    mediaPlayerObserver->ResizeTextures(cropWidth, cropHeight);
                                }
                            }
                        }

                        if( config.videoDecodeMode == movie::VideoDecodeMode_Cpu )
                        {
                            movie::Status movieStatus = player->FillOutputBuffer(trackNumber, bufferIndex, &(mediaPlayerObserver->m_videobufferArray[0]), &presentationTimeUs, &flags);
                            if( movieStatus == movie::Status_Success )
                            {
                                videoAvailableForRendering = true;
                            }

                            uvOffset = nn::util::align_up(uvOffset, nv12UvAlignment);   // If "nv12-uv-alignment" is specified in the
                        }                                                               // buffer properties then uvOffset needs to be
                                                                                        // up-aligned accordingly.
                        if( videoAvailableForRendering )
                        {
                            mediaPlayerObserver->DrawVideoNvn(mediaPlayerObserver->m_PresentationIndex, cropWidth, cropHeight, yOffset, uvOffset, ystride, colorSpace);
                        }
                    }
                    else
                    {
                        nn::os::UnlockMutex( &mediaPlayerObserver->m_VideoOutputBuffersListMutex );
                    }
                }
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }

        NN_LOG( "\n MediaPlayerSample:: Playback is complete! \n" );

        NN_SDK_LOG( "\n MediaPlayerSample:: Playback Stop \n" );
        movieStatus = player->Stop();
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG( "\n MediaPlayerSample:: Failed to Stop: Error : 0x%x \n", movieStatus );
        }
        gettimeofday(&playerTimeVal, NULL);
        endTime = (int64_t)playerTimeVal.tv_sec * 1000 + (int64_t)playerTimeVal.tv_usec / 1000;

        NN_SDK_LOG(" \n Track Duration : %" PRId64 " msec ******* Total Playback Time : %" PRId64 " msec", trackDuration , (endTime - startTime));

        player->Reset();
        movie::Player::Destroy(player);
        //mediaPlayerObserver->DestroyNvn();
        NN_SDK_LOG( "\n MediaPlayerSample:: Player destroyed \n" );
    }
    else
    {
        NN_SDK_LOG( "\n MediaPlayerSample:: Could not create player \n" );
    }

Exit_PlayerSample:

    if( errorCleanUp == true )
    {
        movieStatus = player->Stop();
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG( "\n MediaPlayerSample:: Failed to Stop: Error : 0x%x \n", movieStatus );
            player->Reset();
        }
        movie::Player::Destroy(player);
        NN_SDK_LOG( "\n MediaPlayerSample:: Player destroyed \n" );
    }

    mediaPlayerObserver->Finalize(&config);

    if( isNetworkFile == true )
    {
        nn::socket::Finalize();
        nn::nifm::CancelNetworkRequest();
        curl_global_cleanup();
    }
    else if( sdcardMounted == true )
    {
#ifdef MOUNT_SDCARD
        nn::fs::Unmount("sdcard");
#endif
    }
    nn::fs::UnmountHostRoot();
    delete mediaPlayerObserver;

    nv::FinalizeGraphics();

    } // End of embedded scope

    if (USE_HEAP_TRACKING) {
        NN_LOG("mm heap\n");
        MMHeap().OutputUsage();
        NN_LOG("core heap\n");
        CoreHeap().OutputUsage();
        NN_LOG("malloc heap\n");
        MallocHeap().OutputUsage();
        NN_LOG("new heap\n");
        NewHeap().OutputUsage();
    }

} //NOLINT(impl/function_size)

