﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "StreamUtils.h"
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <sys/types.h>

#include <stdint.h>
#include <string>
#include <mutex>

#include "MseSourceBuffer.h"
#include <cinttypes>
#include <nn/nn_SdkLog.h>
#include <numeric>

//#define VERBOSE_LOGGING

template <typename...Ts>
static void VERBOSE_LOG(Ts&&...ts) noexcept {
#ifdef VERBOSE_LOGGING
    NN_LOG(ts...);
#endif
}

const int MseMaxVideoBufferSize = 64000;
const int MseMaxAudioBufferSize = 5000;
const int SourceBufferDefaultTrack = 0;

using namespace movie;
using namespace movie::sample;

MseSourceBuffer::MseSourceBuffer()
    :
    mCurrentSample(0),
    mCurrentSampleTrackIndex(0),
    mCurrentTimeStampOffset(0),
    mTimeStampOffset(0),
    mLength(-1),       ///< The file's size, negative if unknown
    mTimeOffsetUs(0),
    mSegmentIndex(0),
    mEOFReached(false),
    mMutex(true),
    mSegmentBuffers(NULL),
    mCurrentInitializationSegment(NULL),
    mCurrentDataSegment(NULL),
    mLastAppendedInitSegment(NULL),
    mLastAppendedSegment(NULL)
{

}

// ---------------------------------------------------------------------------
// ---------------------------------------------------------------------------
// ---------------------------------------------------------------------------

MseSourceBuffer::~MseSourceBuffer()
{
    DataSegment* iter = mSegmentBuffers;
    DataSegment* temp = nullptr;
    while (iter != nullptr)
    {
        temp = iter->next;
        delete iter;
        iter = temp;
    }
}

////////////////////////////////////////////////////////////////////////////////
// List Management

int64_t getDuration(movie::ClientStreamReader& client_stream_)
{
    movie::Extractor extractor{ movie::ContainerType::ContainerType_Mpeg4, movie::CacheSize::CacheSize_zero };
    auto result = extractor.SetDataSource(&client_stream_, "unused");
    if (result != movie::Status::Status_Success)
    {
        return -1; // invalid?
    }

    extractor.SelectTrack(SourceBufferDefaultTrack);
    int64_t begin_{},end_{};
    extractor.GetSampleTime(&begin_);

    while (extractor.Advance() == movie::Status_Success)
    {
        result = extractor.GetSampleTime(&end_);
    }
    return end_ - begin_;
}

int64_t MseSourceBuffer::appendData(void* data, size_t size, MseDataType type)
{
    if (!mLastAppendedInitSegment)
    {
        movie::sample::SimpleStream str_{reinterpret_cast<char const*>(data),size};
        int64_t dur = getDuration(str_);
        NN_SDK_LOG("init segment duration (sz == %u): %" PRId64 "\n", size, dur);
    }
    else {
        movie::sample::ComposedStream str_{reinterpret_cast<char const*>(mLastAppendedInitSegment->mData), static_cast<size_t>(mLastAppendedInitSegment->mLength),
                                           reinterpret_cast<char const*>(data),size};
        int64_t dur = getDuration(str_);
        NN_SDK_LOG("media segment duration (sz == %u): %" PRId64 "\n", size, dur);
    }

    std::lock_guard<nn::os::Mutex> lock_{ mMutex };
    // Check input parameters: data, size, type
    if ((type == MseDataType::DATA_SEGMENT || type == MseDataType::END_OF_FILE_SEGMENT)
        && mLastAppendedInitSegment == nullptr)
    {
        //return ERROR because you can't append data segments without an initialization segment
        return -1;
    }

    if (mLastAppendedSegment && mLastAppendedSegment->type == MseDataType::END_OF_FILE_SEGMENT)
    {
        //return ERROR because you can't append after signaling end of file segment
        return -1;
    }

    if (type == MseDataType::DATA_SEGMENT && size == 0)
    {
        //Return ERROR because you can't append a data segment of size 0
        return -1;
    }

    if (mLength < 0)
    {
        mLength = 0;
    }

    DataSegment* segment = new DataSegment(size);
    segment->mOffset = (mLastAppendedSegment != nullptr) ? mLastAppendedSegment->mOffset + mLastAppendedSegment->mLength : 0;
    segment->mLength = size;
    segment->type = type;
    segment->mCachedDurationUs = -1;
    segment->mSegmentIndex = mSegmentIndex;
    segment->mDataBegin = mLength;
    segment->mDataEnd = segment->mDataBegin + size;
    segment->mTimeOffsetUs = mTimeOffsetUs;
    segment->next = NULL;
    mSegmentIndex++;
    if (size > 0)
    {
        memcpy(segment->mData, data, size);
    }

    mLength += size;

    {
        switch (type)
        {
        case MseDataType::INITIALIZATION_SEGMENT:
        {
            // Signify the next data segment is different in some other way
            // (different bit rate, video dimensions, etc.)

            // If first initialization segment, append to the current buffer
            // If not first initialization segment, add to buffer list.
            // Then add to initialization segment list
            if (mSegmentBuffers == nullptr)
            {
                mSegmentBuffers = segment;
                mCurrentInitializationSegment = segment;
                mCurrentDataSegment = segment;
            }
            else
            {
                mLastAppendedInitSegment->nextInitialization = segment;
                mLastAppendedSegment->next = segment;
            }

            mLastAppendedSegment = segment;
            mLastAppendedInitSegment = segment;

            segment->mStartTimeUs = -1;

            movie::sample::SimpleStream str_{reinterpret_cast<char*>(data),size};
            movie::Extractor extractor{ movie::ContainerType::ContainerType_Mpeg4, movie::CacheSize::CacheSize_zero };
            movie::Status result = extractor.SetDataSource(&str_, "unused");

            NN_LOG("MSESource: InitSegment: SetDataSource: %d\n", result);
            //This should only succeed if its a single file broken up into internal segments
            {
                int trackCount;
                result = extractor.GetTrackCount(&trackCount);
                NN_LOG("MSESource: InitSegment: CountTracks: %d, %d\n", trackCount, result);

                result = extractor.SelectTrack(SourceBufferDefaultTrack);

                result = extractor.GetSampleTime((&(segment->mStartTimeUs)));
                NN_LOG("MSESource: InitSegment: GetSampleTime: %lld\n", segment->mStartTimeUs);

                if (result == movie::Status_Success) {
                    int64_t endTimeUs = 0;

                    while (extractor.Advance() == movie::Status_Success)
                    {
                        result = extractor.GetSampleTime(&endTimeUs);
                    };

                    if (result == movie::Status_EndOfStream && endTimeUs > segment->mStartTimeUs)
                    {
                        segment->mCachedDurationUs = endTimeUs - segment->mStartTimeUs;
                        result = movie::Status_Success;
                    }
                }

                NN_LOG("MSESource: InitSegment: GetCachedDuration: %lld\n", segment->mCachedDurationUs);
            }
        }
        break;
        case MseDataType::DATA_SEGMENT:
        {
            if (mLastAppendedSegment != nullptr)
            {
                mLastAppendedSegment = segment;
            }

            // Data Segment just appends to the current buffer

            // Create a ComposedStream for the data segment in order to caculate the duration
            // of the segment and then store the duration in DataSegment::mCachedDurationUs
            movie::sample::ComposedStream str_{reinterpret_cast<char*>(mLastAppendedInitSegment->mData),static_cast<size_t>(mLastAppendedInitSegment->mLength),
                                               reinterpret_cast<char*>(data), size};
            movie::Extractor extractor{ movie::ContainerType::ContainerType_Mpeg4, movie::CacheSize::CacheSize_zero };

            movie::Status result = extractor.SetDataSource(&str_,"unused");

            if (result == movie::Status_Success)
            {
                int trackCount;
                result = extractor.GetTrackCount(&trackCount);
                NN_LOG("MSESource: CountTracks: %d\n", trackCount);
                if (result != movie::Status_Success)
                {
                    VERBOSE_LOG("MSESource: Failed to get track count\n");
                }

                result = extractor.SelectTrack(SourceBufferDefaultTrack);

                result = extractor.GetSampleTime((&(segment->mStartTimeUs)));
                if (result != movie::Status_Success)
                {
                    VERBOSE_LOG("MSESource: Failed to get start time\n");
                }
                else {
                    int64_t endTimeUs = 0;

                    while (extractor.Advance() == movie::Status_Success)
                    {
                        result = extractor.GetSampleTime(&endTimeUs);
                    };

                    if (result == movie::Status_EndOfStream && endTimeUs > segment->mStartTimeUs)
                    {
                        segment->mCachedDurationUs = endTimeUs - segment->mStartTimeUs;
                        result = movie::Status_Success;
                    }
                }

                if (result != movie::Status_Success)
                {
                    VERBOSE_LOG("MSESource: Failed to get cached duration\n");
                }

                NN_LOG("MSESource: GetCachedDuration: %lld\n", segment->mCachedDurationUs);


                DataSegment* iter = mSegmentBuffers;

                //This code assumes the data segments are coming in the correct order
                for (; iter && iter->next; iter = iter->next)
                {
                }
                if (iter) iter->next = segment;

            }
            else
            {
                VERBOSE_LOG("MSESource: Failed to set datasource\n");
            }

            if (segment->mCachedDurationUs >= 0)
            {
                mTimeOffsetUs += segment->mCachedDurationUs;
            }
        }
        break;
        case MseDataType::END_OF_FILE_SEGMENT:
        {
            // Reached the end of the file and not just the end of the data buffer
            if (mLastAppendedSegment != nullptr)
            {
                mLastAppendedSegment = segment;
            }
            DataSegment* iter = mSegmentBuffers;
            while (iter->next != NULL)
            {
                iter = iter->next;
            }
            iter->next = segment;
            segment->next = NULL;

            segment->mStartTimeUs = -1;
            // Data Segment just appends to the current buffer
        }
        break;

        default:
        {
            break;
        }

        }
        DataSegment* iter = mSegmentBuffers;
        int64_t offsetTotal = 0;
        while (iter != NULL)
        {
            iter->mOffset = offsetTotal;
            iter->mDataBegin = iter->mOffset;
            iter->mDataEnd = iter->mOffset + iter->mLength;
            offsetTotal += iter->mLength;
            iter = iter->next;
        }
        return segment->mSegmentIndex;
    }
    return segment->mSegmentIndex; // There is an error
}//NOLINT(impl/function_size)

movie::Status MseSourceBuffer::processAppendedData()
{
    std::lock_guard<nn::os::Mutex> lock_{ mMutex };

    if (mSegmentBuffers == NULL)
    {
        return movie::Status::Status_NotEnoughData;
    }

    for (auto p = mSegmentBuffers->next; p != NULL; p = p->next)
    {
        NN_SDK_LOG("appending segment\n");
        movie::sample::ComposedStream str_{reinterpret_cast<char*>(mCurrentInitializationSegment->mData), static_cast<size_t>(mCurrentInitializationSegment->mLength),
                                           reinterpret_cast<char*>(p->mData), static_cast<size_t>(p->mLength)};
        movie::Extractor extractor{ movie::ContainerType::ContainerType_Mpeg4, movie::CacheSize::CacheSize_zero };
        extractor.SetDataSource(&str_, "unused");

        int track_count_{};
        extractor.GetTrackCount(&track_count_);
        for (int i = 0; i < track_count_; ++i)
        {
            extractor.SelectTrack(i);
        }
        if (mTrackFormats.empty()) {
            for (int i = 0; i < track_count_; ++i)
            {
                mTrackFormats.emplace_back();
                extractor.GetTrackConfiguration(i, &mTrackFormats.back());
                mTrackBuffers.emplace_back();
                mTrackBuffers.back().mCurrentSample = 0;
            }
        }
        int32_t max_input_sizes[2]{}; // we assume there are only two tracks
        for (size_t i = 0; i < std::min(size_t{ 2 }, mTrackFormats.size()); ++i)
        {
            mTrackFormats[i].FindInt32("max-input-size", &max_input_sizes[i]);
        }
        NN_SDK_LOG("initial input sizes == %d, %d\n",max_input_sizes[0],max_input_sizes[1]);

        // Work around below: "max-input-size" entries above are returning
        // bad (/high) values with our fragmented mp4 test content.
        // Because we're storing  way less than the "max-input-size" indicates after extraction
        // this leads to running out of memory.
        max_input_sizes[0] = MseMaxVideoBufferSize; // work-around value for video max input size, may need to be higher
        max_input_sizes[1] = MseMaxAudioBufferSize;  // work-around value for audio max input size, may need to be higher

        //This code assumes the data segments are coming in order
        //Otherwise the frames may need to be reordered by start time
        int64_t lastPresentationTime = mTimeStampOffset;
        movie::Status status_{ movie::Status::Status_Success };
        for(;;)
        {
            size_t track_index{};
            extractor.GetTrackIndexForAvailableData(&track_index);
            mTrackBuffers[track_index].frameBuffers.emplace_back(max_input_sizes[track_index], track_index);

            auto& buf = mTrackBuffers[track_index].frameBuffers.back().mData;
            buf.SetRange(0, 0);
            status_ = extractor.Read(&buf);
            if (status_ == movie::Status::Status_EndOfStream)
            {
                mTrackBuffers[track_index].frameBuffers.pop_back();
                break;
            }
            if (status_ != movie::Status::Status_Success)
            {
                mTrackBuffers[track_index].frameBuffers.pop_back();
                break;
            }
            extractor.GetSampleTime(&mTrackBuffers[track_index].frameBuffers.back().mPresentationTime);
            mTrackBuffers[track_index].frameBuffers.back().mPresentationTime += mTimeStampOffset; // We add a timestamp offset to make sure timestamps are monotonically increasing
            if (mTrackBuffers[track_index].frameBuffers.back().is_idr()) {
                mTrackBuffers[track_index].mIdrIndices.push_back(MseIdrEntry{ static_cast<int>(mTrackBuffers[track_index].frameBuffers.size()) - 1,
                    mTrackBuffers[track_index].frameBuffers.back().mPresentationTime });
            }
            extractor.Advance();
            lastPresentationTime = mTrackBuffers[track_index].frameBuffers.back().mPresentationTime;
        }
        mTimeStampOffset = lastPresentationTime;
    }
    for (int i = 0; i < mTrackBuffers.size(); ++i)
    {
        mTrackBuffers[i].frameBuffers.emplace_back(0, 0);
        mTrackBuffers[i].frameBuffers.back().mStatus = movie::Status::Status_EndOfStream;
        mTrackBuffers[i].frameBuffers.back().mPresentationTime = mTimeStampOffset;
    }
    return movie::Status::Status_Success;
}

movie::Status MseSourceBuffer::Advance()
{
    int64_t minPresentationTime = LLONG_MAX;
    int32_t targetTrackBuffer = -1;
    for (size_t i = 0; i < mTrackBuffers.size(); ++i)
    {
        if (mTrackBuffers[i].frameBuffers[mTrackBuffers[i].mCurrentSample].mPresentationTime <= minPresentationTime)
        {
            targetTrackBuffer = i;
            minPresentationTime = mTrackBuffers[i].frameBuffers[mTrackBuffers[i].mCurrentSample].mPresentationTime;
        }
    }

    if (targetTrackBuffer >= 0)
    {
        mCurrentSampleTrackIndex = targetTrackBuffer;
        mTrackBuffers[targetTrackBuffer].mCurrentSample++;
    }
    return movie::Status::Status_Success;
}

movie::Status MseSourceBuffer::ReadSampleData(movie::Buffer *buffer)
{
    if (buffer == nullptr)
    {
        return movie::Status::Status_Success;
    }

    if (mCurrentSampleTrackIndex >= 0 && mTrackBuffers[mCurrentSampleTrackIndex].frameBuffers.size() > mTrackBuffers[mCurrentSampleTrackIndex].mCurrentSample)
    {
        auto& src = mTrackBuffers[mCurrentSampleTrackIndex].frameBuffers[mTrackBuffers[mCurrentSampleTrackIndex].mCurrentSample].mData;
        mCurrentTimeStampOffset = mTrackBuffers[mCurrentSampleTrackIndex].frameBuffers[mTrackBuffers[mCurrentSampleTrackIndex].mCurrentSample].mPresentationTime;
        memcpy(buffer->Base(), src.Data(), src.Size());
        buffer->SetRange(src.Offset(), src.Size());
        buffer->SetInt32Data(src.GetInt32Data());
        buffer->SetInt64Data(src.GetInt64Data());
        buffer->SetUint32Data(src.GetUint32Data());
        return mTrackBuffers[mCurrentSampleTrackIndex].frameBuffers[mTrackBuffers[mCurrentSampleTrackIndex].mCurrentSample].mStatus;
    }

    return movie::Status::Status_EndOfStream;
}

// End List Management
////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
// Query Management

bool MseSourceBuffer::getEndOfData()
{
    return mEOFReached;
}

int MseSourceBuffer::getTotalDataBuffered()
{
    return mLength;
}

bool MseSourceBuffer::isTypeSupported(const char* in_mimetype, const char* in_codec, const char* in_system, const char* in_url)
{
    return true;
}

// End Query Management
////////////////////////////////////////////////////////////////////////////////
