﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "MediaPlayerUtilities.h"

// External config from MediaPlayerMseSample
extern bool USE_PROXY;
extern std::string PROXY_SETTING_HOST;
extern int PROXY_SETTING_PORT;
extern std::string PROXY_SETTING_USERPWD;
extern unsigned long PROXY_SETTING_AUTH;
extern bool USE_HEAP_TRACKING;

// Global Heaps/Trackers
nn::mem::StandardAllocator* g_MultimediaAllocator {nullptr};
HeapTracker* g_heap {nullptr};

char g_mmHeapBuffer[mmHeapSize];

nn::mem::StandardAllocator &GlobalAllocator()
{
    static nn::mem::StandardAllocator g_MultimediaAllocator(g_mmHeapBuffer, sizeof(g_mmHeapBuffer));
    return g_MultimediaAllocator;
}

HeapTracker &MMHeap()
{
    static HeapTracker g_heap;
    return g_heap;
}

HeapTracker &CoreHeap()
{
    static HeapTracker g_heap;
    return g_heap;
}

HeapTracker &MallocHeap()
{
    static HeapTracker g_heap;
    return g_heap;
}

HeapTracker &NewHeap()
{
    static HeapTracker g_heap;
    return g_heap;
}

void *mmAllocate(size_t size, size_t align, void *userPtr)
{
    if(size == 0)
        return NULL;
    static_cast<HeapTracker*>(userPtr)->m_mutex.Lock();
    void *address = GlobalAllocator().Allocate(size, align);
    static size_t totalAllocatedSize = 0;

    totalAllocatedSize += size;
    //~ NN_SDK_LOG("Allocated memory address   : 0x%p\n", address);
    //~ NN_SDK_LOG("Allocated memory size      : 0x%08zx\n", size);
    //~ NN_SDK_LOG("Total allocated memory size: 0x%08zx\n", totalAllocatedSize);
    //~ NN_SDK_LOG("\n");
    if (USE_HEAP_TRACKING)
    {
        static_cast<HeapTracker*>(userPtr)->Track(address, size);
    }
    static_cast<HeapTracker*>(userPtr)->m_mutex.Unlock();

    return address;
}

void mmDeallocate(void *address, void *userPtr)
{
    static_cast<HeapTracker*>(userPtr)->m_mutex.Lock();
    //~ NN_SDK_LOG("Deallocate memory address  : 0x%p\n", address);
    GlobalAllocator().Free(address);
    if (USE_HEAP_TRACKING)
    {
        static_cast<HeapTracker*>(userPtr)->Untrack(address);
    }
    static_cast<HeapTracker*>(userPtr)->m_mutex.Unlock();
}

void *mmReallocate(void* address, size_t newSize, void *userPtr)
{
    static_cast<HeapTracker*>(userPtr)->m_mutex.Lock();
    if (USE_HEAP_TRACKING)
    {
        if (address)
            static_cast<HeapTracker*>(userPtr)->Untrack(address);
    }
    void *memory = GlobalAllocator().Reallocate(address, newSize);
    if (USE_HEAP_TRACKING)
    {
        static_cast<HeapTracker*>(userPtr)->Track(memory, newSize);
    }
    static_cast<HeapTracker*>(userPtr)->m_mutex.Unlock();
    return memory;
}

//------------------------------------------------------------------------------
extern "C" void* malloc(size_t aSize)
{
    return mmAllocate(aSize, 8, &MallocHeap());
}

//------------------------------------------------------------------------------
extern "C" void free(void* aPtr)
{
    if (aPtr) {
        mmDeallocate(aPtr, &MallocHeap());
    }
}

//------------------------------------------------------------------------------
extern "C" void* calloc(size_t aNum, size_t aSize)
{
    const size_t sum = aNum * aSize;
    void* p = malloc(sum);
    if (p)
    {
        ::std::memset(p, 0, sum);
    }
    return p;
}

//------------------------------------------------------------------------------
extern "C" void* realloc(void* aPtr, size_t aNewSize)
{
    return mmReallocate(aPtr, aNewSize, &MallocHeap());
}

//------------------------------------------------------------------------------
extern "C" void* aligned_alloc(size_t aAlignment, size_t aSize)
{
    return mmAllocate(aSize, aAlignment, &MallocHeap());
}

//------------------------------------------------------------------------------
void* operator new(::std::size_t aSize, const ::std::nothrow_t&) throw()
{
    if(aSize == 0)
        aSize = 1;
    void *memory = mmAllocate(static_cast<int>(aSize), 8, &NewHeap());
    return memory;
}

//------------------------------------------------------------------------------
void* operator new[](::std::size_t aSize, const ::std::nothrow_t&) throw()
{
    return operator new(aSize, ::std::nothrow_t());
}

//------------------------------------------------------------------------------
void* operator new(::std::size_t aSize) throw(::std::bad_alloc)
{
    return operator new(aSize, ::std::nothrow_t());
}

//------------------------------------------------------------------------------
void* operator new[](::std::size_t aSize) throw(::std::bad_alloc)
{
    return operator new(aSize, ::std::nothrow_t());
}

//------------------------------------------------------------------------------
void operator delete(void* aPtr) throw()
{
    if (aPtr) {
        mmDeallocate(aPtr, &NewHeap());
    }
}

//------------------------------------------------------------------------------
void operator delete[](void* aPtr) throw()
{
    operator delete(aPtr);
}

bool ProcessConfigFile(const char *file)
{
    nn::fs::FileHandle fileHandle;
    nn::Result result = nn::fs::OpenFile(&fileHandle, file, nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        NN_SDK_LOG( "\n Failed to open %s \n \n Terminating PlayerSample MediaPlayerMseSample\n", file );
        return false;
    }
    int64_t fileSize;
    nn::fs::GetFileSize(&fileSize, fileHandle);
    std::string fileData(fileSize, 'a');
    nn::fs::ReadFile(fileHandle, 0, &fileData[0], fileSize);
    for(size_t location = 0; location != std::string::npos; )
    {
        size_t newLine = fileData.find('\n', location);
        std::string line = fileData.substr(location, newLine - location);
        location = newLine;
        if(location != std::string::npos)
            ++location;

        size_t equals = line.find('=');
        std::string k = line.substr(0, equals);
        std::string v = line.substr(equals + 1);
        if(k == "USE_PROXY")
            USE_PROXY = atoi(v.c_str());
        else if(k == "PROXY_SETTING_HOST")
            PROXY_SETTING_HOST = v;
        else if(k == "PROXY_SETTING_PORT")
            PROXY_SETTING_PORT = atoi(v.c_str());
        else if(k == "PROXY_SETTING_USERPWD")
            PROXY_SETTING_USERPWD = v;
        else if(k == "PROXY_SETTING_AUTH")
        {
            if(v == "CURLAUTH_BASIC")
                PROXY_SETTING_AUTH = CURLAUTH_BASIC;
            else
                NN_SDK_LOG("Unknown PROXY_SETTING_AUTH option %s\n", v.c_str());
        }
        else
            NN_SDK_LOG("Unknown config file option %s\n", k.c_str());
    }
    nn::fs::CloseFile(fileHandle);
    return true;
}

bool ProcessUrlFile(const char *file, std::string *str)
{
    nn::fs::FileHandle fileHandle;
    nn::Result result = nn::fs::OpenFile(&fileHandle, file, nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        NN_SDK_LOG("\n Failed to open %s \n \n Terminating PlayerSample MediaPlayerMseSample\n", file);
        return false;
    }
    int64_t fileSize;
    nn::fs::GetFileSize(&fileSize, fileHandle);
    std::string fileData(fileSize, '\0');
    nn::fs::ReadFile(fileHandle, 0, &fileData[0], fileSize);
    size_t pos;
    pos = fileData.find_first_of('\r');
    if (pos != std::string::npos)
    {
        fileData[pos] = '\0';
    }
    pos = fileData.find_first_of('\n');
    if (pos != std::string::npos)
    {
        fileData[pos] = '\0';
    }
    *str = fileData;
    nn::fs::CloseFile(fileHandle);
    return true;
}
