﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include "UpnpDeviceGamePlayer.h"

#include <cstring>

namespace nns {
namespace libupnp {

//***************************************************************
//*
//*                    G L O B A L s
//*
//***************************************************************

const int    UpnpDeviceGamePlayer::UpnpSearchTimeout = 5;
const int    UpnpDeviceGamePlayer::UpnpSubscriptionTimeout = 1801;
const int    UpnpDeviceGamePlayer::MaxMessages = 5;

//***************************************************************
//*
//* UpnpDeviceGamePlayer
//*
//***************************************************************

UpnpDeviceGamePlayer::UpnpDeviceGamePlayer() NN_NOEXCEPT : m_MessagesChanged(false),
                                                           m_UpnpClientHandle(-1),
                                                           m_ValueChanged(false),
                                                           m_IsUpnpRegistered(false)
{
    nn::os::InitializeMutex(&m_MutexLock, false, 0);
    nn::os::InitializeMutex(&m_MessagesLock, false, 0);

    memset(&m_DeviceSockAddr, 0, sizeof(m_DeviceSockAddr));
    memset(m_Udn, 0, sizeof(m_Udn));
    memset(m_EventUrl, 0, sizeof(m_EventUrl));
    memset(m_ControlUrl, 0, sizeof(m_ControlUrl));
    memset(m_PresentationUrl, 0, sizeof(m_PresentationUrl));
    m_SubTimeout  = UpnpSubscriptionTimeout;
    m_DeviceFound = false;

    for(int idx = 0; idx < UpnpDeviceGameMaxVars; idx++)
    {
        m_VariableValues[idx] = m_VariableValuesDef[idx];
        strcpy(m_VariableValues[idx], "0");
    }

    m_Messages.clear();
}

UpnpDeviceGamePlayer::~UpnpDeviceGamePlayer() NN_NOEXCEPT
{
    // If object is (still) Registered to UPNP..
    if (m_IsUpnpRegistered == true)
    {
        Stop();
    }

    nn::os::LockMutex(&m_MutexLock);

    memset(&m_DeviceSockAddr, 0, sizeof(m_DeviceSockAddr));
    memset(m_Udn, 0, sizeof(m_Udn));
    memset(m_EventUrl, 0, sizeof(m_EventUrl));
    memset(m_ControlUrl, 0, sizeof(m_ControlUrl));
    memset(m_PresentationUrl, 0, sizeof(m_PresentationUrl));
    m_SubTimeout  = UpnpSubscriptionTimeout;
    m_DeviceFound = false;
    m_ValueChanged = false;
    m_MessagesChanged = false;
    m_Messages.clear();

    nn::os::UnlockMutex(&m_MutexLock);

    // Destroy the Mutex
    nn::os::FinalizeMutex(&m_MutexLock);
    nn::os::FinalizeMutex(&m_MessagesLock);
}

bool UpnpDeviceGamePlayer::IsNintendoGameDiscovered() NN_NOEXCEPT
{
    return(m_DeviceFound);
}

bool UpnpDeviceGamePlayer::DidNintendoGameValueChange() NN_NOEXCEPT
{
    return(m_ValueChanged);
}

bool UpnpDeviceGamePlayer::DidMessagesChange() NN_NOEXCEPT
{
    return(m_MessagesChanged);
}

const char* UpnpDeviceGamePlayer::GetUdn() NN_NOEXCEPT
{
    return(m_Udn);
}

const char* UpnpDeviceGamePlayer::GetEventUrl() NN_NOEXCEPT
{
    return(m_EventUrl);
}

const char* UpnpDeviceGamePlayer::GetControlUrl() NN_NOEXCEPT
{
    return(m_ControlUrl);
}

int UpnpDeviceGamePlayer::SetA(const int inValue) NN_NOEXCEPT
{
    return(SetGameValue(ActionSetNames[0], VariableNames[0], inValue));
}

int UpnpDeviceGamePlayer::SetB(const int inValue) NN_NOEXCEPT
{
    return(SetGameValue(ActionSetNames[1], VariableNames[1], inValue));
}

int UpnpDeviceGamePlayer::SetC(const int inValue) NN_NOEXCEPT
{
    return(SetGameValue(ActionSetNames[2], VariableNames[2], inValue));
}

int UpnpDeviceGamePlayer::Start() NN_NOEXCEPT
{
    int          rc = 0;
    int          ret = 0;
    const char*  pIpAddress = nullptr;
    uint16_t     port = 0;

    nn::os::LockMutex(&m_MutexLock);

    do
    {
        // If already registered?
        if (m_IsUpnpRegistered == true)
        {
            NN_LOG( "[UpnpDeviceGamePlayer] is already initialized!\n");
            ret = -1;
            break;
        }

        pIpAddress = UpnpGetServerIpAddress();
        if (pIpAddress == nullptr)
        {
            NN_LOG( "Failed to get IP Address from libupnp\n" );
            ret = -1;
            break;
        }

        port = UpnpGetServerPort();
        if (port < 1)
        {
           NN_LOG( "Failed to get Network Port from libupnp\n" );
            ret = -1;
            break;
        }
        AddMessage("[UpnpDeviceGamePlayer] Initializing on IP Address=[%s], Port=[%d]\n", pIpAddress, port);

        rc = UpnpRegisterClient(UpnpCallbackEventHandler, reinterpret_cast<void*>(this), &m_UpnpClientHandle);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG("** ERROR UpnpRegisterClient(): %d/%s\n", rc, UpnpGetErrorMessage(rc));
            ret = -1;
            break;
        }
        m_IsUpnpRegistered = true;

        NN_LOG ( "[UpnpDeviceGamePlayer] Initialized\n" );

        // Search specifically for devices that support the Nintendo Game Device
        rc = UpnpSearchAsync(m_UpnpClientHandle, UpnpSearchTimeout, DeviceType, reinterpret_cast<void*>(this));
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG("** ERROR UpnpSearchAsync(): %d/%s\n", rc, UpnpGetErrorMessage(rc));
            ret = -1;
            break;
        }

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    if (ret != 0)
    {
        if (m_IsUpnpRegistered == true)
        {
            rc = UpnpUnRegisterClient(m_UpnpClientHandle);
            if (rc != UPNP_E_SUCCESS)
            {
                NN_LOG ("** ERROR UpnpUnRegisterClient(): %d/%s\n", rc, UpnpGetErrorMessage(rc));
                // Fall Thru
            }
            m_IsUpnpRegistered = false;
        }
    }

    nn::os::UnlockMutex(&m_MutexLock);

    return(ret);
}

int UpnpDeviceGamePlayer::Stop() NN_NOEXCEPT
{
    int   rc = -1;
    int   ret = 0;

    nn::os::LockMutex(&m_MutexLock);

    if (m_IsUpnpRegistered == true)
    {
        rc = UpnpUnRegisterClient(m_UpnpClientHandle);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG ("** ERROR UpnpUnRegisterClient(): %d/%s\n", rc, UpnpGetErrorMessage(rc));
            ret = -1;
            // Fall Thru
        }
        m_IsUpnpRegistered = false;
    }

    nn::os::UnlockMutex(&m_MutexLock);

    return(ret);
}

int UpnpDeviceGamePlayer::AddDevice(IXML_Document* pDescDoc, struct Upnp_Discovery* pDiscovery, bool* isNintendoGame) NN_NOEXCEPT
{
    IXML_NodeList*         pServiceList = nullptr;
    IXML_Element*          pService = nullptr;
    struct sockaddr_in*    pSockAddr = nullptr;
    int                    ret = 0;
    int                    length = 0;
    int                    idx = 0;
    int                    idx2 = 0;
    char                   tempVal[200];
    bool                   serviceFound = false;

    do
    {
        if (pDescDoc == nullptr || pDiscovery == nullptr)
        {
            NN_LOG( "Input arguments are NULL\n" );
            ret = -1;
            break;
        }
        *isNintendoGame = false;

        ret = GetFirstDocumentItem(tempVal, sizeof(tempVal), pDescDoc, "deviceType");
        if (ret != 0)
        {
            NN_LOG("GetFirstDocumentItem() did not find deviceType!\n" );
            ret = -1;
            break;
        }

        NN_LOG("Compare Device: [%s] to [%s]!\n", tempVal, DeviceType);

        // If this UPnP Device is a Nintendo Game?
        if ( (strlen(tempVal) == strlen(DeviceType))                    &&
             (strncmp(tempVal, DeviceType, strlen(DeviceType)) == 0) )
        {
            *isNintendoGame = true;   // Yes
        }
        else
        {
            break;   // No
        }

        ret = GetFirstDocumentItem(m_Udn, sizeof(m_Udn), pDescDoc, "UDN");
        if (ret != 0)
        {
            NN_LOG("GetFirstDocumentItem() did not find UDN!\n" );
            ret = -1;
            break;
        }

        ret = GetFirstDocumentItem(tempVal, sizeof(tempVal), pDescDoc, "presentationURL");
        if (ret != 0)
        {
            NN_LOG("GetFirstDocumentItem() did not find presentationURL!\n" );
            ret = -1;
            break;
        }

        ret = UpnpResolveURL(pDiscovery->Location, tempVal, m_PresentationUrl);
        if (ret != UPNP_E_SUCCESS)
        {
            NN_LOG("Failed calling 'UpnpResolveURL' to resolve the Presentation URL - Error: %d/%s\n",
                                                                 ret, UpnpGetErrorMessage(ret));
            ret = -1;
            break;
        }

        // Process the Nintendo Game Service
        serviceFound = false;
        idx = 0;
        while(serviceFound == false)
        {
            pServiceList = GetNthServiceList(pDescDoc, idx);
            if (pServiceList == nullptr)
            {
                break;
            }
            idx++;

            length = ixmlNodeList_length(pServiceList);
            for (idx2 = 0; idx2 < length; idx2++)
            {
                pService = reinterpret_cast<IXML_Element*>(ixmlNodeList_item(pServiceList, idx2));
                if (pService == nullptr)
                {
                    NN_LOG("Failed to get 'service' at level: %d\n", idx2);
                    ret = -1;
                    break;
                }

                ret = GetFirstElementItem(tempVal, sizeof(tempVal), pService, "serviceType");
                if (ret != 0)
                {
                    NN_LOG("Failed to get 'serviceType'\n");
                    ret = -1;
                    break;
                }

                NN_LOG("Compare Service: [%s] to [%s]!\n", tempVal, ServiceType);

                // Is this the right Service?
                if ( (strlen(tempVal) == strlen(ServiceType))                     &&
                     (strncmp(tempVal, ServiceType, strlen(ServiceType)) == 0) )
                {
                    ;   // Yes!
                }
                else
                {
                    continue;   // No
                }

                ret = GetFirstElementItem(tempVal, sizeof(tempVal), pService, "controlURL");
                if (ret != 0)
                {
                    NN_LOG("Failed to get 'controlURL' from service: %s\n", ServiceType);
                    ret = -1;
                    break;
                }

                ret = UpnpResolveURL(pDiscovery->Location, tempVal, m_ControlUrl);
                if (ret != UPNP_E_SUCCESS)
                {
                    NN_LOG("Failed calling 'UpnpResolveURL' to resolve the Control URL - Error: %d/%s\n",
                                                                 ret, UpnpGetErrorMessage(ret));
                    ret = -1;
                    break;
                }

                ret = GetFirstElementItem(tempVal, sizeof(tempVal), pService, "eventSubURL");
                if (ret != 0)
                {
                    NN_LOG("Failed to get 'eventURL' from service: %s\n", ServiceType);
                    ret = -1;
                    break;
                }

                ret = UpnpResolveURL(pDiscovery->Location, tempVal, m_EventUrl);
                if (ret != UPNP_E_SUCCESS)
                {
                    NN_LOG("Failed calling 'UpnpResolveURL' to resolve the Event URL - Error: %d/%s\n",
                                                                 ret, UpnpGetErrorMessage(ret));
                    ret = -1;
                    break;
                }

                // Service fields found
                serviceFound = true;

                // All done!
                break;
            }
        }

        // If error while searching for Service Fields...
        if (ret < 0)
        {
            break;
        }

        // If we did not find the Service Fields
        if (serviceFound == false)
        {
            NN_LOG( "Failed to find all required Service Fields!\n" );
            break;
        }

        // Save Socket Address of Advertised Device
        memset(&m_DeviceSockAddr, 0, sizeof(m_DeviceSockAddr));
        memcpy(&m_DeviceSockAddr, &pDiscovery->DestAddr, sizeof(m_DeviceSockAddr));

        // Display the Record
        {
            NN_LOG( "UDN              : %s\n", m_Udn);
            NN_LOG( "Event URL        : %s\n", m_EventUrl);
            NN_LOG( "Control URL      : %s\n", m_ControlUrl);
            NN_LOG( "Presentation URL : %s\n", m_PresentationUrl);
        }

        pSockAddr = (struct sockaddr_in *) &m_DeviceSockAddr;
        AddMessage("[UPnP Discovery] Nintendo Game [%s] discovered at [%s]\n", m_Udn, nn::socket::InetNtoa(pSockAddr->sin_addr));

        // Nintendo Game found!
        m_DeviceFound = true;

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    return(ret);

}  // NOLINT(impl/function_size)

int UpnpDeviceGamePlayer::Subscribe() NN_NOEXCEPT
{
    int   ret = 0;
    int   rc = -1;

    do
    {
        if (m_DeviceFound == false)
        {
            NN_LOG( "No Nintendo Games have been found on the LAN - Please discover a Nintendo Game first\n");
            break;
        }

        // Initialize Subscription Timeout
        m_SubTimeout  = UpnpSubscriptionTimeout;

        rc = UpnpSubscribe(m_UpnpClientHandle, m_EventUrl, &m_SubTimeout, m_SubId);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG("** Error UpnpSubscribe() to Event URL [%s] Failed - Error: %d/%s\n", m_EventUrl, rc, UpnpGetErrorMessage(rc));
            ret = -1;
            break;
        }

        NN_LOG( "Successfully Subscribed to Event URL [%s] - Subscription Timeout is [%d]\n", m_EventUrl, m_SubTimeout);

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    return(ret);
}

int UpnpDeviceGamePlayer::UnSubscribe() NN_NOEXCEPT
{
    int   ret = 0;
    int   rc = -1;

    do
    {
        if (m_DeviceFound == false)
        {
            NN_LOG( "No Nintendo Games have been found on the LAN - Please discover a Nintendo Game first\n");
            break;
        }

        rc = UpnpUnSubscribe(m_UpnpClientHandle, m_SubId);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG("** Error UpnpUnSubscribe() to Event URL [%s] Failed - Error: %d/%s\n", m_EventUrl, rc, UpnpGetErrorMessage(rc));
            ret = -1;
            break;
        }

        // Initialize Subscription Timeout
        m_SubTimeout = 0;
        memset(&m_SubId, 0, sizeof(m_SubId));

        NN_LOG( "Successfully UnSubscribed to Event URL [%s]\n", m_EventUrl);

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    return(ret);
}

int UpnpDeviceGamePlayer::HandleEvent(struct Upnp_Event* pEvent) NN_NOEXCEPT
{
    IXML_NodeList* pProperties = nullptr;
    IXML_NodeList* pVariables = nullptr;
    IXML_Element*  pPropertyElem = nullptr;
    IXML_Element*  pVarElem = nullptr;
    long           len = 0;
    long           len1 = 0;
    long           idx = 0;
    int            idx2 = 0;
    int            ret = 0;
    int            rc = -1;

    NN_LOG( "HandleEvent() - Enter\n");

    do
    {
        if (pEvent == nullptr)
        {
            NN_LOG( "Event is NULL");
            ret = -1;
            break;
        }

        pProperties = ixmlDocument_getElementsByTagName(pEvent->ChangedVariables, "e:property");
        if (pProperties == nullptr)
        {
            NN_LOG( "ixmlDocument_getElementsByTagName() Failed to find e:property");
            ret = -1;
            break;
        }

        len = ixmlNodeList_length(pProperties);
        if (len < 1)
        {
            NN_LOG( "ixmlNodeList_length() returned (NO) properties\n");
            ret = -1;
            break;
        }

        for(idx = 0; idx < len; idx++)
        {
            pPropertyElem = reinterpret_cast<IXML_Element *>(ixmlNodeList_item(pProperties, idx));

            for(idx2 = 0; idx2 < UpnpDeviceGameMaxVars; idx2++)
            {
                pVariables = ixmlElement_getElementsByTagName(pPropertyElem, VariableNames[idx2]);
                if (pVariables == nullptr)
                {
                    continue;
                }

                len1 = ixmlNodeList_length(pVariables);
                if (len1 < 1)
                {
                    ixmlNodeList_free(pVariables);
                    pVariables = nullptr;
                    continue;
                }

                pVarElem = reinterpret_cast<IXML_Element *>(ixmlNodeList_item(pVariables, 0));
                if (pVarElem == nullptr)
                {
                    ixmlNodeList_free(pVariables);
                    pVariables = nullptr;
                    continue;
                }

                rc = GetElementValue(m_VariableValues[idx2], UpnpDeviceGameVarLength, pVarElem);
                if (rc != 0)
                {
                    ixmlNodeList_free(pVariables);
                    pVariables = nullptr;
                    continue;
                }

                NN_LOG("Variable Name: [%s] New Value: [%s]\n",
                            VariableNames[idx2], m_VariableValues[idx2]);

                // Value Changed!
                m_ValueChanged = true;

                // Free Variables
                ixmlNodeList_free(pVariables);
                pVariables = nullptr;
            }
        }

        ixmlNodeList_free(pProperties);
        pProperties = nullptr;

        AddMessage("[UPnP Subscription Event Recieved]: Game Variable: A = [%s], B = [%s], C = [%s]\n",
                                   m_VariableValues[0], m_VariableValues[1], m_VariableValues[2] );

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    NN_LOG( "HandleEvent() - Exit\n");

    return(ret);
}

int UpnpDeviceGamePlayer::SetGameValue(const char* pActionname, const char* pVariableName, const int variableValue) NN_NOEXCEPT
{
    int          ret = 0;
    char         numericValue[UpnpDeviceGameVarLength];
    const char*  pNumericValue = nullptr;

    ret = snprintf(numericValue, UpnpDeviceGameVarLength, "%d", variableValue);
    if (ret > UpnpDeviceGameVarLength)
    {
        NN_LOG( "Turning numeric value [%d] into a String value had length [%d], which exceeded Max Value size [%d]!\n",
                        variableValue, ret, UpnpDeviceGameVarLength);
        return(-1);
    }
    AddMessage("[UPnP SOAP Action Sent]: Set Variable [%s] to Value [%s]\n", pVariableName, numericValue);

    pNumericValue = numericValue;
    ret = SendAction(pActionname, reinterpret_cast<const char **>(&pVariableName), &pNumericValue, 1);
    if (ret != 0)
    {
        NN_LOG("SendAction() failed trying to send Value [%s]\n", pVariableName);
        return(-1);
    }

    return(0);
}

int UpnpDeviceGamePlayer::GetGameValues(int& outValueA, int& outValueB, int& outValueC) NN_NOEXCEPT
{
    int  ret = 0;

    nn::os::LockMutex(&m_MutexLock);

    do
    {
        // ValueA
        outValueA = atoi(m_VariableValues[0]);
        if (outValueA >= UpnpDeviceGameColor_Min && outValueA <= UpnpDeviceGameColor_Max)
        {
            ;  // Legal Value
        }
        else
        {
            NN_LOG( "[%s] - Game Value [%d] is not a legal value between [%d] and [%d]!\n",
                        VariableNames[0], outValueA, UpnpDeviceGameColor_Min, UpnpDeviceGameColor_Max);
            ret = -1;
            break;
        }

        // ValueB
        outValueB = atoi(m_VariableValues[1]);
        if (outValueB >= UpnpDeviceGameColor_Min && outValueB <= UpnpDeviceGameColor_Max)
        {
            ;  // Legal Value
        }
        else
        {
            NN_LOG( "[%s] - Game Value [%d] is not a legal value between [%d] and [%d]!\n",
                        VariableNames[1], outValueB, UpnpDeviceGameColor_Min, UpnpDeviceGameColor_Max);
            ret = -1;
            break;
        }

        // ValueC
        outValueC = atoi(m_VariableValues[2]);
        if (outValueC >= UpnpDeviceGameColor_Min && outValueC <= UpnpDeviceGameColor_Max)
        {
            ;  // Legal Value
        }
        else
        {
            NN_LOG( "[%s] - Game Value [%d] is not a legal value between [%d] and [%d]!\n",
                        VariableNames[2], outValueC, UpnpDeviceGameColor_Min, UpnpDeviceGameColor_Max);
            ret = -1;
            break;
        }

        // Clear Value Changed Indicator
        m_ValueChanged = false;

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    nn::os::UnlockMutex(&m_MutexLock);

    return(ret);
}

int UpnpDeviceGamePlayer::SendAction(const char* pActionname, const char** pParam_name, const char** pParam_val, int paramCount) NN_NOEXCEPT
{
    IXML_Document* pActionNode = nullptr;
    int            idx = 0;
    int            ret = 0;
    int            rc = -1;

    nn::os::LockMutex(&m_MutexLock);

    do
    {
        if (paramCount == 0)
        {
            pActionNode = UpnpMakeAction(pActionname, ServiceType, 0, nullptr);
            if (pActionNode == nullptr)
            {
                NN_LOG( "UpnpMakeAction() Failed: %d/%s\n", rc, UpnpGetErrorMessage(rc));
                ret = -1;
                break;
            }
        }
        else
        {
            for(idx = 0; idx < paramCount; idx++ )
            {
                NN_LOG( "Setting [%s] to [%s] using Action [%s]\n", pParam_name[idx], pParam_val[idx], pActionname);
                rc = UpnpAddToAction(&pActionNode, pActionname, ServiceType, pParam_name[idx], pParam_val[idx]);
                if (rc != UPNP_E_SUCCESS)
                {
                    NN_LOG( "UpnpAddToAction() Failed: %d/%s\n", rc, UpnpGetErrorMessage(rc));
                    ret = -1;
                    break;
                }
            }
        }

        rc = UpnpSendActionAsync( m_UpnpClientHandle, m_ControlUrl, ServiceType, nullptr,
                                  pActionNode, UpnpCallbackEventHandler, reinterpret_cast<void*>(this));
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG( "UpnpSendActionAsync() Failed: %d/%s\n", rc, UpnpGetErrorMessage(rc));
            ret = -1;
            break;
        }

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    nn::os::UnlockMutex(&m_MutexLock);

    return(ret);
}

int UpnpDeviceGamePlayer::HandleAction(struct Upnp_Action_Complete* pEvent) NN_NOEXCEPT
{
    IXML_NodeList* pNodeList = nullptr;
    int            ret = -1;
    int            idx = 0;
    char           newValue[UpnpDeviceGameVarLength];

    NN_LOG("UpnpDeviceGamePlayer::HandleAction() - Entry\n");

    do
    {
        for(idx = 0; idx < UpnpDeviceGameMaxVars; idx++)
        {
            pNodeList = ixmlDocument_getElementsByTagName(pEvent->ActionResult, VariableNames[idx]);
            if (pNodeList == nullptr)
            {
                continue;
            }
            ixmlNodeList_free(pNodeList);
            pNodeList = nullptr;

            ret = GetFirstDocumentItem(newValue, sizeof(newValue), pEvent->ActionResult, VariableNames[idx]);
            if (ret != 0)
            {
                continue;
            }

            NN_LOG("Found [%s] - New Value [%s]\n", VariableNames[idx], newValue);

            AddMessage("[UPnP SOAP Action Recieved]: Game Variable [%s] set to [%s]\n",
                                   VariableNames[idx], newValue);

            // Save Legal Value
            memset(m_VariableValues[idx], 0, UpnpDeviceGameVarLength);
            memcpy(m_VariableValues[idx], newValue, strlen(newValue));

            // Value Changed!
            m_ValueChanged = true;
        }

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    if (pNodeList != nullptr)
    {
        ixmlNodeList_free(pNodeList);
        pNodeList = nullptr;
    }

    NN_LOG("UpnpDeviceGamePlayer::HandleAction() - Exit\n");

    return(ret);
}

void UpnpDeviceGamePlayer::UpnpPrintDiscoveryStructure(struct Upnp_Discovery * pDiscover) NN_NOEXCEPT
{
    NN_LOG("ErrCode     =  %s(%d)\n", UpnpGetErrorMessage(pDiscover->ErrCode), pDiscover->ErrCode);
    NN_LOG("Expires     =  %d\n",  pDiscover->Expires);
    NN_LOG("DeviceId    =  %s\n",  pDiscover->DeviceId);
    NN_LOG("DeviceType  =  %s\n",  pDiscover->DeviceType);
    NN_LOG("ServiceType =  %s\n",  pDiscover->ServiceType);
    NN_LOG("ServiceVer  =  %s\n",  pDiscover->ServiceVer);
    NN_LOG("Location    =  %s\n",  pDiscover->Location);
    NN_LOG("OS          =  %s\n",  pDiscover->Os);
    NN_LOG("Ext         =  %s\n",  pDiscover->Ext);
}

int UpnpDeviceGamePlayer::UpnpCallbackEventHandler(Upnp_EventType eventType, void* pEvent, void* pCookie) NN_NOEXCEPT
{
    UpnpDeviceGamePlayer* pThisObj = reinterpret_cast<UpnpDeviceGamePlayer*>(pCookie);

    switch ( eventType )
    {
    case UPNP_DISCOVERY_ADVERTISEMENT_ALIVE:
    case UPNP_DISCOVERY_SEARCH_RESULT:
         {
             struct Upnp_Discovery* pDiscover = (struct Upnp_Discovery *) pEvent;
             IXML_Document*         pDescDoc = nullptr;
             int                    ret = -1;
             bool                   isNintendoGame = false;

             nn::os::LockMutex(&pThisObj->m_MutexLock);

             do
             {
                 // If a Nintendo Game has already been found..
                 if (pThisObj->m_DeviceFound == true)
                 {
                     break;
                 }

                 // Skip Devkits for now
                 if ( (strlen(pDiscover->DeviceId) > 10)                    &&
                      (memcmp(pDiscover->DeviceId, "uuid:DevKit", 11) == 0) )
                 {
                     break;
                 }

#ifdef COMMENTED_OUT
                 pThisObj->UpnpPrintDiscoveryStructure(pDiscover);
#endif

                 if (pDiscover->ErrCode != UPNP_E_SUCCESS)
                 {
                     NN_LOG( "Error in Discovery Callback -- %d/%s\n",
                                      pDiscover->ErrCode, UpnpGetErrorMessage(pDiscover->ErrCode));
                     break;
                 }

                 ret = UpnpDownloadXmlDoc(pDiscover->Location, &pDescDoc);
                 if (ret != UPNP_E_SUCCESS)
                 {
                     NN_LOG( "Error obtaining device description from [%s] -- error = %d/%s\n",
                                 pDiscover->Location, ret, UpnpGetErrorMessage(ret));
                     break;
                 }

                 ret = pThisObj->AddDevice(pDescDoc, pDiscover, &isNintendoGame);
                 if (ret != 0)
                 {
                     NN_LOG( "Failed calling AddDevice()\n" );
                     break;
                 }

                 // All done
                 break;

             } while (NN_STATIC_CONDITION(false));

             nn::os::UnlockMutex(&pThisObj->m_MutexLock);

             if (pDescDoc != nullptr)
             {
                 ixmlDocument_free(pDescDoc);
                 pDescDoc = nullptr;
             }
         }
         break;

         // Subscription Event Received
         case UPNP_EVENT_RECEIVED:
         {
             struct Upnp_Event* pUpnpEvent = reinterpret_cast<struct Upnp_Event*>(pEvent);
             int                ret = 0;

             nn::os::LockMutex(&pThisObj->m_MutexLock);

             do
             {
                 ret = pThisObj->HandleEvent(pUpnpEvent);
                 if (ret != 0)
                 {
                     NN_LOG( "Failed calling HandleEvent()\n" );
                     break;
                 }

                 // All done
                 break;

            } while (NN_STATIC_CONDITION(false));

            nn::os::UnlockMutex(&pThisObj->m_MutexLock);
         }
         break;

    case UPNP_DISCOVERY_SEARCH_TIMEOUT:
        /* Nothing to do here... */
        break;

    case UPNP_DISCOVERY_ADVERTISEMENT_BYEBYE:
         /* Nothing to do here... */
         break;

    /* SOAP Stuff */
    case UPNP_CONTROL_ACTION_COMPLETE:
         {
             struct Upnp_Action_Complete* pUpnpEvent = reinterpret_cast<struct Upnp_Action_Complete*>(pEvent);
             int                          ret = 0;

             nn::os::LockMutex(&pThisObj->m_MutexLock);

             do
             {
                 if (pUpnpEvent->ErrCode != UPNP_E_SUCCESS)
                 {
                     NN_LOG("Error in Action Complete Callback -- [%d]\n", pUpnpEvent->ErrCode);
                     pThisObj->AddMessage("[UPnP SOAP Action Callback] Error Returned - Code [%d]\n", pUpnpEvent->ErrCode);
                     break;
                 }

                 NN_LOG("Action Complete Callback Successful!\n" );

                 ret = pThisObj->HandleAction(pUpnpEvent);
                 if (ret != 0)
                 {
                     NN_LOG( "Failed calling HandleAction()\n" );
                     break;
                 }

                 // All done
                 break;

            } while (NN_STATIC_CONDITION(false));

            nn::os::UnlockMutex(&pThisObj->m_MutexLock);
         }
         break;


    /* Not an Event we process */
    default:
          NN_LOG( "We don't handle event: %d\n", eventType);
          break;
    }

    return 0;

}  // NOLINT(impl/function_size)

int UpnpDeviceGamePlayer::AddMessage(const char* pFormat, ...) NN_NOEXCEPT
{
    va_list       ArgList;
    std::string   prevMessage;
    std::string   saveMessage;
    int           ret = 0;
    int           idx = 0;
    int           bufferLength = -1;
    char          buffer[1024];

    nn::os::LockMutex(&m_MessagesLock);

    do
    {
        // Initialize the message buffer
        memset(buffer,0,sizeof(buffer));

        // If No format to the Message..
        if (pFormat == nullptr)
        {
            ret = -1;
            break;
        }

        // Convert VARARG style message
        va_start(ArgList, pFormat);

        bufferLength = vsnprintf(buffer, sizeof(buffer), pFormat, ArgList);
        if (bufferLength < 1)
        {
            va_end(ArgList);
            ret = -1;
            break;
        }

        // End the ArgList
        va_end(ArgList);

        // If Internal Message Vector is empty..
        if (m_Messages.size() != MaxMessages)
        {
            m_Messages.clear();
            for(idx = 0; idx < MaxMessages; idx++)
            {
                m_Messages.push_back("");
            }
        }

        // Message Buffer changed!
        m_MessagesChanged = true;

        // New Message is the previous Message. Scroll down remaining Messages
        prevMessage.assign(buffer);
        for( m_MessagesIter = m_Messages.begin();
             m_MessagesIter != m_Messages.end();
             m_MessagesIter++)
        {
            saveMessage = *m_MessagesIter;
            *m_MessagesIter = prevMessage;
            prevMessage = saveMessage;
        }

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    nn::os::UnlockMutex(&m_MessagesLock);

    return(ret);
}

int UpnpDeviceGamePlayer::GetMessages(std::vector<std::string> & outMessages) NN_NOEXCEPT
{
    std::vector<std::string>::iterator outIter;
    int                                ret = 0;

    nn::os::LockMutex(&m_MessagesLock);

    do
    {
        // Unset Messages Changed flag
        m_MessagesChanged = false;

        // If Outbound Message Vector is the wrong size..
        if (outMessages.size() != MaxMessages)
        {
            outMessages.clear();
            for(int idx = 0; idx < MaxMessages; idx++)
            {
                outMessages.push_back("");
            }
        }

        // If No Messages yet..
        if (m_Messages.size() != MaxMessages)
        {
            return(0);
        }

        // Copy Messages to Out Message Vector
        for( m_MessagesIter = m_Messages.begin(), outIter = outMessages.begin();
             m_MessagesIter != m_Messages.end() && outIter != outMessages.end();
             m_MessagesIter++, outIter++)
        {
            *outIter = *m_MessagesIter;
        }

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    nn::os::UnlockMutex(&m_MessagesLock);

    return(ret);
}

}}  // Namespace nns / libupnp
