﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn.h>
#include <nn/os.h>
#include "Task.h"

namespace nns { namespace ldn
{
    /**
     * @brief       LDN ライブラリを初期化します。
     */
    class LdnInitializeTask : public Task<nn::Result>
    {
    public:

        /**
         * @brief       コントラクタです。
         */
        LdnInitializeTask() NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~LdnInitializeTask() NN_NOEXCEPT;

    private:

        virtual TaskResult RunImpl(nn::Result* pOutResult) NN_NOEXCEPT NN_OVERRIDE;
    };

    /**
     * @brief       LDN ライブラリを終了します。
     * @details     LDN ライブラリが初期化されていない場合には何もしません。
     */
    class LdnFinalizeTask : public Task<nn::Result>
    {
    public:

        /**
         * @brief       コントラクタです。
         */
        LdnFinalizeTask() NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~LdnFinalizeTask() NN_NOEXCEPT;

    private:

        virtual TaskResult RunImpl(nn::Result* pOutResult) NN_NOEXCEPT NN_OVERRIDE;
    };

    /**
     * @brief       アクセスポイントとしての動作を開始します。
     * @details     既にアクセスポイントとして起動している場合には何もしません。
     */
    class LdnOpenAccessPointTask : public Task<nn::Result>
    {
    public:

        /**
         * @brief       コントラクタです。
         */
        LdnOpenAccessPointTask() NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~LdnOpenAccessPointTask() NN_NOEXCEPT;

    private:

        virtual TaskResult RunImpl(nn::Result* pOutResult) NN_NOEXCEPT NN_OVERRIDE;
    };

    /**
     * @brief       ステーションとしての動作を開始します。
     * @details     既にステーションとして起動している場合には何もしません。
     */
    class LdnOpenStationTask : public Task<nn::Result>
    {
    public:

        /**
         * @brief       コントラクタです。
         */
        LdnOpenStationTask() NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~LdnOpenStationTask() NN_NOEXCEPT;

    private:

        virtual TaskResult RunImpl(nn::Result* pOutResult) NN_NOEXCEPT NN_OVERRIDE;
    };

    /**
     * @brief       アクセスポイントとしての動作を終了します。
     * @details     既にアクセスポイントとして起動していない場合には何もしません。
     */
    class LdnCloseAccessPointTask : public Task<nn::Result>
    {
    public:

        /**
         * @brief       コントラクタです。
         */
        LdnCloseAccessPointTask() NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~LdnCloseAccessPointTask() NN_NOEXCEPT;

    private:

        virtual TaskResult RunImpl(nn::Result* pOutResult) NN_NOEXCEPT NN_OVERRIDE;
    };

    /**
     * @brief       ステーションとしての動作を終了します。
     * @details     既にステーションとして起動していない場合には何もしません。
     */
    class LdnCloseStationTask : public Task<nn::Result>
    {
    public:

        /**
         * @brief       コントラクタです。
         */
        LdnCloseStationTask() NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~LdnCloseStationTask() NN_NOEXCEPT;

    private:

        virtual TaskResult RunImpl(nn::Result* pOutResult) NN_NOEXCEPT NN_OVERRIDE;
    };

    /**
     * @brief       ローカル・ネットワークをスキャンします。
     * @details     また、このタスクはスタックに置くにはサイズが大きすぎるので注意してください。
     */
    class LdnScanTask : public Task<nn::Result>
    {
    public:

        /**
         * @brief       コントラクタです。
         */
        LdnScanTask() NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~LdnScanTask() NN_NOEXCEPT;

        /**
         * @brief       スキャン結果を絞り込むフィルタを設定します。
         * @param[in]   filter          スキャン結果を絞り込むフィルタです。
         * @details     デフォルトではスキャン結果を絞り込みません。
         */
        void SetScanFilter(const nn::ldn::ScanFilter& filter) NN_NOEXCEPT;

        /**
         * @brief       スキャン対象の無線チャンネルを指定します。
         * @param[in]   channel         無線チャンネルです。
         * @details     デフォルトでは nn::ldn::AutoChannel です。
         */
        void SetChannel(int channel) NN_NOEXCEPT;

        /**
         * @brief       スキャンで発見したネットワークの数を取得します。
         * @return      スキャンで発見したネットワークの数です。
         */
        int GetNetworkCount() const NN_NOEXCEPT;

        /**
         * @brief       ネットワーク情報を取得します。
         * @param[in]   index           ネットワークのインデックスです。
         */
        const nn::ldn::NetworkInfo& GetNetwork(int index) const NN_NOEXCEPT;

    private:

        virtual TaskResult RunImpl(nn::Result* pOutResult) NN_NOEXCEPT NN_OVERRIDE;

        nn::ldn::NetworkInfo m_Networks[nn::ldn::ScanResultCountMax];
        nn::ldn::ScanFilter m_ScanFilter;
        int32_t m_ScanResultCount;
        int32_t m_Channel;
    };

    /**
     * @brief       ローカル・ネットワークに接続します。
     * @details     このタスクはスタックに置くにはサイズが大きすぎるので注意してください。
     */
    class LdnConnectTask : public Task<nn::Result>
    {
    public:

        /**
         * @brief       コントラクタです。
         */
        LdnConnectTask() NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~LdnConnectTask() NN_NOEXCEPT;

        /**
         * @brief       ローカル通信バージョンを設定します。
         * @param[in]   version         ローカル通信バージョンです。
         * @details     デフォルトでは 0 が設定されています。
         */
        void SetLocalCommunicationVersion(int version) NN_NOEXCEPT;

        /**
         * @brief       接続先のネットワーク情報を設定します。
         * @param[in]   network         接続先のネットワーク情報です。
         */
        void SetNetworkInfo(const nn::ldn::NetworkInfo& network) NN_NOEXCEPT;

        /**
         * @brief       セキュリティ情報を設定します。
         * @param[in]   security        セキュリティ情報です。
         */
        void SetSecurityConfig(const nn::ldn::SecurityConfig& security) NN_NOEXCEPT;

        /**
         * @brief       ユーザ情報を設定します。
         * @param[in]   user            ユーザ情報です。
         */
        void SetUserConfig(const nn::ldn::UserConfig& user) NN_NOEXCEPT;

        /**
         * @brief       接続試行のリトライ回数を設定します。
         * @param[in]   retryCount      接続試行のリトライ回数です。
         * @details     デフォルトではリトライせず、1 度だけ接続を試行します。
         */
        void SetRetryCount(int retryCount) NN_NOEXCEPT;

        /**
         * @brief       IPv4 アドレスを取得します。
         * @return      IPv4 アドレスです。
         */
        nn::ldn::Ipv4Address GetIpv4Address() const NN_NOEXCEPT;

        /**
         * @brief       サブネットマスクを取得します。
         * @return      サブネットマスクです。
         */
        nn::ldn::SubnetMask GetSubnetMask() const NN_NOEXCEPT;

    private:

        virtual TaskResult RunImpl(nn::Result* pOutResult) NN_NOEXCEPT NN_OVERRIDE;
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE;

        nn::ldn::NetworkInfo m_NetworkInfo;
        nn::ldn::SecurityConfig m_SecurityConfig;
        nn::ldn::UserConfig m_UserConfig;
        nn::ldn::Ipv4Address m_Ipv4Address;
        nn::ldn::SubnetMask m_SubnetMask;
        int m_LocalCommunicationVersion;
        int m_RetryCount;
        bool m_IsCanceled;
    };

    /**
     * @brief       ローカル・ネットワークをスキャンします。
     * @details     また、このタスクはスタックに置くにはサイズが大きすぎるので注意してください。
     */
    class LdnCreateNetworkTask : public Task<nn::Result>
    {
    public:

        /**
         * @brief       コントラクタです。
         */
        LdnCreateNetworkTask() NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~LdnCreateNetworkTask() NN_NOEXCEPT;

        /**
         * @brief       ネットワーク情報を設定します。
         * @param[in]   network         ネットワーク情報です。
         */
        void SetNetworkConfig(const nn::ldn::NetworkConfig& network) NN_NOEXCEPT;

        /**
         * @brief       セキュリティ情報を設定します。
         * @param[in]   security        セキュリティ情報です。
         */
        void SetSecurityConfig(const nn::ldn::SecurityConfig& security) NN_NOEXCEPT;

        /**
         * @brief       ユーザ情報を設定します。
         * @param[in]   user            ユーザ情報です。
         */
        void SetUserConfig(const nn::ldn::UserConfig& user) NN_NOEXCEPT;

        /**
         * @brief       IPv4 アドレスを取得します。
         * @return      IPv4 アドレスです。
         */
        nn::ldn::Ipv4Address GetIpv4Address() const NN_NOEXCEPT;

        /**
         * @brief       サブネットマスクを取得します。
         * @return      サブネットマスクです。
         */
        nn::ldn::SubnetMask GetSubnetMask() const NN_NOEXCEPT;

    private:

        virtual TaskResult RunImpl(nn::Result* pOutResult) NN_NOEXCEPT NN_OVERRIDE;

        nn::ldn::NetworkConfig m_NetworkConfig;
        nn::ldn::SecurityConfig m_SecurityConfig;
        nn::ldn::UserConfig m_UserConfig;
        nn::ldn::Ipv4Address m_Ipv4Address;
        nn::ldn::SubnetMask m_SubnetMask;
    };

    /**
     * @brief       ステーションを切断します。
     */
    class LdnRejectTask : public Task<nn::Result>
    {
    public:

        /**
         * @brief       コントラクタです。
         */
        LdnRejectTask() NN_NOEXCEPT;

        /**
         * @brief       コントラクタです。
         * @param[in]   ipv4Address     切断対象のステーションの IPv4 アドレスです。
         */
        explicit LdnRejectTask(nn::ldn::Ipv4Address ipv4Address) NN_NOEXCEPT;

        /**
         * @brief       コントラクタです。
         * @param[in]   node            切断対象のステーションです。
         */
        explicit LdnRejectTask(const nn::ldn::NodeInfo& node) NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        virtual ~LdnRejectTask() NN_NOEXCEPT;

        /**
         * @brief       切断するステーションを設定します。
         * @param[in]   ipv4Address     切断対象のステーションの IPv4 アドレスです。
         */
        void SetStation(nn::ldn::Ipv4Address ipv4Address) NN_NOEXCEPT;

        /**
         * @brief       切断するステーションを設定します。
         * @param[in]   node            切断対象のステーションです。
         */
        void SetStation(const nn::ldn::NodeInfo& node) NN_NOEXCEPT;

    private:

        virtual TaskResult RunImpl(nn::Result* pOutResult) NN_NOEXCEPT NN_OVERRIDE;

        nn::ldn::Ipv4Address m_Ipv4Address;
    };

}} // namespace nns::ldn
