﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "HandAnalysisModeState.h"

#include <algorithm>
#include <sstream>

#include <nn/nn_Assert.h>

HandAnalysisModeState::HandAnalysisModeState(IrSensorMode* pNextProcessor, int* pMenuSelection, nn::irsensor::IrCameraHandle irCameraHandle, GraphicsSystem* pGraphicsSystem)
    : IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
    , m_NextHandAnalysisConfig()
    , m_CurrentHandAnalysisConfig()
    , m_HandAnalysisSilhouetteState()
    , m_HandAnalysisImageState()
    , m_PolygonProperties()
    , m_pGraphicsSystem(pGraphicsSystem)
{
    {
        m_NextHandAnalysisConfig.mode = nn::irsensor::HandAnalysisMode_SilhouetteAndImage;
        m_CurrentHandAnalysisConfig = m_NextHandAnalysisConfig;

        CreateTexture(pGraphicsSystem, nn::irsensor::IrHandAnalysisImageWidth, nn::irsensor::IrHandAnalysisImageHeight, nn::gfx::ImageFormat_R16_Unorm, &m_Texture);
        CreateSampler(pGraphicsSystem, &m_Sampler);

        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_Read);
        info.SetSize(ImageSize);
        pGraphicsSystem->AllocateBuffer(&m_Buffer, &info);
    }

    nn::irsensor::HandAnalysisConfig* pHandAnalysisConfig = &m_NextHandAnalysisConfig;
    m_ReadWriteMenu.emplace_back("Mode",
        [pHandAnalysisConfig](std::stringstream& sstr) {
            if (pHandAnalysisConfig->mode == nn::irsensor::HandAnalysisMode_Silhouette)
                sstr << "Silhouette";
            else if (pHandAnalysisConfig->mode == nn::irsensor::HandAnalysisMode_Image)
                sstr << "Image";
            else if (pHandAnalysisConfig->mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage)
                sstr << "Silhouette and Image";
            else if (pHandAnalysisConfig->mode == nn::irsensor::HandAnalysisMode_SilhouetteOnly)
                sstr << "Silhouette only";
        },
        [pHandAnalysisConfig](int8_t delta) {
            int8_t mode = static_cast<int8_t>(pHandAnalysisConfig->mode) + delta;
            mode = std::min(std::max(mode, static_cast<int8_t>(nn::irsensor::HandAnalysisMode_Silhouette)), static_cast<int8_t>(nn::irsensor::HandAnalysisMode_SilhouetteOnly));
            pHandAnalysisConfig->mode = static_cast<nn::irsensor::HandAnalysisMode>(mode);
        }
    );

    AddCommonReadOnlyMenu(&m_ReadOnlyMenu, &m_HandAnalysisImageState.samplingNumber, &m_HandAnalysisImageState.ambientNoiseLevel );
    AddStatisticsMenu(&m_ReadOnlyMenu, &m_Statistics.m_PacketDropPercentage);
    const PolygonProperties* pPolygonProperties = &m_PolygonProperties;
    nn::irsensor::HandAnalysisSilhouetteState* pHandAnalysisSilhouetteState = &m_HandAnalysisSilhouetteState;
    m_ReadOnlyMenu.emplace_back("Shape Area",
        [pPolygonProperties](std::stringstream& sstr) {
            sstr << pPolygonProperties->signedArea;
        },
        [](int8_t delta) { NN_UNUSED(delta); }
    );
    m_ReadOnlyMenu.emplace_back("Shape",
        [pHandAnalysisSilhouetteState](std::stringstream& sstr) {
            if (pHandAnalysisSilhouetteState->handCount > 0)
            {
                const nn::irsensor::Hand& hand = pHandAnalysisSilhouetteState->hands[0];
                int numFinger = 0;

                for (int i = 0; i < nn::irsensor::HandFinger_Count; ++i)
                {
                    if (hand.fingers[i].isValid)
                    {
                        numFinger++;
                    }
                }

                switch (numFinger)
                {
                case 0:
                case 1:
                    sstr << "Rock";
                    break;
                case 2:
                case 3:
                    sstr << "Scissor";
                    break;
                case 5:
                case 6:
                    sstr << "Paper";
                    break;
                default:
                    sstr << "Unknown";
                    break;
                }
            }
            else
            {
                sstr << "Unknown";
            }
        },
        [](int8_t delta) { NN_UNUSED(delta); }
    );
}

HandAnalysisModeState::~HandAnalysisModeState()
{
    m_pGraphicsSystem->FreeBuffer(&m_Buffer);

    m_pGraphicsSystem->FreeTexture(&m_Texture.texture);
    m_Texture.view.Finalize(&m_pGraphicsSystem->GetDevice());

    m_Sampler.sampler.Finalize(&m_pGraphicsSystem->GetDevice());
}

void HandAnalysisModeState::Start()
{
    nn::irsensor::RunHandAnalysis(m_IrCameraHandle, m_NextHandAnalysisConfig);
    m_CurrentHandAnalysisConfig = m_NextHandAnalysisConfig;

    m_HandAnalysisSilhouetteState.handCount = 0;
    m_PolygonProperties.signedArea = 0.0f;
}

void HandAnalysisModeState::Update()
{
    if ((m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_Silhouette) || (m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage) || (m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_SilhouetteOnly))
    {
        int count = 0;
        nn::irsensor::GetHandAnalysisSilhouetteState(&m_HandAnalysisSilhouetteState, &count, 1, 0, m_IrCameraHandle);

        if(count > 0)
        {
            m_Statistics.Update(&m_HandAnalysisSilhouetteState.samplingNumber, IrSensorModeStatistics::ExpectedHandAnalysisFramerate);
        }
    }
    if ((m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_Image) || (m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))
    {
        int count = 0;
        nn::irsensor::GetHandAnalysisImageState(&m_HandAnalysisImageState, &count, 1, 0, m_IrCameraHandle);

        if (count > 0)
        {
            m_Statistics.Update(&m_HandAnalysisImageState.samplingNumber, IrSensorModeStatistics::ExpectedHandAnalysisFramerate);

            uint16_t* pSource = static_cast<uint16_t*>(m_HandAnalysisImageState.image);
            uint16_t* pDest = m_Buffer.Map< uint16_t >();
            memcpy(pDest, pSource, ImageSize);
            for (size_t i = 0; i < ImageSize; i++)
            {
                const float max = static_cast<float>(0xffff);
                const float value = static_cast<float>(pDest[i]) / max;
                const float scaledValue = value * 100.0f > 1.0f ? 1.0f : value * 100.0f;

                pDest[i] = static_cast<uint16_t>(scaledValue * max);
            }
            m_Buffer.FlushMappedRange(0, ImageSize);
            m_Buffer.Unmap();
        }
    }
}

void HandAnalysisModeState::RenderImageState(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer,
    const int screenIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    const nn::util::Uint8x4 White ={ { 255, 255, 255, 255 } };

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer,
        nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType corner;
    nn::util::Vector3fType size;

    const float xStart = -0.2f, xEnd = -0.8f;
    float yStart, yEnd;
    if (screenIndex == 0)
        yStart = 0.9f, yEnd = 0.1f;
    else
        yStart = -0.1f, yEnd = -0.9f;

    nn::util::VectorSet(&corner, xStart, yStart, 0.0f);
    nn::util::VectorSet(&size, xEnd - xStart, yEnd - yStart, 0.0f);

    pPrimitiveRenderer->SetColor(White);
    pPrimitiveRenderer->DrawQuad(
        pCommandBuffer,
        corner,
        size,
        m_Texture.descriptor,
        m_Sampler.descriptor);
}

namespace
{
void WriteHandAnalysisState(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::irsensor::HandAnalysisSilhouetteState* pSilhouetteState,
    PolygonProperties* pPolygonProperties,
    const int screenIndex) NN_NOEXCEPT
{
    const nn::util::Uint8x4 Red ={ { 255, 0, 0, 255 } };
    const nn::util::Uint8x4 Green ={ { 0, 255, 0, 255 } };
    const nn::util::Uint8x4 Blue ={ { 0, 0, 255, 255 } };
    const nn::util::Uint8x4 Yellow ={ { 255, 255, 0, 255 } };
    const nn::util::Uint8x4 Purple = { { 255, 0, 255, 255 } };

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;
    pPrimitiveRenderer->SetLineWidth(10.f);

    const float xStart = -0.2f, xEnd = -0.8f;
    float yStart, yEnd;
    if (screenIndex == 0)
        yStart = 0.1f, yEnd = 0.9f;
    else
        yStart = -0.9f, yEnd = -0.1f;

    // NOTE: Coordinate are normalized according to Y axis.
    // Because the image ratio is 4/3, X points needs to be scaled
    // by 3/4 to fit in image space.
    const float xscale = 0.75f;
    const float yscale = 1.0f;

    // シルエットを描画
    for (int shapeIndex = 0; shapeIndex < pSilhouetteState->shapeCount; shapeIndex++)
    {
        std::vector<nn::util::Float2> silhouette;
        const nn::irsensor::Shape& shape = pSilhouetteState->shapes[shapeIndex];

        nn::util::Float2 p1, p2;
        p2 = pSilhouetteState->points[shape.firstPointIndex];
        pPrimitiveRenderer->SetColor(Green);
        for (auto i = 0; i < shape.pointCount - 1; ++i)
        {
            p1 = p2;
            p2 = pSilhouetteState->points[(shape.firstPointIndex + i + 1) % NN_ARRAY_SIZE(pSilhouetteState->points)];

            nn::util::VectorSet(&begin,
                xStart + (xEnd - xStart) * (p1.x * xscale + 1.0f) / 2.0f,
                yStart + (yEnd - yStart) * (p1.y * yscale + 1.0f) / 2.0f, 0.0f);
            nn::util::VectorSet(&end,
                xStart + (xEnd - xStart) * (p2.x * xscale + 1.0f) / 2.0f,
                yStart + (yEnd - yStart) * (p2.y * yscale + 1.0f) / 2.0f, 0.0f);

            pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);

            silhouette.push_back(p1);
        }
        silhouette.push_back(p2);

        std::vector<size_t> convexHullIndices;
        // Convex Hull を計算する
        ExtractConvexHull(convexHullIndices, silhouette);
        if (convexHullIndices.size() > 0)
        {
            p2 = pSilhouetteState->points[shape.firstPointIndex + convexHullIndices[0]];
            pPrimitiveRenderer->SetColor(Yellow);
            for (size_t i = 0; i < convexHullIndices.size() - 1; ++i)
            {
                p1 = p2;
                p2 = pSilhouetteState->points[shape.firstPointIndex + convexHullIndices[i + 1]];

                nn::util::VectorSet(&begin,
                    xStart + (xEnd - xStart) * (p1.x * xscale + 1.0f) / 2.0f,
                    yStart + (yEnd - yStart) * (p1.y * yscale + 1.0f) / 2.0f, 0.0f);
                nn::util::VectorSet(&end,
                    xStart + (xEnd - xStart) * (p2.x * xscale + 1.0f) / 2.0f,
                    yStart + (yEnd - yStart) * (p2.y * yscale + 1.0f) / 2.0f, 0.0f);

                pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
            }
        }

        ComputePolygonProperties(pPolygonProperties, &silhouette[0], silhouette.size());
    }

    // 指先を描画
    for (int handIndex = 0; handIndex < pSilhouetteState->handCount; handIndex++)
    {
        const nn::irsensor::Hand& hand = pSilhouetteState->hands[handIndex];

        // 検出物の中央を描画
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * (hand.palm.center.x * xscale + 1.0f) / 2.0f,
            yStart + (yEnd - yStart) * (hand.palm.center.y * yscale + 1.0f) / 2.0f, 0.0f);
        pPrimitiveRenderer->SetColor(Green);
        pPrimitiveRenderer->DrawSphere(
            pCommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface_Solid,
            nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
            begin,
            8.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));

        for (int i = 0; i < nn::irsensor::HandFinger_Count; ++i)
        {
            if (hand.fingers[i].isValid)
            {
                nn::util::VectorSet(&begin,
                    xStart + (xEnd - xStart) * (hand.fingers[i].tip.x * xscale + 1.0f) / 2.0f,
                    yStart + (yEnd - yStart) * (hand.fingers[i].tip.y * yscale + 1.0f) / 2.0f, 0.0f);
                pPrimitiveRenderer->SetColor(Red);
                pPrimitiveRenderer->DrawSphere(
                    pCommandBuffer,
                    nns::gfx::PrimitiveRenderer::Surface_Solid,
                    nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
                    begin,
                    8.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));

                nn::util::VectorSet(&begin,
                    xStart + (xEnd - xStart) * (hand.fingers[i].root.x * xscale + 1.0f) / 2.0f,
                    yStart + (yEnd - yStart) * (hand.fingers[i].root.y * yscale + 1.0f) / 2.0f, 0.0f);
                pPrimitiveRenderer->SetColor(Blue);
                pPrimitiveRenderer->DrawSphere(
                    pCommandBuffer,
                    nns::gfx::PrimitiveRenderer::Surface_Solid,
                    nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
                    begin,
                    8.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));
            }
        }

        const nn::irsensor::Palm& palm = pSilhouetteState->hands[handIndex].palm;

        // 手平を描画
        pPrimitiveRenderer->SetColor(Red);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * (palm.center.x * xscale + 1.0f) / 2.0f,
            yStart + (yEnd - yStart) * (palm.center.y * yscale + 1.0f) / 2.0f, 0.0f);
        pPrimitiveRenderer->DrawSphere(
            pCommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface_Solid,
            nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
            begin,
            8.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));

        // 手首を描画
        if (hand.arm.isValid)
        {
            pPrimitiveRenderer->SetColor(Purple);
            nn::util::VectorSet(&begin,
                xStart + (xEnd - xStart) * (hand.arm.wristPosition.x * xscale + 1.0f) / 2.0f,
                yStart + (yEnd - yStart) * (hand.arm.wristPosition.y * yscale + 1.0f) / 2.0f, 0.0f);
            pPrimitiveRenderer->DrawSphere(
                pCommandBuffer,
                nns::gfx::PrimitiveRenderer::Surface_Solid,
                nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
                begin,
                8.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));
        }
    }
}//NOLINT(readability/fn_size)

}

void HandAnalysisModeState::Render(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer, int index)
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    // 読み込んだカメラ画像 nn::gfx::buffer を、テクスチャにタイリングします
    if ((m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_Image) || (m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))
    {
        nn::gfx::TextureCopyRegion dstRegion;
        dstRegion.SetDefault();
        dstRegion.SetDepth(1);

        dstRegion.SetWidth(nn::irsensor::IrHandAnalysisImageWidth);
        dstRegion.SetHeight(nn::irsensor::IrHandAnalysisImageHeight);
        pCommandBuffer->CopyBufferToImage(&m_Texture.texture, dstRegion, &m_Buffer, 0);

        RenderImageState(pPrimitiveRenderer, pCommandBuffer, index);
    }
    if ((m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_Silhouette) || (m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_SilhouetteOnly) || (m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))
    {
        WriteHandAnalysisState(pPrimitiveRenderer, pCommandBuffer, &m_HandAnalysisSilhouetteState, &m_PolygonProperties, index);
    }
}
