﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "MarkerPlaneEstimation.h"

#include <algorithm>
#include <sstream>

#include <nn/nn_Assert.h>

const nn::irsensor::PointList MarkerPlaneEstimation::StaticReferenceList =
{
    7,
    {
        {{0.0f, 3.731f}, 1.0f},
        {{-17.491f, -14.175f}, 1.0f},
        {{17.491f, -14.175f}, 1.0f},
        {{-54.75f, 26.076f}, 1.0f},
        {{54.75f, 26.076f}, 1.0f},
        {{-54.75f, -31.032f}, 1.0f},
        {{54.75f, -31.032f}, 1.0f},
    },
};

namespace
{

void WriteMarkerPlaneState(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::irsensor::MarkerPlaneState* pMarkerPlaneState,
    const int screenIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pMarkerPlaneState);

    const nn::util::Uint8x4 White ={ { 255, 255, 255, 255 } };

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;

    const float xStart = -0.2f, xEnd = -0.8f;
    float yStart, yEnd;
    if (screenIndex == 0)
        yStart = 0.1f, yEnd = 0.9f;
    else
        yStart = -0.9f, yEnd = -0.1f;

    pPrimitiveRenderer->SetLineWidth(10.f);
    pPrimitiveRenderer->SetColor(White);

    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    nn::util::VectorSet(&end, xStart, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xEnd, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&end, xEnd, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);


     pPrimitiveRenderer->SetLineWidth(5.f);

    // 各面を描画
}

}

MarkerPlaneEstimation::MarkerPlaneEstimation(unsigned char* initMemory, size_t initMemorySize)
    : m_Handle()
    , m_pWorkBuffer(initMemory)
    , m_WorkBufferSize(initMemorySize)
    , m_State()
{
    nn::irsensor::MarkerPlaneConfig config = {};
    config.pointList = StaticReferenceList;
    m_Handle = nn::irsensor::MarkerPlaneEstimationInitialize(m_pWorkBuffer, m_WorkBufferSize, config);
}


MarkerPlaneEstimation::~MarkerPlaneEstimation()
{
    nn::irsensor::MarkerPlaneEstimationFinalize(m_Handle);
}

void MarkerPlaneEstimation::Update(const nn::irsensor::PointList& currentPosition)
{
    nn::irsensor::GetMarkerPlaneState(&m_State, m_Handle);
    nn::irsensor::UpdateMarkerPlaneEstimation(currentPosition, m_Handle);
}

void MarkerPlaneEstimation::Render(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer, int index)
{
    WriteMarkerPlaneState(pPrimitiveRenderer, pCommandBuffer, &m_State, index);
}

void MarkerPlaneEstimation::GetState(nn::irsensor::MarkerPlaneState** pOutState)
{
    *pOutState = &m_State;
}
