﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "IrSensorDemo.h"

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/hid/hid_ControllerSupport.h>

#include "AdaptiveClusteringModeState.h"


#include <sstream>
namespace
{

void WriteIrCameraStatus(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    nn::irsensor::IrCameraStatus irCameraStatus,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    const nn::util::Unorm8x4 White ={ { 255, 255, 255, 255 } };

    pTextWriter->SetScale(2.0f, 2.0f);
    pTextWriter->SetTextColor(White);

    pTextWriter->SetCursor(offsetX + 480, offsetY + 100);
    if (offsetY < 360.0f)
    {
        pTextWriter->Print("1st IrCamera Status");
    }
    else
    {
        pTextWriter->Print("2nd IrCamera Status");
    }

    pTextWriter->SetCursor(offsetX + 420, offsetY + 170);
    if (irCameraStatus == nn::irsensor::IrCameraStatus_Unsupported)
    {
        pTextWriter->Print("IrCameraStatus_Unsupported");
    }
    else if (irCameraStatus == nn::irsensor::IrCameraStatus_Unconnected)
    {
        pTextWriter->Print("IrCameraStatus_Unconnected");
    }

}


void DrawMenu(const std::vector<MenuItem>& menu,
    const float offsetX, const float offsetY,
    int selectIndex,
    nn::gfx::util::DebugFontTextWriter* pTextWriter)
{

    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4 White ={ { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 Red ={ { 255, 0, 0, 255 } };

    pTextWriter->SetScale(1.0f, 1.0f);

    for (size_t index = 0; index < menu.size(); index++)
    {
        pTextWriter->SetTextColor(((size_t) selectIndex == index) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + index * 20.0f);

        std::stringstream text;
        text << menu[index].GetName() << ": ";
        menu[index].Read(text);

        pTextWriter->Print(text.str().c_str());
    }
}

void WriteProcessorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const float offsetX,
    const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4 White ={ { 255, 255, 255, 255 } };

    pTextWriter->SetScale(1.0f, 1.0f);

    pTextWriter->SetTextColor(White);
    pTextWriter->SetCursor(offsetX + 330, offsetY + 240);
    pTextWriter->Print("Press X / B to change component");
    pTextWriter->SetCursor(offsetX + 330, offsetY + 260);
    pTextWriter->Print("Press Y / A to change value");
    pTextWriter->SetCursor(offsetX + 330, offsetY + 280);
    pTextWriter->Print("Press R to reflect user setting");
}

}

void CreateSampler(GraphicsSystem* pGraphicsSystem, Sampler* pSampler)
{
    nn::gfx::Sampler::InfoType samplerInfo;
    samplerInfo.SetDefault();
    samplerInfo.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
    samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
    samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
    samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
    pSampler->sampler.Initialize(&pGraphicsSystem->GetDevice(), samplerInfo);

    pGraphicsSystem->RegisterSamplerSlot(&pSampler->descriptor, pSampler->sampler);
}

void CreateTexture(GraphicsSystem* pGraphicsSystem, int width, int height, nn::gfx::ImageFormat format, Texture* pTexture)
{
    {
        nn::gfx::Texture::InfoType textureInfo;

        textureInfo.SetDefault();
        textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
        textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        textureInfo.SetMipCount(1);
        textureInfo.SetTileMode(nn::gfx::TileMode_Linear);
        textureInfo.SetWidth(width);
        textureInfo.SetHeight(height);
        textureInfo.SetImageFormat(format);

        pGraphicsSystem->AllocateTexture(&pTexture->texture, &textureInfo);
    }

    {
        nn::gfx::TextureView::InfoType viewInfo;
        viewInfo.SetDefault();
        viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        viewInfo.SetImageFormat(format);
        viewInfo.SetTexturePtr(&pTexture->texture);
        viewInfo.SetChannelMapping(nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red,
            nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red);

        pTexture->view.Initialize(&pGraphicsSystem->GetDevice(), viewInfo);
    }

    {
        pGraphicsSystem->RegisterTextureViewSlot(&pTexture->descriptor, pTexture->view);
    }
}

void IrSensorDemo::CheckFirmwareUpdate(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    const int FirmwareCheckTrialCountMax = 300;

    // FirmwareVersion のチェック
    int counter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        bool isUpdateNeeded = false;
        nn::Result result = nn::irsensor::CheckFirmwareUpdateNecessity(&isUpdateNeeded, handle);
        if (result.IsSuccess())
        {
            // 成功時は、ファームウェアの更新が必要かどうかのフラグが返ります。
            NN_LOG("    Controller firmware update checking succeed.\n");
            if (isUpdateNeeded)
            {
                // コントローラの更新が必要な場合は、更新アプレットを呼び出すことを推奨します。
                // 更新アプレットはブロック処理で、接続された全てのコントローラのアップデートを行うため、呼び出しには注意が必要です。
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
                // 選択 UI なしの強制アップデート版コンサポ呼び出し
                // 成功しても失敗しても 1 回だけ呼び出し。
                nn::hid::ControllerFirmwareUpdateArg arg;
                arg.enableForceUpdate = false;
                nn::hid::ShowControllerFirmwareUpdate(arg);
#elif defined( NN_BUILD_CONFIG_OS_WIN )
                NN_LOG("Invoke ControllerFirmwareUpdate is not suppported on Windows environment.\n");
#else
                #error "unsupported os"
#endif
            }
            break;
        }
        else if (::nn::irsensor::ResultIrsensorUnavailable::Includes(result))
        {
            // コントローラが使用不可状態の場合は、スキップする。
            NN_LOG("    Controller is not available.\n");
            break;
        }
        else if (::nn::irsensor::ResultIrsensorFirmwareCheckIncompleted::Includes(result))
        {
            // FirmwareVersion のチェック中、もしくはアプレット呼び出し中なのでリトライする
            counter++;
            if (counter > FirmwareCheckTrialCountMax)
            {
                // チェック中の状態がしばらく続いた場合は、タイムアウトすることを推奨します。
                NN_LOG("    Firmware update checking is timeout.\n");
                break;
            }
            else
            {
                NN_LOG("    Firmware update checking is running.\n");
            }
        }
        else
        {
            NN_ABORT("    Firmware update checking unexpected error\n");
        }
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
    }
}

void IrSensorDemo::Initialize(nn::hid::NpadIdType id, GraphicsSystem* pGraphicsSystem, void* pImageTransferWorkeMemory)
{
    NN_UNUSED(pGraphicsSystem);
    NN_UNUSED(pImageTransferWorkeMemory);
    m_NpadId = id;
    nn::hid::SetNpadJoyAssignmentModeSingle(m_NpadId);

    //IR カメラのハンドルの取得
    m_IrCameraHandle = nn::irsensor::GetIrCameraHandle(m_NpadId);

    //IR カメラの初期化
    nn::irsensor::Initialize(m_IrCameraHandle);
    NN_LOG("NpadPlayerNumber(%d)\n", m_NpadId);

    // FirmwareUpdate のチェック
    CheckFirmwareUpdate(m_IrCameraHandle);

    // IrSensor を Moment として初期化
    m_ModeStates[IrSensorMode_AdaptiveClustering] = new AdaptiveClusteringModeState(&m_NextProcessor, &m_MenuSelection, m_IrCameraHandle);

    m_CurrentProcessor = m_NextProcessor = IrSensorMode_AdaptiveClustering;
    m_ModeStates[m_CurrentProcessor]->Start();
}

void IrSensorDemo::Finalize()
{
    m_ModeStates[m_CurrentProcessor]->Stop();

    delete m_ModeStates[IrSensorMode_AdaptiveClustering];

    nn::irsensor::Finalize(m_IrCameraHandle);
}

bool IrSensorDemo::Run(int i, GraphicsSystem* pGraphicsSystem, nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
{
    //IR カメラの状態の取得

    const nn::irsensor::IrCameraStatus irCameraStatus = nn::irsensor::GetIrCameraStatus(m_IrCameraHandle);

    if (nn::irsensor::IrCameraStatus_Available == irCameraStatus)
    {
        if (!ProcessGamePad())
        {
            return false;
        }

        m_ModeStates[m_CurrentProcessor]->Update();

        {
            auto rwMenu = m_ModeStates[m_NextProcessor]->GetReadWriteMenu();
            auto rMenu  = m_ModeStates[m_NextProcessor]->GetReadOnlyMenu();

            int selectIndex = GetMenuIndex();

            WriteProcessorState(pTextWriter, 640, 15 + 360.0f * i);

            DrawMenu(rwMenu, 700.0f, 50.f + 360.0f * i,
                selectIndex, pTextWriter);
            DrawMenu(rMenu, 970.f, 50.f + 360.f * i,
                -1, pTextWriter);

        }

        m_ModeStates[m_CurrentProcessor]->Render(&pGraphicsSystem->GetPrimitiveRenderer(), &pGraphicsSystem->GetCommandBuffer(), i);
    }
    else
    {
        WriteIrCameraStatus(pTextWriter, irCameraStatus, 0, 360.0f * i);
    }

    return true;
}


bool IrSensorDemo::ProcessGamePad() NN_NOEXCEPT
{
    nn::hid::NpadJoyRightState currentState;
    nn::hid::NpadButtonSet buttonDownSet;

    // 押されたボタンを検出
    nn::hid::GetNpadState(&currentState, m_NpadId);
    buttonDownSet = currentState.buttons & ~m_PreviousPadState.buttons;
    m_PreviousPadState = currentState;

    // Menu Item Selection
    if ((buttonDownSet & nn::hid::NpadButton::B::Mask).IsAnyOn())
    {
        m_MenuSelection += 1;
    };

    if ((buttonDownSet & nn::hid::NpadButton::X::Mask).IsAnyOn())
    {
        m_MenuSelection -= 1;
    };

    // Menu Action
    if ((buttonDownSet & nn::hid::NpadButton::Y::Mask).IsAnyOn() || (buttonDownSet & nn::hid::NpadButton::A::Mask).IsAnyOn())
    {
        int8_t delta = ((buttonDownSet & nn::hid::NpadButton::A::Mask).IsAnyOn()) ? 1 : -1;

        const std::vector<MenuItem> menu = this->m_ModeStates[m_NextProcessor]->GetReadWriteMenu();
        const int menuIndex = GetMenuIndex();
        if (menuIndex >= 0)
        {
            menu[menuIndex].Increment(delta);
        }
    }

    // イベントトリガー
    if ((buttonDownSet & nn::hid::NpadButton::R::Mask).IsAnyOn())
    {
        m_ModeStates[m_CurrentProcessor]->Stop();
        m_ModeStates[m_NextProcessor]->Start();
        m_CurrentProcessor = m_NextProcessor;
    }

    if ((buttonDownSet & nn::hid::NpadButton::Plus::Mask).IsAnyOn() || (buttonDownSet & nn::hid::NpadButton::Minus::Mask).IsAnyOn())
    {
        return false;
    }

    return true;

}//NOLINT(readability/fn_size)

int IrSensorDemo::GetMenuIndex() const
{
    const std::vector<MenuItem> menu = this->m_ModeStates[m_NextProcessor]->GetReadWriteMenu();
    int size = static_cast<int>(menu.size());
    int menuIndex = ((m_MenuSelection % size) + size) % size;

    NN_ASSERT(menuIndex >= 0);
    NN_ASSERT(menuIndex < size);

    return menuIndex;
}
