﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{InitStartup.cpp,PageSampleInitStartup}
 *
 * @brief
 *  nninitStartup() 関数を使ったサンプルプログラムその 1
 */

/**
 * @page PageSampleInitStartup nninitStartup
 * @tableofcontents
 *
 * @brief
 *  nninitStartup() 関数のサンプルプログラムその 1 の解説です。
 *
 * @section PageSampleInitStartup_SectionBrief 概要
 *  ここでは、アプリケーションが独自の nninitStartup() 関数を定義する場合の
 *  サンプルプログラムについて説明します。
 *  nninitStartup() のリファレンスもあわせて参照して下さい。
 *
 * @section PageSampleInitStartup_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/InitStartup Samples/Sources/Applications/InitStartup @endlink 以下にあります。
 *
 * @section PageSampleInitStartup_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleInitStartup_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleInitStartup_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleInitStartup_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleInitStartup_SectionDetail 解説
 *
 * @subsection PageSampleInitStartup_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  InitStartup.cpp
 *  @includelineno InitStartup.cpp
 *
 * @subsection PageSampleInitStartup_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nninitStartup() にてメモリヒープを確保、malloc() のメモリ設定を実施
 *  - StaticObjectClass クラスを静的オブジェクトとして配置し、コンストラクタで malloc() を発行
 *  - nnMain() で上記 malloc() で獲得したメモリのアドレスをログ出力して終了
 *
 *  重要なポイントは、 nninitStartup()、静的オブジェクトのコンストラクタ、
 *  nnMain() これら 3 つの起動順序についてです。
 *
 *  まず最初に nninitStartup() が呼ばれ、ここでメモリヒープなどの初期化を
 *  行ないます。さらに nn::init::InitializeAllocator() を発行することで、
 *  SDK がデフォルトで用意している malloc() や new で使用されるメモリ領域を
 *  設定します。 nninitStartup() の中から利用できる機能については、
 *  @ref PageNotificationInitAvailableApiListInStartup を参照してください。
 *
 *  次に静的オブジェクトのコンストラクタが起動します。このサンプルでは、
 *  StaticObjectClass のインスタンスをグローバルに配置しています。
 *  コンストラクタの中で malloc() が呼べることを示しています。
 *
 *  最後に nnMain() が起動します。先程のコンストラクタの中で malloc() で
 *  獲得したアドレスをログ出力して終了しています。
 *
 *  最終的なログ表示を以下に示します。
 *
 *  @verbinclude  InitStartup_OutputExample.txt
 *
 *  ここでは SDK がデフォルトで用意している malloc() を利用するサンプルを
 *  示しましたが、 malloc() 自身を独自で定義したい場合には、
 *  @ref PageSampleInitStartupWithMalloc を参照して下さい。
 *
 *  なお、 nninitStartup() の定義を行なわなかった場合は、SDK がデフォルトで
 *  定義している nninitStartup() 関数が呼ばれます。この中では、
 *  開発の初期段階で容易に malloc() などを利用できるようにするために
 *  nn::init::InitializeAllocator() を使ってアロケータの初期化を
 *  行なっています。詳細はそちらのリファレンスを参照して下さい。
 *
 */

#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------
//  スタートアップ関数
//-----------------------------------------------------------------------------

extern "C" void nninitStartup()
{
    NN_LOG("step1: A nninitStartup() function is invoked.\n\n");

    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );
    NN_ASSERT(result.IsSuccess(), "Cannot set memory heap size.");

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address;
    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    NN_ASSERT( result.IsSuccess(), "Cannot allocate memory block." );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

//-----------------------------------------------------------------------------
//  静的オブジェクト
//-----------------------------------------------------------------------------

class StaticObjectClass
{
public:
    // コンストラクタ
    StaticObjectClass()
    {
        NN_LOG("step2: A constructor of StaticObjectClass is invoked.\n");

        NN_LOG("step2: Because nn::init::InitializeAllocator() has been already called, \n");
        NN_LOG("step2: we can use malloc() here.\n\n");

        m_AllocatedMemory = malloc( 1024 );
    }

    // デストラクタ
    // 定義はしているものの、NX32/NX64 実機環境ではデストラクタは呼ばれない
    ~StaticObjectClass()
    {
        free( m_AllocatedMemory );
    }

    // アドレス取得
    void* GetMallocAddress()
    {
        return m_AllocatedMemory;
    }

private:
    void*   m_AllocatedMemory;
};

// 静的オブジェクト
StaticObjectClass   g_StaticObject;

//-----------------------------------------------------------------------------
//  メイン関数
//-----------------------------------------------------------------------------

extern "C" void nnMain()
{
    NN_LOG("step3: nnMain() function is invoked.\n");

    void* MallocAddress = g_StaticObject.GetMallocAddress();
    NN_LOG("step3: A malloc() address in StaticObjectClass: address = 0x%p\n", MallocAddress);

    NN_LOG("step3: nnMain() function is finished.\n");
}
