﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


 /**
 * @examplesource{ImguiDemo_UserInputs.cpp,PageSampleImguiDemo}
 *
 * @brief
 *  nns::gfx::dbgui　入力の使い方を示すサンプル
 */

#include "ImguiDemo_UserInputs.h"

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Common.h>

#include <nns/dbgui/dbgui_Interface.h>

#include "ImguiDemo_IrSensor.h"
#include "ImguiDemo_SixAxis.h"
#include "ImguiDemo_SixAxisSensorPointer.h"

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#define NNS_IMGUIDEMO_ENABLE_WINDOWS_OS_FEATURES
#endif // defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)

#if defined(NNS_IMGUIDEMO_ENABLE_WINDOWS_OS_FEATURES)
#include <nn/vi.h>
#include <nn/nn_Windows.h>
#include <nn/os/os_Mutex.h>
#endif


namespace
{

#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
bool g_IsSixAxisPointerActive = false;
SixAxisSensorPointer g_SixAxisSensorPointer;
#endif

nn::hid::MouseState g_SimulatedMouseState = {};

bool TestApplicationHasFocus(nn::vi::Layer* pLayer)
{
#if defined(NNS_IMGUIDEMO_ENABLE_WINDOWS_OS_FEATURES)
    nn::vi::NativeWindowHandle nativeWindowHandle = {};
    nn::vi::GetNativeWindow(&nativeWindowHandle, pLayer);
    HWND hWnd = reinterpret_cast<HWND>(nativeWindowHandle);
    HWND hForegroundWnd = GetForegroundWindow();
    if (hWnd != hForegroundWnd)
    {
        return false;
    }

    return true;
#else

    return true;

#endif
}

template<typename T>
inline T Clamp(T value, T min, T max)
{
    if (value < min)
    {
        return min;
    }

    if (value > max)
    {
        return max;
    }

    return value;
}

int ComputeMouseDeltaFromAnalogStick(int32_t analogStickAxisValue, int threshold, int speedMin, int speedMax)
{
    int speed = 0;

    if (analogStickAxisValue <= -threshold)
    {
        int speedFactor = analogStickAxisValue + threshold;
        int speedRange = nn::hid::AnalogStickMax - threshold;
        speed = ((speedFactor * (speedMax - speedMin)) / speedRange) - speedMin;
    }
    else if (analogStickAxisValue >= threshold)
    {
        int speedFactor = analogStickAxisValue - threshold;
        int speedRange = nn::hid::AnalogStickMax - threshold;
        speed = ((speedFactor * (speedMax - speedMin)) / speedRange) + speedMin;
    }

    return speed;
}

void UpdateMouseFromDebugPad(
    nn::hid::MouseState* pMouseState,
    const nn::hid::DebugPadState& debugPadState)
{
    const int mouseMoveThreshold = 1024;
    const int mouseSpeedMin = 1;
    const int mouseSpeedMax = 20;
    const int mouseResolutionX = 1280;
    const int mouseResolutionY = 720;

    int mouseDeltaX = ComputeMouseDeltaFromAnalogStick(
        debugPadState.analogStickR.x + debugPadState.analogStickL.x,
        mouseMoveThreshold, mouseSpeedMin, mouseSpeedMax);

    int mouseDeltaY = ComputeMouseDeltaFromAnalogStick(
        -(debugPadState.analogStickR.y + debugPadState.analogStickL.y),
        mouseMoveThreshold, mouseSpeedMin, mouseSpeedMax);

    int newMouseX = Clamp(pMouseState->x + mouseDeltaX, 0, mouseResolutionX);
    int newMouseY = Clamp(pMouseState->y + mouseDeltaY, 0, mouseResolutionY);

    pMouseState->deltaX = newMouseX - pMouseState->x;
    pMouseState->deltaY = newMouseY - pMouseState->y;

    pMouseState->x = newMouseX;
    pMouseState->y = newMouseY;

    pMouseState->buttons.Set<nn::hid::MouseButton::Left>(
        debugPadState.buttons.Test<nn::hid::DebugPadButton::A>()
        || debugPadState.buttons.Test<nn::hid::DebugPadButton::Right>());
    pMouseState->buttons.Set<nn::hid::MouseButton::Right>(
        debugPadState.buttons.Test<nn::hid::DebugPadButton::B>()
        || debugPadState.buttons.Test<nn::hid::DebugPadButton::Down>());
    pMouseState->buttons.Set<nn::hid::MouseButton::Middle>(
        debugPadState.buttons.Test<nn::hid::DebugPadButton::X>()
        || debugPadState.buttons.Test<nn::hid::DebugPadButton::Up>());
}

#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
bool UpdateMouseFromSixAxisPointer(
    nn::hid::MouseState* pMouseState,
    const nn::hid::NpadJoyDualState& npadDualJoyState,
    const nn::hid::SixAxisSensorState& sixAxisSensorState,
    bool sixAxisPointerActive)
{
    if (!sixAxisPointerActive)
    {
        g_SixAxisSensorPointer.Reset(
            sixAxisSensorState.direction,
            static_cast<float>(pMouseState->x),
            static_cast<float>(pMouseState->y));
    }

    g_SixAxisSensorPointer.Update(sixAxisSensorState.direction);
    nn::util::Vector3f cursor = g_SixAxisSensorPointer.GetCursor();

    pMouseState->x = static_cast<int>(cursor.GetX());
    pMouseState->y = static_cast<int>(cursor.GetY());

    pMouseState->buttons.Set<nn::hid::MouseButton::Left>(
        npadDualJoyState.buttons.Test<nn::hid::NpadButton::A>()
        || npadDualJoyState.buttons.Test<nn::hid::NpadButton::Right>());
    pMouseState->buttons.Set<nn::hid::MouseButton::Right>(
        npadDualJoyState.buttons.Test<nn::hid::NpadButton::B>()
        || npadDualJoyState.buttons.Test<nn::hid::NpadButton::Down>());
    pMouseState->buttons.Set<nn::hid::MouseButton::Middle>(
        npadDualJoyState.buttons.Test<nn::hid::NpadButton::X>()
        || npadDualJoyState.buttons.Test<nn::hid::NpadButton::Up>());

    return sixAxisPointerActive;
}
#endif // defined(NN_BUILD_TARGET_PLATFORM_OS_NN)

#if defined(NNS_IMGUIDEMO_ENABLE_WINDOWS_OS_FEATURES)
bool InitializeWindowsOsCharacterInput(nn::vi::Layer* pLayer);
void FlushWindowsOsCharacterInput();
#endif // defined(NNS_IMGUIDEMO_ENABLE_WINDOWS_OS_FEATURES)

} // anonymous namespace



void InitializeApplicationInputs(nn::vi::Layer* pLayer, ApplicationInputs* pApplicationInputs)
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeKeyboard();
    nn::hid::InitializeMouse();

#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
    nn::hid::InitializeTouchScreen();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleHandheld::Mask | nn::hid::NpadStyleJoyDual::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));
    InitializeIrSensor(nn::hid::NpadId::No1);

    const int sensorIndex = 1; // 右コントローラ
    InitializeSixAxis(&pApplicationInputs->sixAxisSensorHandle, nn::hid::NpadId::No1, sensorIndex);

    const float cursorResolutionWidth = 1280.0f;
    const float cursorResolutionHeight = 720.0f;
    g_SixAxisSensorPointer.Initialize(cursorResolutionWidth, cursorResolutionHeight);
#else
    NN_UNUSED(pApplicationInputs);
#endif

#if defined(NNS_IMGUIDEMO_ENABLE_WINDOWS_OS_FEATURES)
    InitializeWindowsOsCharacterInput(pLayer);
#else
    NN_UNUSED(pLayer);
#endif
}

void FinalizeApplicationInputs(ApplicationInputs* pApplicationInputs)
{
#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
    FinalizeSixAxis(pApplicationInputs->sixAxisSensorHandle);
#else
    NN_UNUSED(pApplicationInputs);
#endif
}

void GetApplicationInputs(nn::vi::Layer* pLayer, ApplicationInputs* pApplicationInputs)
{

    pApplicationInputs->previous = pApplicationInputs->current;

    if (!TestApplicationHasFocus(pLayer))
    {
        memset(&pApplicationInputs->current, 0, sizeof(pApplicationInputs->current));
        return;
    }

    nn::hid::GetDebugPadState(&pApplicationInputs->current.debugPadState);
    nn::hid::GetKeyboardState(&pApplicationInputs->current.keyboardState);
    nn::hid::GetMouseState(&pApplicationInputs->current.mouseState);

#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
    nn::hid::GetTouchScreenState(&pApplicationInputs->current.touchScreenState);
    if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
    {
        nn::hid::GetNpadState(
            &pApplicationInputs->current.npadHandheldState,
            nn::hid::NpadId::Handheld);
    }
    if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleJoyDual>())
    {
        nn::hid::GetNpadState(
            &pApplicationInputs->current.npadJoyDualState,
            nn::hid::NpadId::No1);
    }
    UpdateSixAxis(
        &pApplicationInputs->current.sixAxisSensorState,
        pApplicationInputs->sixAxisSensorHandle);
#endif
}

void UpdateDdguiInputs(const ApplicationInputs* pApplicationInputs)
{
    const InputData* pInputData = &pApplicationInputs->current;

    bool updateSimulatedMouse = false;

#if defined(NNS_IMGUIDEMO_ENABLE_WINDOWS_OS_FEATURES)
    FlushWindowsOsCharacterInput();
#endif // #if defined(NNS_IMGUIDEMO_ENABLE_WINDOWS_OS_FEATURES)

    if (pInputData->keyboardState.attributes.Test<nn::hid::KeyboardAttribute::IsConnected>())
    {
        nns::dbgui::SetKeyboardState(pInputData->keyboardState);
#if !defined(NNS_IMGUIDEMO_ENABLE_WINDOWS_OS_FEATURES)
        nns::dbgui::AddInputCharactersFromKeyboardState(pInputData->keyboardState);
#endif // !defined(NNS_IMGUIDEMO_ENABLE_WINDOWS_OS_FEATURES)

    }
    else
    {
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        nns::dbgui::UpdateSoftwareKeyboard();
#endif // defined(NN_BUILD_TARGET_PLATFORM_NX)
    }

    if (pInputData->debugPadState.attributes.Test<nn::hid::DebugPadAttribute::IsConnected>())
    {
        // スタートボタンを押すと、アナログスティックでマウスの作動できます。
        if (pInputData->debugPadState.buttons.Test<nn::hid::DebugPadButton::Start>())
        {
            updateSimulatedMouse = true;
            UpdateMouseFromDebugPad(
                &g_SimulatedMouseState,
                pInputData->debugPadState);
        }
        else
        {
            nns::dbgui::SetDebugPadState(
                pInputData->debugPadState.buttons,
                pInputData->debugPadState.analogStickL);
        }
    }

    if (pInputData->mouseState.attributes.Test<nn::hid::MouseAttribute::IsConnected>())
    {
        nns::dbgui::SetMouseState(pInputData->mouseState);
    }

#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
    if (pInputData->touchScreenState.count > 0)
    {
        nns::dbgui::SetTouchState(pInputData->touchScreenState.touches[0]);
    }

    if (pInputData->npadHandheldState.attributes.Test<nn::hid::NpadAttribute::IsConnected>())
    {
        nns::dbgui::SetNpadState(
            pInputData->npadHandheldState.buttons,
            pInputData->npadHandheldState.analogStickL);
    }
    else if (pInputData->npadJoyDualState.attributes.Test<nn::hid::NpadAttribute::IsConnected>())
    {
        UpdateIrSensor();

        // プラスボタンを押すと、SixAxisポインタでマウスの作動できます。
        if (pInputData->npadJoyDualState.buttons.Test<nn::hid::NpadButton::Plus>())
        {
            updateSimulatedMouse = true;
            UpdateMouseFromSixAxisPointer(
                &g_SimulatedMouseState,
                pInputData->npadJoyDualState, pInputData->sixAxisSensorState,
                g_IsSixAxisPointerActive);

            g_IsSixAxisPointerActive = true;
        }
        else
        {
            g_IsSixAxisPointerActive = false;

            nn::hid::NpadJoyDualState state = pInputData->npadJoyDualState;

            IrSensorHandInput irSensorHandInput = GetIrSensorHandInput();
            if (irSensorHandInput == IrSensorHandInput_Validate)
            {
                state.buttons.Set<nn::hid::NpadButton::A>();
            }
            else if (irSensorHandInput == IrSensorHandInput_Cancel)
            {
                state.buttons.Set<nn::hid::NpadButton::B>();
            }

            nns::dbgui::SetNpadState(
                state.buttons,
                state.analogStickL);
        }
    }
#endif // defined(NN_BUILD_TARGET_PLATFORM_OS_NN)

    if (updateSimulatedMouse)
    {
        nns::dbgui::SetMouseState(g_SimulatedMouseState);
    }
}

template<typename TState>
bool TestToggleDebugGuiInput(
    const TState& previous, const TState& current,
    int connected,
    int b0, int b1)
{

    if (previous.attributes.Test(connected) && current.attributes.Test(connected))
    {
        decltype(current.buttons) toggledButtons = current.buttons ^ previous.buttons;

        if (toggledButtons.Test(b0) || toggledButtons.Test(b1))
        {
            return previous.buttons.Test(b0) && previous.buttons.Test(b1);
        }
    }

    return false;
}

bool TestToggleDebugGui(const ApplicationInputs* pApplicationInputs)
{
    bool toggle = false;

    toggle |= TestToggleDebugGuiInput(
        pApplicationInputs->previous.debugPadState,
        pApplicationInputs->current.debugPadState,
        nn::hid::DebugPadAttribute::IsConnected::Index,
        nn::hid::DebugPadButton::Start::Index,
        nn::hid::DebugPadButton::R::Index);

    if (pApplicationInputs->current.keyboardState.modifiers.Test<nn::hid::KeyboardModifier::LeftAlt>())
    {
        nn::hid::KeyboardKeySet toggledKeyboardKeySet =
            pApplicationInputs->previous.keyboardState.keys ^ pApplicationInputs->current.keyboardState.keys;

        if (pApplicationInputs->current.keyboardState.keys.Test<nn::hid::KeyboardKey::F1>()
            && toggledKeyboardKeySet.Test<nn::hid::KeyboardKey::F1>())
        {
            toggle = true;
        }
    }

#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
    toggle |= TestToggleDebugGuiInput(
        pApplicationInputs->previous.npadHandheldState,
        pApplicationInputs->current.npadHandheldState,
        nn::hid::NpadAttribute::IsConnected::Index,
        nn::hid::NpadButton::Plus::Index,
        nn::hid::NpadButton::R::Index);

    toggle |= TestToggleDebugGuiInput(
        pApplicationInputs->previous.npadJoyDualState,
        pApplicationInputs->current.npadJoyDualState,
        nn::hid::NpadAttribute::IsConnected::Index,
        nn::hid::NpadButton::Plus::Index,
        nn::hid::NpadButton::R::Index);
#endif // defined(NN_BUILD_TARGET_PLATFORM_OS_NN)

    return toggle;
}

void OnMouseMoveFunction(int x, int y, void* pUserData)
{
    NN_ASSERT(pUserData == nullptr);
    NN_UNUSED(pUserData);

    int newX = x;
    int newY = y;
    int deltaX = newX - g_SimulatedMouseState.x;
    int deltaY = newY - g_SimulatedMouseState.y;

    g_SimulatedMouseState.x = newX;
    g_SimulatedMouseState.y = newY;
    g_SimulatedMouseState.deltaX = deltaX;
    g_SimulatedMouseState.deltaY = deltaY;
}


#if defined(NNS_IMGUIDEMO_ENABLE_WINDOWS_OS_FEATURES)

namespace
{

struct WindowsOsCharacterInputData
{
    WNDPROC previousProcedure = nullptr;
    char outputBuffer[128] = { 0 };
    int outputBufferIndex = 0;
    nn::os::Mutex outputBufferMutex;


    WindowsOsCharacterInputData()
    : outputBufferMutex(false)
    {
    }

    ~WindowsOsCharacterInputData()
    {
    }
};

WindowsOsCharacterInputData g_WindowsOsCharacterInputData;


LRESULT CALLBACK ImguiDemoWndProc(
    HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam) NN_NOEXCEPT

{
    if (uMsg == WM_CHAR)
    {
        switch (wParam)
        {
        case VK_BACK:
        case VK_TAB:
        case VK_CLEAR:
        case VK_RETURN:
        case VK_SHIFT:
        case VK_CONTROL:
        case VK_MENU:
        case VK_PAUSE:
        case VK_CAPITAL:
        case VK_ESCAPE:
        case VK_PRIOR:
        case VK_NEXT:
        case VK_END:
        case VK_HOME:
        case VK_LEFT:
        case VK_UP:
        case VK_RIGHT:
        case VK_DOWN:
            break;

        default:
            {
                WindowsOsCharacterInputData* pData = &g_WindowsOsCharacterInputData;
                pData->outputBufferMutex.Lock();
                NN_ASSERT(pData->outputBufferIndex < (sizeof(WindowsOsCharacterInputData::outputBuffer) - 1));
                if (pData->outputBufferIndex < (sizeof(WindowsOsCharacterInputData::outputBuffer) - 1))
                {
                    wchar_t inputBuffer[2] = { static_cast<wchar_t>(wParam), 0 };
                    int convertionResult = WideCharToMultiByte(
                        CP_UTF8, 0,
                        inputBuffer, 1,
                        pData->outputBuffer + pData->outputBufferIndex,
                        sizeof(WindowsOsCharacterInputData::outputBuffer) - pData->outputBufferIndex - 1,
                        nullptr, nullptr);
                    NN_ASSERT(convertionResult > 0);
                    pData->outputBufferIndex += convertionResult;
                }
                pData->outputBufferMutex.Unlock();
            }
            break;
        }
    }
    else if (uMsg == WM_UNICHAR)
    {
        NN_LOG("WM_UNICHAR\n");
        NN_ASSERT(0); // 対応されていません。
    }

    return ::CallWindowProc(g_WindowsOsCharacterInputData.previousProcedure, hwnd, uMsg, wParam,
        lParam);
}

bool InitializeWindowsOsCharacterInput(nn::vi::Layer* pLayer)
{
    nn::vi::NativeWindowHandle nativeWindowHandle = {};
    nn::vi::GetNativeWindow(&nativeWindowHandle, pLayer);
    HWND hWnd = reinterpret_cast<HWND>(nativeWindowHandle);

    auto defaultProcedure =
        reinterpret_cast<WNDPROC>(::GetClassLongPtr(hWnd, GCLP_WNDPROC));

    if (defaultProcedure != 0)
    {
        g_WindowsOsCharacterInputData.previousProcedure = defaultProcedure;

        auto previousProcedure =
            reinterpret_cast<WNDPROC>(
                ::SetWindowLongPtr(
                    hWnd,
                    GWLP_WNDPROC,
                    reinterpret_cast<LONG_PTR>(ImguiDemoWndProc)));

        if (previousProcedure != 0)
        {
            g_WindowsOsCharacterInputData.previousProcedure = previousProcedure;
            return true;
        }
    }

    return false;
}

void FlushWindowsOsCharacterInput()
{
    WindowsOsCharacterInputData* pData = &g_WindowsOsCharacterInputData;
    pData->outputBufferMutex.Lock();
    if (pData->outputBufferIndex > 0)
    {
        NN_ASSERT(pData->outputBufferIndex < NN_ARRAY_SIZE(WindowsOsCharacterInputData::outputBuffer));
        pData->outputBuffer[pData->outputBufferIndex] = '\0';
        nns::dbgui::AddInputCharactersUtf8(pData->outputBuffer);
        pData->outputBufferIndex = 0;
    }
    pData->outputBufferMutex.Unlock();
}



} // anonymous namespace

#endif // defined(NNS_IMGUIDEMO_ENABLE_WINDOWS_OS_FEATURES)
