﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>

#include "IgeGraphicsSystem.h"

//------------------------------------------------------------------------------
//  コマンド生成
//------------------------------------------------------------------------------

void IgeGraphicsSystem::MakeCommand(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData) NN_NOEXCEPT
{
    IgeGraphicsSystem* pSystem = reinterpret_cast<IgeGraphicsSystem*>(pUserData);

    // フォント表示
    if (pSystem && pSystem->m_pPrintFunc)
    {
        pSystem->m_pPrintFunc(&pSystem->m_Writer);
    }

    // コマンド生成
    pGraphicsFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

        // レンダーターゲット、ビューポートシザー設定
        nn::gfx::ColorTargetView* target = pGraphicsFramework->GetColorTargetView();
        rootCommandBuffer->ClearColor(target, 0.1f, 0.1f, 0.1f, 1.0f, nullptr);
        rootCommandBuffer->SetRenderTargets(1, &target, nullptr);
        rootCommandBuffer->SetViewportScissorState(pGraphicsFramework->GetViewportScissorState());

        // デバッグフォント用のコマンド生成
        pSystem->m_Writer.Draw(rootCommandBuffer);
    }

    pGraphicsFramework->EndFrame(bufferIndex, true);
}

IgeGraphicsSystem::IgeGraphicsSystem(PrintFunc pPrintFunc)
    : m_Framework()
    , m_Writer()
    , m_pPrintFunc(pPrintFunc)
    , m_pDebugFontHeap(nullptr)
{
    Initialize();
}

IgeGraphicsSystem::~IgeGraphicsSystem()
{
    Finalize();
}

void IgeGraphicsSystem::Initialize() NN_NOEXCEPT
{
    // フレームワーク初期化
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsSystemMemorySize);

    const int BufferCount = 2;
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetBufferCount(BufferCount);
    fwInfo.SetSwapChainBufferCount(BufferCount);

    m_Framework.Initialize(fwInfo);

    // デバッグフォント初期化
    const int charCountMax = 1024;
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(charCountMax);
    info.SetBufferCount(BufferCount);
    info.SetUserMemoryPoolEnabled(false);

    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(m_Framework.GetDevice(), info);
    m_pDebugFontHeap = new uint8_t[debugFontHeapSize];

    m_Writer.Initialize(m_Framework.GetDevice(), info, m_pDebugFontHeap, debugFontHeapSize, nullptr, 0, 0);

    int textureDescriptorIndex = m_Framework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    int samplerDescriptorIndex = m_Framework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);

    m_Writer.SetDisplayWidth(m_Framework.GetDisplayWidth());
    m_Writer.SetDisplayHeight(m_Framework.GetDisplayHeight());
    m_Writer.SetTextureDescriptor(m_Framework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView), textureDescriptorIndex);
    m_Writer.SetSamplerDescriptor(m_Framework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler), samplerDescriptorIndex);

    m_Framework.SetMakeCommandCallback(MakeCommand, this);
}

void IgeGraphicsSystem::ProcessFrame() NN_NOEXCEPT
{
    m_Framework.ProcessFrame();
}

void IgeGraphicsSystem::Finalize() NN_NOEXCEPT
{
    m_Framework.QueueFinish();

    int textureDescriptorIndex = m_Writer.GetTextureDescriptorIndexSlot();
    int samplerDescriptorIndex = m_Writer.GetSamplerDescriptorIndexSlot();

    // デバッグフォント終了
    m_Writer.Finalize();
    delete[] m_pDebugFontHeap;
    m_pDebugFontHeap = nullptr;

    m_Framework.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, textureDescriptorIndex);
    m_Framework.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, samplerDescriptorIndex);

    // フレームワーク終了
    m_Framework.Finalize();
}

