﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{HtcsTargetApp.cpp,PageSampleHtcsTargetApp}
 *
 * @brief
 *  ホスト PC との通信のサンプルプログラム
 */

/**
 * @page PageSampleHtcsTargetApp ホスト PC との通信
 * @tableofcontents
 *
 * @brief
 *  ホスト PC との通信のサンプルプログラムの解説です。
 *
 * @section PageSampleHtcsTargetApp_SectionBrief 概要
 *  ここでは、HTCS ライブラリを使ってホスト PC との通信を行うサンプルプログラムの説明をします。
 *
 *  @confluencelink{83956008,HTCS ライブラリのマニュアル} も併せて参照してください。
 *
 * @section PageSampleHtcsTargetApp_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/Htcs Samples/Sources/Applications/Htcs @endlink 以下にあります。
 *
 *  <p>
 *  <table>
 *  <tr><th> ディレクトリ </th><th> 概要 </th></tr>
 *  <tr><td> HtcsDll        </td><td> Target Manager が開いたポートから送られてくるテキストプロトコルの解釈を行う DLL のサンプルです。 </td></tr>
 *  <tr><td> HtcsHostApp    </td><td> ホスト PC 上で動くアプリのサンプルです。 </td></tr>
 *  <tr><td> HtcsTargetApp  </td><td> ターゲット上で動くアプリのサンプルです。本項ではこちらを解説しています。 </td></tr>
 *  </table>
 *  </p>
 *
 * @section PageSampleHtcsTargetApp_SectionNecessaryEnvironment 必要な環境
 *  Target Manager がインストールされている必要があります。
 *
 * @section PageSampleHtcsTargetApp_SectionHowToOperate 操作方法
 *  特になし。
 *
 * @section PageSampleHtcsTargetApp_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleHtcsTargetApp_SectionHowToExecute 実行手順
 *  Target Manager を起動していない場合は、サンプルの実行前に起動しておいてください。
 *  その後 HtcsHostApp と HtcsTargetApp をそれぞれビルドし、実行してください。
 *
 * @section PageSampleHtcsTargetApp_SectionDetail 解説
 *
 * @subsection PageSampleHtcsTargetApp_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  HtcsTargetApp.cpp
 *  @includelineno HtcsTargetApp.cpp
 *
 * @subsection PageSampleHtcsTargetApp_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - ターゲット側をサーバーとした htcs 通信のサンプルを実行
 *  - ターゲット側をクライアントとした htcs 通信のサンプルを実行
 *
 *  サーバー、クライアントそれぞれでホスト側から送られてきたデータをそのまま送り返しています。
 *
 *  このサンプルの実行結果の例を以下に示します。
 *
 *  HtcsTargetApp の実行結果
 *  @verbinclude HtcsTargetApp_OutputExample.txt
 *
 *  HtcsHostApp の実行結果
 *  @verbinclude HtcsHostApp_OutputExample.txt
 *
 */

#include <cstdlib>
#include <cstdio>
#include <memory>
#include <nn/nn_Log.h>
#include <nn/htcs.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

namespace {
    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }
}

// 送られてきたデータをそのまま相手に送り返す
void EchoBack(int sock)
{
    char buf[128];
    nn::htcs::ssize_t receivedBytes = nn::htcs::Recv(sock, buf, sizeof(buf) - 1, 0);
    if (receivedBytes < 0)
    {
        NN_LOG("Recv error %d\n", nn::htcs::GetLastError());
        return;
    }
    else if (receivedBytes == 0)
    {
        NN_LOG("gracefully closed\n");
        return;
    }

    buf[receivedBytes] = '\0';
    NN_LOG("Received \"%s\".\n", buf);

    nn::htcs::ssize_t sentBytes = nn::htcs::Send(sock, buf, receivedBytes, 0);
    if (sentBytes < 0)
    {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
    }
}

void ServerSample()
{
    // サービスを登録
    nn::htcs::SockAddrHtcs serviceAddr;
    serviceAddr.family = nn::htcs::HTCS_AF_HTCS;
    serviceAddr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(serviceAddr.portName.name, "ServerInTarget");

    int serviceSocket = nn::htcs::Socket();
    NN_ABORT_UNLESS(nn::htcs::Bind(serviceSocket, &serviceAddr) == 0, "Server: Bind failed\n");
    NN_ABORT_UNLESS(nn::htcs::Listen(serviceSocket, 1) == 0, "Server: Listen failed\n");

    // 接続を待ち受ける
    NN_LOG("Waiting for connection from host.\n");
    int sock = nn::htcs::Accept(serviceSocket, nullptr);
    NN_ABORT_UNLESS(sock >= 0, "Server: Accept failed. Error(%d).\n", nn::htcs::GetLastError());
    NN_LOG("Accepted client on host.\n");

    // エコーバックテスト
    EchoBack(sock);

    // 切断
    nn::htcs::Close(sock);
    nn::htcs::Close(serviceSocket);
    NN_LOG("Closed.\n");
}

void ClientSample()
{
    // 接続したいホスト側サーバーのアドレス情報構築
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetDefaultHostName();
    std::strcpy(addr.portName.name, "ServerInHost");

    // 繋がるまで接続トライ
    int sock = nn::htcs::Socket();
    while (NN_STATIC_CONDITION(true))
    {
        if (nn::htcs::Connect(sock, &addr) == 0)
        {
            NN_LOG("Connected to server on host.\n");
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    }

    // エコーバックテスト
    EchoBack(sock);

    // 切断
    nn::htcs::Close(sock);
    NN_LOG("Closed.\n");
}

extern "C" void nnMain()
{
    // ホストデーモンに接続
    nn::htcs::Initialize(&Allocate, &Deallocate);

    ServerSample();
    ClientSample();

    // 終了
    nn::htcs::Finalize();

    NN_LOG("Sample end\n");
    while (NN_STATIC_CONDITION(true))
    {
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    }

    return;
}
