﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{HidbusRonde.cpp,PageSampleHidbusRonde}

    @brief
    RONDE を使用するためのサンプルプログラム
 */

/**
    @page PageSampleHidbus RONDE 機能
    @tableofcontents

    @brief
    RONDE を使用するためのサンプルプログラムの解説です。

    @section PageSampleHidbusRonde_SectionBrief 概要
    RONDE を使用する方法の説明をします。

    @section PageSampleHidbusRonde_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidbusRonde @endlink 以下にあります。

    @section PageSampleHidbusRonde_SectionNecessaryEnvironment 必要な環境
    Windows 環境で動作させる場合は、事前に PC に Bluetooth ドングルを接続した上で、PC とコントローラーをペアリングしてください。
    SDEV/EDEV 環境で動作させる場合は、事前に SDEV/EDEV とコントローラーをペアリングしてください。

    @section PageSampleHidbusRonde_SectionHowToOperate 操作方法
    サンプルプログラムを実行して、コントローラーのボタンを押してください。
    Bluetooth の接続が確立すると、コントローラーの LED が点滅から点灯に変わります。

    接続後、コントローラの拡張バス機能に対応したデバイスを接続すると RONDE へポーリングしてデータを要求し、取得されたデータを表示します。
    その他、フォーカス状態に変更があった場合や、 RONDE が外された時のハンドリングも実装しています。

    @section PageSampleHidbusRonde_SectionPrecaution 注意事項
    コントローラは十分に充電した状態でお使いください。
    Hidbus の API には呼び出し元スレッドを長時間ブロックするものが存在するため、
    メインループとは別のスレッドから呼び出すことを推奨します。 詳細については、API リファレンスをご覧ください。

    @section PageSampleHidbusRonde_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleHidbusRonde_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - NpadID を設定
    - NpadID から Hidbus のハンドルを取得 (本サンプルでは右 Joy-Con を使用)
    - Ronde との通信を有効化
    - Ronde の各種データの表示
    - Ronde Sensor の値を表示
    - 上記を右 Joy-Con の + ボタンが押されるまで繰り返す。
 */

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_ControllerSupport.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/hidbus/hidbus.h>

#include <nns/gfxLog.h>
#include <nns/nns_Log.h>
#include <nns/hidbus/hidbus_Ronde.h>

namespace {

    enum HidbusState
    {
        HidbusState_NotGotHandle = 0,
        HidbusState_GotHandle    = 1,
        HidbusState_Enabled      = 2,
        HidbusState_End          = 3,
    };

    HidbusState state = HidbusState_NotGotHandle;

    struct Npad
    {
        typedef nn::hid::NpadIdType Id;
        typedef nn::hid::NpadButton Button;
    };

    Npad::Id npadIds[] =
    {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::No2,
    };

    nn::hidbus::BusHandle rightJoyHandle;

    // PollingMode 時に渡すバッファ (handle ごとに別のバッファが必要です。)
    NN_ALIGNAS(nn::os::MemoryPageSize) char s_Buffer[nn::os::MemoryPageSize];

    bool IsFinished()
    {
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(npadIds[0]);

        if (style.Test<nn::hid::NpadStyleJoyRight>())
        {
            nn::hid::NpadJoyRightState stateJoyRight;
            nn::hid::GetNpadState(&stateJoyRight, npadIds[0]);

            // ＋ ボタンが押されたら終了
            if (stateJoyRight.buttons.Test<nn::hid::NpadButton::Plus>())
            {
                return true;
            }
        }
        return false;
    }

    bool IsAButtonPushed()
    {
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(npadIds[0]);

        if (style.Test<nn::hid::NpadStyleJoyRight>())
        {
            nn::hid::NpadJoyRightState stateJoyRight;
            nn::hid::GetNpadState(&stateJoyRight, npadIds[0]);

            if (stateJoyRight.buttons.Test<nn::hid::NpadButton::A>())
            {
                return true;
            }
        }
        return false;
    }

    HidbusState HandleCommunicationErrorResult(nn::Result result, HidbusState currentState)
    {
        if (nn::hidbus::ResultExternalDeviceTimeout::Includes(result))
        {
            // 拡張バスに接続されたデバイスとの通信中にタイムアウトが発生しました。
            // リトライしてください。
            NNS_LOG("Timeout Occur. Retry\n");
            return currentState;
        }
        else if (nn::hidbus::ResultExternalDeviceNotEnabled::Includes(result))
        {
            // 拡張バスに接続されたデバイスの取り外しや、アプリケーションが InForcus でなくなった等の理由により、
            // 拡張バスに接続されたデバイスとの通信機能が無効化されています。
            // 再度 Enable からやり直してください。
            NNS_LOG("NotEnabled. \n");
            return HidbusState_GotHandle;
        }
        else if (nn::hidbus::ResultInvalidHandle::Includes(result))
        {
            // handle に対応したコントローラーが切断されており、 handle が無効化されています。
            // 再度 handle の取得からやり直してください。
            NNS_LOG("Invalid Handle. \n");
            return HidbusState_NotGotHandle;
        }
        else
        {
            // 通信コマンド送受信時は他の Result は返りません。
            NN_ABORT("Unexpected Error\n");
        }
    }

    HidbusState ViewRondeInfomation(nn::hidbus::BusHandle handle, HidbusState currentState)
    {
        // 各種情報を取得し、表示します。
        uint8_t uniqueId[nns::hidbus::RondeUniqueIdSize];
        nns::hidbus::RondeManufactureCalibrationData calData;
        while (NN_STATIC_CONDITION(true))
        {
            // UniqueID の取得
            auto uniqueIdResult = nns::hidbus::GetRondeUniqueId(uniqueId, sizeof(uniqueId), handle);
            if (uniqueIdResult.IsSuccess())
            {
                break;
            }
            else
            {
                auto handledState = HandleCommunicationErrorResult(uniqueIdResult, currentState);
                if (handledState != currentState)
                {
                    return handledState;
                }
            }
        }

        while (NN_STATIC_CONDITION(true))
        {
            // 工程キャルの取得
            auto calibrationResult = nns::hidbus::GetRondeManufactureCalibrationData(&calData, handle);
            if (calibrationResult.IsSuccess())
            {
                break;
            }
            else
            {
                auto handledState = HandleCommunicationErrorResult(calibrationResult, currentState);
                if (handledState != currentState)
                {
                    return handledState;
                }
            }
        }

        NNS_LOG("-------------- Ronde Info ---------------\n");
        NNS_LOG("Unique ID = ");
        for (int i = nns::hidbus::RondeUniqueIdSize - 1; i >= 0; --i)
        {
            NNS_LOG("%02x", uniqueId[i]);
        }
        NNS_LOG("\n");

        NNS_LOG("Manufacture Cal : \n");
        NNS_LOG(" osMax   = %d\n", calData.osMax);
        NNS_LOG(" hkMax   = %d\n", calData.hkMax);
        NNS_LOG(" zeroMax = %d\n", calData.zeroMax);
        NNS_LOG(" zeroMin = %d\n", calData.zeroMin);
        NNS_LOG("-----------------------------------------\n");

        return currentState;
    }


    HidbusState WaitSupportedJoyConConnected(nn::hidbus::BusHandle* pOutHandle)
    {
        while (NN_STATIC_CONDITION(true))
        {
            // RightJoyRail を持つコントローラーが接続されるまで待ちます。
            while (!nn::hidbus::GetBusHandle(pOutHandle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
            {
                NNS_LOG("Please connect supported controller.\n");
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }

            // 対応するコントローラーが接続されたら、ライブラリを初期化します。
            auto result = nn::hidbus::Initialize(*pOutHandle);
            if (result.IsSuccess())
            {
                return HidbusState_GotHandle;
            }
            // Initialize に失敗した時はコントローラーが切断される等によって handle が無効化されているため、
            // 再度 handle 取得からやり直します。
        }
    }

    HidbusState EnableDevice(nn::hidbus::BusHandle handle, char* pBuffer, size_t bufferSize)
    {
        while (NN_STATIC_CONDITION(true))
        {
            auto enableResult = nns::hidbus::EnableRonde(handle, pBuffer, bufferSize);
            if (enableResult.IsSuccess())
            {
                // Unique ID とキャル値を表示します。
                auto handledState = ViewRondeInfomation(handle, HidbusState_Enabled);
                if (handledState != HidbusState_Enabled)
                {
                    return handledState;
                }
                NNS_LOG("Push A to start getting sensor data\n");
                while (!IsAButtonPushed())
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
                }
                return HidbusState_Enabled;
            }
            else if (nn::hidbus::ResultFWUpdateFailed::Includes(enableResult))
            {
                // コントローラーの Firmware Update が発生したが失敗しました。
                // リトライするか、アプリ側で Firmware Update アプレットを立ち上げてください。
                // 本サンプルではリトライします。
                NNS_LOG("Firmware Update Failed. retry.\n");
            }
            else if (nn::hidbus::ResultInvalidHandle::Includes(enableResult))
            {
                // コントローラの切断等が発生し、handle が無効になっています。handle 取得からリトライしてください。
                NNS_LOG("Handle is invalid.\n");
                return HidbusState_NotGotHandle;
            }
            else if (nn::hidbus::ResultNotInForcus::Includes(enableResult))
            {
                // アプリケーションが InForcus 状態でないため、拡張デバイスを有効化できません。
                // アプリケーション側で nn::oe のライブラリ群を使用してアプリケーションが InForcus になるのを待ってください。
                // 本サンプルではリトライします。
                NNS_LOG("Not InForcus.\n");
            }
            else if (nn::hidbus::ResultActivateFailureNpadBusy::Includes(enableResult))
            {
                // 指定された Npad が拡張バス機能と排他で使用されるべき機能を使用しているため、有効化できませんでした。
                // アプリケーション側で拡張バスと排他で使用されるべき機能を終了して、再度本関数を実行してください。
                NNS_LOG("Please deactivate the dependency function\n");
            }
            else if (nn::hidbus::ResultExternalDeviceNotAttached::Includes(enableResult))
            {
                // 拡張バスにデバイスが接続されていません。
                // 拡張バスにデバイスが接続されるまでリトライします。
                NNS_LOG("Waiting External Device Connected..\n");
            }
            else if (nn::hidbus::ResultExternalDeviceReadyTimeout::Includes(enableResult))
            {
                // 拡張バスに接続されたデバイスとの通信準備中にタイムアウトしました。
                // 本サンプルではリトライします。
                NNS_LOG("Communication Ready Timeout. Please retry.\n");
            }
            else if (nn::hidbus::ResultExternalDeviceTimeout::Includes(enableResult))
            {
                // 拡張バスに接続されたデバイスとの通信中にタイムアウトしました。
                // ハンドリング方法はデバイスごとに異なりますが、本サンプルではリトライします。
                NNS_LOG("Communication Timeout. Please retry.\n");
            }
            else if (nn::hidbus::ResultExternalDeviceInvalidId::Includes(enableResult))
            {
                // 拡張バスに接続されたデバイスが RONDE ではありません。
                // ユーザーに分かりやすいような表示を行ってください。本サンプルではリトライします。
                NNS_LOG("Attached device is not RONDE. Please attach RONDE.\n");
            }
            else if(nn::hidbus::ResultExternalDeviceNotEnabled::Includes(enableResult))
            {
                // nns::hidbus::EnableRonde では内部で PollingMode の Enable をしているため、
                // NotEnabled が返る可能性がある。
                // この時はリトライする
                NN_LOG("NotEnabled. Please retry.\n");
            }

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }

    HidbusState PollingReadRondeData(nn::hidbus::BusHandle handle)
    {
        while (NN_STATIC_CONDITION(true))
        {
            nns::hidbus::RondeSensorData sensorData;
            auto result = nns::hidbus::GetRondeSensorData(&sensorData, handle);
            if (result.IsSuccess())
            {
                NNS_LOG("Ronde Sensor Value = %d ", sensorData.data);
            }
            else
            {
                auto handledState = HandleCommunicationErrorResult(result, HidbusState_Enabled);
                if (handledState != HidbusState_Enabled)
                {
                    return handledState;
                }
            }

            int16_t thermistor;
            auto thermistorResult = nns::hidbus::GetRondeThermisterData(&thermistor, handle);
            if (thermistorResult.IsSuccess())
            {
                NNS_LOG("Thermistor = %d\n", thermistor);
            }
            else
            {
                auto handledState = HandleCommunicationErrorResult(thermistorResult, HidbusState_Enabled);
                if (handledState != HidbusState_Enabled)
                {
                    return handledState;
                }
            }

            if (IsFinished())
            {
                return HidbusState_End;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }
    }

}//namespace

extern "C" void nnMain()
{
    NNS_LOG("Hidbus RONDE Sample Start.\n");

    nn::hid::InitializeNpad();

    // Npadのスタイルを設定
    nn::hid::SetSupportedNpadStyleSet(
        ::nn::hid::NpadStyleJoyLeft::Mask |
        ::nn::hid::NpadStyleJoyRight::Mask
    );

    // 使用するNpadId の設定
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    // 全体のループ
    bool isFinished = false;
    while (!isFinished)
    {
        switch (state)
        {
        case HidbusState_NotGotHandle:
            state = WaitSupportedJoyConConnected(&rightJoyHandle);
            break;

        case HidbusState_GotHandle:
            state = EnableDevice(rightJoyHandle, s_Buffer, sizeof(s_Buffer));
            break;

        case HidbusState_Enabled:
            state = PollingReadRondeData(rightJoyHandle);
            break;

        case HidbusState_End:
            isFinished = true;
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    // RONDE の無効化
    nns::hidbus::DisableRonde(rightJoyHandle);

    // 拡張デバイス機能の終了
    nn::hidbus::Finalize(rightJoyHandle);

    NN_LOG("Hidbus Ronde Sample Done\n");
}
