﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "VirtualBall.h"

VirtualBall::VirtualBall() NN_NOEXCEPT
{
    m_VibrationNodeConnectionHitWallL.Connect(&m_VibrationHitWall, &m_VibrationMixerL);
    m_VibrationNodeConnectionHitWallR.Connect(&m_VibrationHitWall, &m_VibrationMixerR);
    m_VibrationNodeConnectionRollingL.Connect(&m_VibrationRolling, &m_VibrationMixerL);
    m_VibrationNodeConnectionRollingR.Connect(&m_VibrationRolling, &m_VibrationMixerR);
    m_VibrationNodeConnectionOutputL.Connect(&m_VibrationMixerL, &m_VibrationTargetL);
    m_VibrationNodeConnectionOutputR.Connect(&m_VibrationMixerR, &m_VibrationTargetR);
    this->Reset();
}

VirtualBall::~VirtualBall() NN_NOEXCEPT
{
    m_VibrationNodeConnectionHitWallL.Disconnect();
    m_VibrationNodeConnectionHitWallR.Disconnect();
    m_VibrationNodeConnectionRollingL.Disconnect();
    m_VibrationNodeConnectionRollingR.Disconnect();
    m_VibrationNodeConnectionOutputL.Disconnect();
    m_VibrationNodeConnectionOutputR.Disconnect();
}

void VirtualBall::SetVibrationMixMode(nn::hid::VibrationMixMode mode) NN_NOEXCEPT
{
    m_VibrationMixerL.SetMixMode(mode);
    m_VibrationMixerR.SetMixMode(mode);
}

void VirtualBall::Update(float gx, float gy, float gz) NN_NOEXCEPT
{
    // 重力を調整
    float fy = (((gy < 0.0f) ? -gy : gy) + ((gz < 0.0f) ? -gz : gz)) * m_pDescriptor->weight;

    if (fy > m_pDescriptor->weight)
    {
        fy = m_pDescriptor->weight;
    }

    float fx = ((fy > 0.0f) ? gx : ((gx > 0) ? 1.0f : -1.0f)) * m_pDescriptor->weight;

    // 床方向にかかる力から摩擦を計算
    m_StaticFriction  = m_pDescriptor->coefStaticFriction * fy;
    m_DynamicFriction = m_pDescriptor->coefDynamicFriction * fy;

    // この時点で物体にかかる力
    float fm = m_pDescriptor->weight * m_Velocity + fx;

    // 移動状態かどうかを判断
    m_IsMoving = (m_Velocity != 0.0f);

    if (fm < 0)
    {
        // 左方向に力がかかっている
        m_Velocity = ((fm + (m_IsMoving ? m_DynamicFriction : m_StaticFriction)) / m_pDescriptor->weight);
        if (m_Velocity > 0.0f)
        {
            m_Velocity = 0.0f;
        }
        // すでに左側の壁に接触しているなら移動できない
        if (m_Position <= -m_FieldHalfWidthEfficient)
        {
            m_Velocity = 0.0f;
        }
    }
    else
    {
        // 右方向に力がかかっている
        m_Velocity = ((fm - (m_IsMoving ? m_DynamicFriction : m_StaticFriction)) / m_pDescriptor->weight);
        if (m_Velocity < 0.0f)
        {
            m_Velocity = 0.0f;
        }
        // すでに右側の壁に接触しているなら移動できない
        if (m_Position >= m_FieldHalfWidthEfficient)
        {
            m_Velocity = 0.0f;
        }
    }

    // 速度分移動する
    m_Position += m_Velocity;

    UpdateMoving(fy);
    UpdateHitWall();
}

void VirtualBall::UpdateMoving(float force) NN_NOEXCEPT
{
    // 転がりの判定
    m_IsMoving = (m_Velocity != 0.0f);

    if (m_IsMoving)
    {
        float fp = m_Velocity / 40.0f * ( force / m_pDescriptor->weight );
        fp = ((fp < 0.0f) ? -fp : fp);
        float dp = (m_Position + m_FieldHalfWidth) / m_FieldWidth;

        float pl = m_pDescriptor->fieldVibrationAmplifierLimit * fp * (1.0f - dp);
        float pr = m_pDescriptor->fieldVibrationAmplifierLimit * fp * dp;

        if (pl < 0.0f)
        {
            pl = 0.0f;
        }
        else if (pl > m_pDescriptor->fieldVibrationAmplifierLimit)
        {
            pl = m_pDescriptor->fieldVibrationAmplifierLimit;
        }

        if (pr < 0.0f)
        {
            pr = 0.0f;
        }
        else if (pr > m_pDescriptor->fieldVibrationAmplifierLimit)
        {
            pr = m_pDescriptor->fieldVibrationAmplifierLimit;
        }

        // 振動の伝わり具合を設定
        auto mod = m_pDescriptor->fieldModulation;
        mod.gainLow  *= pl;
        mod.gainHigh *= pl;
        m_VibrationNodeConnectionRollingL.SetModulation(mod);

        mod = m_pDescriptor->fieldModulation;
        mod.gainLow  *= pr;
        mod.gainHigh *= pr;
        m_VibrationNodeConnectionRollingR.SetModulation(mod);

        // 振動発生
        m_VibrationRolling.Play();
    }
}

void VirtualBall::UpdateHitWall() NN_NOEXCEPT
{
    // 壁衝突の判定
    m_IsHitWall = false;
    // 衝突時の速さから振動させる強さを擬似的に求める
    float hp = m_pDescriptor->weight * m_Velocity * m_Velocity * (1.0f - m_pDescriptor->coefRestitutionWall) / 50.0f;
    m_HitPowerWall = hp;

    if (hp > m_pDescriptor->wallVibrationAmplifierLimit)
    {
        hp = m_pDescriptor->wallVibrationAmplifierLimit;
    }

    if (m_Position > m_FieldHalfWidthEfficient)
    {
        // 右の壁に衝突
        m_Position = m_FieldHalfWidthEfficient;

        if (m_Velocity > 0.0f)
        {
            // 振動の伝わり具合を設定
            m_VibrationNodeConnectionHitWallL.SetModulation(nn::hid::VibrationModulation::Make(0.0f, 1.0f, 0.0f, 1.0f));
            auto modulation = m_pDescriptor->wallModulation;
            modulation.gainLow *= hp;
            modulation.gainHigh *= hp;
            m_VibrationNodeConnectionHitWallR.SetModulation(modulation);

            // 振動発生
            m_VibrationHitWall.Play();
            m_IsHitWall = true;
        }

        // 反発力が動摩擦よりも小さければ停止
        float fw = m_pDescriptor->weight * m_Velocity * m_pDescriptor->coefRestitutionWall;

        if (fw > m_DynamicFriction)
        {
            m_Velocity = (m_DynamicFriction - fw) / m_pDescriptor->weight;
        }
        else
        {
            m_Velocity = 0.0f;
        }
    }
    else if (m_Position < -m_FieldHalfWidthEfficient)
    {
        // 左の壁に衝突
        m_Position = -m_FieldHalfWidthEfficient;

        if (m_Velocity < 0.0f)
        {
            // 振動の伝わり具合を設定
            auto modulation = m_pDescriptor->wallModulation;
            modulation.gainLow *= hp;
            modulation.gainHigh *= hp;
            m_VibrationNodeConnectionHitWallL.SetModulation(modulation);

            m_VibrationNodeConnectionHitWallR.SetModulation(nn::hid::VibrationModulation::Make(0.0f, 1.0f, 0.0f, 1.0f));
            // 振動発生
            m_VibrationHitWall.Play();
            m_IsHitWall = true;
        }

        // 反発力が動摩擦よりも小さければ停止
        float fw = m_pDescriptor->weight * -m_Velocity * m_pDescriptor->coefRestitutionWall;

        if (fw > m_DynamicFriction)
        {
            m_Velocity = (fw - m_DynamicFriction) / m_pDescriptor->weight;
        }
        else
        {
            m_Velocity = 0.0f;
        }
    }
}
