﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NpadController.h"

void NpadControllerBase::InitializeSixAxisSensor() NN_NOEXCEPT
{
    nn::hid::SixAxisSensorHandle handles[SixAxisSensorCountMax];

    m_SixAxisSensorCount = nn::hid::GetSixAxisSensorHandles(
        handles, SixAxisSensorCountMax, m_NpadId, GetStyleSet());

    for (int i = 0; i < m_SixAxisSensorCount; i++)
    {
        SixAxisState& s = m_SixAxisStateArray[i];
        s.sensorHandle = handles[i];
        s.samplingNumber = 0;
        s.gravityVector = nn::util::MakeVector3fType(0.0f, 0.0f, -1.0f);

        nn::hid::StartSixAxisSensor(handles[i]);
    }
}

void NpadControllerBase::InitializeVibrationDevice() NN_NOEXCEPT
{
    // 振動子のハンドルを取得します
    nn::hid::VibrationDeviceHandle handles[VibrationDeviceCountMax];

    m_VibrationDeviceCount = nn::hid::GetVibrationDeviceHandles(
        handles, VibrationDeviceCountMax, m_NpadId, GetStyleSet());

    for (int i = 0; i < m_VibrationDeviceCount; i++)
    {
        VibrationState& v = m_VibrationStateArray[i];
        v.deviceHandle = handles[i];
        v.vibrationPatternId = 0;
        v.currentVibrationValue = nn::hid::VibrationValue::Make();

        // 振動子を初期化します
        nn::hid::InitializeVibrationDevice(v.deviceHandle);

        // 振動子の情報を取得します
        nn::hid::GetVibrationDeviceInfo(&v.deviceInfo, v.deviceHandle);
    }
}

void NpadControllerBase::Initialize() NN_NOEXCEPT
{
    InitializeVibrationDevice();
    InitializeSixAxisSensor();
}

void NpadControllerBase::Update() NN_NOEXCEPT
{
    UpdateButtons();
    UpdateSixAxisSensors();
    m_LoopCount++;
}

bool NpadControllerBase::IsQuitRequired() const NN_NOEXCEPT
{
    // プラスボタンとマイナスボタンが同時に押されていたら終了判定をします
    const nn::hid::NpadButtonSet QuitButtonMask =
        nn::hid::NpadButton::Plus::Mask | nn::hid::NpadButton::Minus::Mask;

    return ((m_Buttons & QuitButtonMask) == QuitButtonMask);
}

void NpadControllerBase::StopVibration() NN_NOEXCEPT
{
    for (int i = 0; i < m_VibrationDeviceCount; i++)
    {
        VibrationState& v = m_VibrationStateArray[i];
        nn::hid::VibrationValue vib = nn::hid::VibrationValue::Make();
        nn::hid::SendVibrationValue(v.deviceHandle, vib);
    }
}

void NpadControllerBase::UpdateSixAxisSensors() NN_NOEXCEPT
{
    for (int i = 0; i < m_SixAxisSensorCount; i++)
    {
        SixAxisState& s = m_SixAxisStateArray[i];
        nn::hid::GetSixAxisSensorState(&(s.sensorState), s.sensorHandle);

        if (s.sensorState.attributes.Test(nn::hid::SixAxisSensorAttribute::IsConnected::Index))
        {
            if (s.samplingNumber != s.sensorState.samplingNumber)
            {
                s.gravityVector.Set(
                    s.sensorState.acceleration.x,
                    s.sensorState.acceleration.y,
                    s.sensorState.acceleration.z
                );
                s.samplingNumber = s.sensorState.samplingNumber;
            }
        }
        else
        {
            s.gravityVector = nn::util::MakeVector3fType(0.0f, 0.0f, -1.0f);
        }
    }
}

void NpadFullKeyController::UpdateButtons() NN_NOEXCEPT
{
    auto style = nn::hid::GetNpadStyleSet(m_NpadId);

    if (style.Test<nn::hid::NpadStyleFullKey>())
    {
        m_IsConnected = true;
        nn::hid::NpadFullKeyState state;
        nn::hid::GetNpadState(&state, m_NpadId);
        m_Buttons = state.buttons;
    }
    else
    {
        if (m_IsConnected)
        {
            StopVibration();
            m_Buttons.Reset();
            m_IsConnected = false;
        }
    }
}

void NpadHandheldController::UpdateButtons() NN_NOEXCEPT
{
    auto style = nn::hid::GetNpadStyleSet(m_NpadId);

    if (style.Test<nn::hid::NpadStyleHandheld>())
    {
        m_IsConnected = true;
        nn::hid::NpadHandheldState state;
        nn::hid::GetNpadState(&state, m_NpadId);
        m_Buttons = state.buttons;
    }
    else
    {
        if (m_IsConnected)
        {
            StopVibration();
            m_Buttons.Reset();
            m_IsConnected = false;
        }
    }
}

void NpadJoyDualController::UpdateButtons() NN_NOEXCEPT
{
    nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(m_NpadId);

    if (style.Test<nn::hid::NpadStyleJoyDual>())
    {
        m_IsConnected = true;
        nn::hid::NpadJoyDualState state;
        nn::hid::GetNpadState(&state, m_NpadId);
        m_Buttons = state.buttons;
    }
    else
    {
        if (m_IsConnected)
        {
            StopVibration();
            m_Buttons.Reset();
            m_IsConnected = false;
        }
    }
}
