﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <cstdio>
#include <nn/nn_Macro.h>

#include "VirtualBall.h"
#include "GraphicsSystem.h"

// デモの種類を表す列挙子
enum DemoEnvironmentPattern
{
    DemoEnvironmentPattern_BouncyBall,
    DemoEnvironmentPattern_Metal,
    DemoEnvironmentPattern_Stone,
    DemoEnvironmentPattern_CountMax
};

// デモの設定を保持するクラス
class Demo
{
    NN_DISALLOW_COPY(Demo);
    NN_DISALLOW_MOVE(Demo);

public:
    static Demo& GetInstance() NN_NOEXCEPT;

    void Initialize() NN_NOEXCEPT
    {
        InitializeEnv();
        InitializeScene();
    }

    void Update() NN_NOEXCEPT
    {
        UpdateEnv();
        UpdateScene();
    }

    void Draw(GraphicsSystem* pGraphicsSystem) const NN_NOEXCEPT
    {
        // 操作説明
        DrawEnv(pGraphicsSystem);

        if (m_IsGraphicsEnabled)
        {
            // オブジェクトの描画
            DrawScene(pGraphicsSystem);
        }
    }

protected:
    Demo() NN_NOEXCEPT {}
    virtual ~Demo() NN_NOEXCEPT {}

    void InitializeEnv() NN_NOEXCEPT;
    void InitializeScene() NN_NOEXCEPT;
    void UpdateScene() NN_NOEXCEPT;
    void UpdateEnv() NN_NOEXCEPT;
    void DrawScene(GraphicsSystem* pGraphicsSystem) const NN_NOEXCEPT;
    void DrawEnv(GraphicsSystem* pGraphicsSystem) const NN_NOEXCEPT;

    void MoveNextEnvironment() NN_NOEXCEPT
    {
        if (m_Pattern < DemoEnvironmentPattern_CountMax - 1)
        {
            int env = (static_cast<int>(m_Pattern) + 1);
            m_Pattern = static_cast<DemoEnvironmentPattern>(env);
            // 物体の設定
            m_Ball.SetDescriptor(GetVirtualBallDesc());
        }
    }

    void MovePreviousEnvironment() NN_NOEXCEPT
    {
        if (m_Pattern > 0)
        {
            int env = (static_cast<int>(m_Pattern) - 1);
            m_Pattern = static_cast<DemoEnvironmentPattern>(env);
            // 物体の設定
            m_Ball.SetDescriptor(GetVirtualBallDesc());
        }
    }

    void AddEnvironment(DemoEnvironmentPattern pattern, const VirtualBallDescriptor* desc, const char* name) NN_NOEXCEPT
    {
        snprintf(m_EnvironmentName[pattern], sizeof(m_EnvironmentName[pattern]) - 1, "%s", name);
        m_Descs[pattern] = desc;
    }

    void ResetEnvironment() NN_NOEXCEPT
    {
        m_Pattern = DemoEnvironmentPattern_BouncyBall;

        // ボールの状態の初期化
        m_Ball.Reset();
        m_Ball.SetDescriptor(GetVirtualBallDesc());
        m_Ball.SetVibrationMixMode(nn::hid::VibrationMixMode_MaxAmplitude);

        m_IsGraphicsEnabled = true;
        m_IsAudioEnabled = true;
        m_IsVibrationEnabled = true;
    }

    const char* GetEnvironmentName() const NN_NOEXCEPT;

    const VirtualBallDescriptor* GetVirtualBallDesc() const NN_NOEXCEPT
    {
        return m_Descs[m_Pattern];
    }

protected:
    DemoEnvironmentPattern m_Pattern = DemoEnvironmentPattern_BouncyBall;
    const VirtualBallDescriptor* m_Descs[DemoEnvironmentPattern_CountMax];
    char m_EnvironmentName[DemoEnvironmentPattern_CountMax][32];

    Sphere      m_Sphere;
    VirtualBall m_Ball;

    // 物体を表示するか否かのフラグ
    bool m_IsGraphicsEnabled = true;

    // 音声を再生するか否かのフラグ
    bool m_IsAudioEnabled = true;

    // 振動を再生するか否かのフラグ
    bool m_IsVibrationEnabled = true;

};

