﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{HidVibrationPlayer.cpp,PageSampleHidVibrationPlayer}

    @brief
    振動ファイルを使って振動子を制御するためのプログラム
 */

/**
    @page PageSampleHidVibrationPlayer 振動ファイルを使った振動子制御
    @tableofcontents

    @brief
    振動ファイルを使って振動子を制御するためのプログラムの解説です。

    @section PageSampleHidVibrationPlayer_SectionBrief 概要
    nn::hid::VibrationPlayer を使って振動ファイルから振動値を読み取る方法、
    nn::hid::VibrationPlayer と nn::hid::VibrationTarget を nn::hid::VibrationNodeConnection で接続することで、コントローラの振動子を制御する方法の説明をします。
    nn::hid::VibrationPlayer を継承した、再生スピード変更に関する機能を追加したクラスがあります。
    nn::hid::VibrationNode を継承したクラスを作る参考にしてください。

    @section PageSampleHidVibrationPlayer_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidVibrationPlayer @endlink 以下にあります。

    @section PageSampleHidVibrationPlayer_SectionNecessaryEnvironment 必要な環境
    Windows 環境で動作させる場合は、事前に PC に Bluetooth ドングルを接続した上で、PC とコントローラをペアリングしてください。
    SDEV/EDEV 環境で動作させる場合は、事前に SDEV/EDEV とコントローラをペアリングしてください。
    1 台のコントローラを無線接続する必要があります。

    @section PageSampleHidVibrationPlayer_SectionHowToOperate 操作方法
    サンプルプログラムを実行して、コントローラのボタンを押してください。
    Bluetooth の接続が確立すると、コントローラの LED が点滅から点灯に変わります。

    コントローラの L or R ボタンを押している際に、振動子に振動値が送信され、コントローラが振動します。
    L と R ではそれぞれことなるパターンの振動が発生します。
    L ボタンには、ループする振動ファイルが割り当てられているので、押している間は常に振動します。
    R ボタンには、単発の振動ファイルが割り当てられているので、押した瞬間に振動が始まります。
    そして、ファイル終端まで再生が終わると振動は止まります。
    両方同時に押すとパターンが重ね合わさります。

    各ボタンを押すと、 nn::hid::VibrationModulation を使用した振動値のリアルタイム編集が可能です。

    - A ボタン：振動の周波数を上げる
    - Y ボタン：振動の周波数を下げる
    - X ボタン：振動の振幅を上げる
    - B ボタン：振動の振幅を下げる

    - 上 ボタン：振動の再生速度を上げる
    - 下 ボタン：振動の再生速度を下げる
    - 左 ボタン：左側に振動をパンニングする
    - 右 ボタン：右側に振動をパンニングする
    ※パンニングをする関係で、最終的に出力される振動の強さが半分になっていることに注意ください。

    画面上に、実際に発生している振動値が出力されます。

    サンプルプログラムを終了させるには + ボタンと - ボタンを同時に押してください。

    @section PageSampleHidVibrationPlayer_SectionPrecaution 注意事項
    コントローラは十分に充電した状態でお使いください。
    コントローラは FullKeyStyle のみに対応しています。

    @section PageSampleHidVibrationPlayer_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。
    TVモードもしくはテーブルモードで実行してください。

    @section PageSampleHidVibrationPlayer_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - NpadID を初期化する
    - NpadID から振動子のハンドルを取得する
    - nn::hid::VibrationPlayer に振動ファイルを読み込む
    - nn::hid::VibrationTarget にハンドルを設定する
    - nn::hid::VibrationPlayer と nn::hid::VibrationTarget を nn::hid::VibrationNodeConnection で接続する
    - ボタンが押されると nn::hid::VibrationPlayer::Play() を実行、離すと nn::hid::VibrationPlayer::Stop() を実行する
    - 別スレッドで nn::hid::VibrationNode を nn::hid::VibrationNode::DefaultVibrationSampleInterval 毎に更新して振動子に振動値を送信する
 */

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_Vibration.h>

#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#include <nn/nn_Windows.h>
#endif

#include "NpadController.h"
#include "GraphicsSystem.h"
#include "VibrationValueDrawer.h"
#include "SpeedChangeableVibrationPlayer.h"

namespace
{
    const int FrameRate = 60;
    const char ProgramName[] = "HidVibrationPlayer Sample";

    const nn::util::Unorm8x4 TextColor = { { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 ActiveColor = { { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 DeactiveColor = { { 32, 32, 32, 255 } };
    const nn::util::Unorm8x4 RedColor = { { 255, 0, 0, 255 } };

    bool g_IsQuitRequired = false;

    GraphicsSystem* g_pGraphicsSystem;
    nn::mem::StandardAllocator* g_pAppAllocator;
    nn::Bit8* g_pAppMemory;

    //!< VibrationPlayerを継承した再生速度を可変にする振動ノード
    nns::hid::SpeedChangeableVibrationPlayer g_PlayerA;
    nns::hid::SpeedChangeableVibrationPlayer g_PlayerB;

    //!< 振動ファイルを格納するメモリ
    const size_t VibrationFileSizeMax = 16 * 1024;
    uint8_t g_FileDataA[VibrationFileSizeMax];
    uint8_t g_FileDataB[VibrationFileSizeMax];
    size_t g_FileSizeA = 0;
    size_t g_FileSizeB = 0;

    //!< 振動ファイルの情報を表示する関数
    void PrintVibrationFileInfo(const nn::hid::VibrationFileInfo& info) NN_NOEXCEPT
    {
        NN_LOG("VibrationFileInfo\n");
        NN_LOG("\tdataSize          : %d\n", info.dataSize);
        NN_LOG("\tformatId          : %d\n", info.formatId);
        NN_LOG("\tisLoop            : %d\n", info.isLoop);
        NN_LOG("\tloopInterval      : %d\n", info.loopInterval);
        NN_LOG("\tloopStartPosition : %d\n", info.loopStartPosition);
        NN_LOG("\tloopEndPosition   : %d\n", info.loopEndPosition);
        NN_LOG("\tmetaDataSize      : %d\n", info.metaDataSize);
        NN_LOG("\tsampleLength      : %d\n", info.sampleLength);
        NN_LOG("\tsamplingRate      : %d\n", info.samplingRate);
        NN_LOG("\n");
    }

    //!< 振動ファイルを読み込む関数
    void ReadVibrationFile(size_t* pOutSize, void* pOutBuffer, const char* filepath) NN_NOEXCEPT
    {
        nn::Result result;
        nn::fs::FileHandle file;
        int64_t filesize;

        result = nn::fs::OpenFile(&file, filepath, nn::fs::OpenMode_Read);
        NN_ASSERT(result.IsSuccess());

        // ファイルサイズを確認する
        result = nn::fs::GetFileSize(&filesize, file);
        *pOutSize = static_cast<size_t>(filesize);
        NN_ASSERT(result.IsSuccess());
        NN_ASSERT(*pOutSize <= VibrationFileSizeMax);

        // データを読み込む
        result = nn::fs::ReadFile(file, 0, pOutBuffer, *pOutSize);
        NN_ASSERT(result.IsSuccess());

        nn::fs::CloseFile(file);

        NN_LOG("Read File(%s) is %s\n", filepath, result.IsSuccess() ? "Success" : "Failed");
    }

    //!< 振動リソースを読み込む関数
    void LoadVibrationResources() NN_NOEXCEPT
    {
        // ファイルから振動データをメモリ上に展開する
        {
            size_t cacheSize = 0;
            nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
            NN_ASSERT(result.IsSuccess());

            char* mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
            NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

            result = nn::fs::MountRom("rom", mountRomCacheBuffer, cacheSize);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            ReadVibrationFile(&g_FileSizeA, g_FileDataA, "rom:/SampleA.bnvib");
            ReadVibrationFile(&g_FileSizeB, g_FileDataB, "rom:/SampleB.bnvib");
            nn::fs::Unmount("rom");
            delete[] mountRomCacheBuffer;
        }

        // 振動データを VibrationPlayer で読み込む
        g_PlayerA.Load(g_FileDataA, g_FileSizeA);
        // 振動ファイル情報の取得
        nn::hid::VibrationFileInfo info = g_PlayerA.GetFileInfo();
        // 振動ファイル情報の表示
        PrintVibrationFileInfo(info);

        g_PlayerB.Load(g_FileDataB, g_FileSizeB);
        info = g_PlayerB.GetFileInfo();
        PrintVibrationFileInfo(info);
    }

    nn::os::TimerEventType  g_TimerEvent;
    const size_t            StackSize = 8 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK char   g_ThreadStack[StackSize];

    //!< 振動処理を行うスレッド
    void VibrationNodeThreadFunction(void *arg) NN_NOEXCEPT
    {
        NN_UNUSED(arg);
        while (!g_IsQuitRequired)
        {
            nn::os::WaitTimerEvent(&g_TimerEvent);
            nn::hid::VibrationNode::Update();
        }
        nn::os::StopTimerEvent(&g_TimerEvent);
    }

    void InitializeGraphics() NN_NOEXCEPT
    {
        // Memory
        g_pAppAllocator = new nn::mem::StandardAllocator();
        const size_t appMemorySize = 128 * 1024 * 1024;
        g_pAppMemory = new nn::Bit8[appMemorySize];
        g_pAppAllocator->Initialize(g_pAppMemory, appMemorySize);

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        const size_t graphicsMemorySize = 256 * 1024 * 1024;
        void* pGraphicsMemory = nns::gfx::GraphicsFramework::DefaultAllocateFunction(graphicsMemorySize, 1, nullptr);
        nv::SetGraphicsAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nns::gfx::GraphicsFramework::DefaultFreeFunction, nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);
        nv::SetGraphicsDevtoolsAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nns::gfx::GraphicsFramework::DefaultFreeFunction, nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);
        nv::InitializeGraphics(pGraphicsMemory, graphicsMemorySize);
#endif
        // Graphics
        g_pGraphicsSystem = new ::GraphicsSystem();
        g_pGraphicsSystem->SetApplicationHeap(g_pAppAllocator);
        g_pGraphicsSystem->Initialize();
    }

    //!< 時系列の振動値を表示する関数
    void DrawTimeSeriesGraph(NpadControllerBase* controllerBase, int ctrlIdx, bool isLow) NN_NOEXCEPT
    {
        int vibrationDeviceCount = controllerBase->GetVibrationDeviceCount();

        float baseX = 30.0f + 600.0f * (ctrlIdx / 4);
        float baseY = 50.0f + 160.0f * (ctrlIdx % 4);

        nn::gfx::util::DebugFontTextWriter* pTextWriter = &g_pGraphicsSystem->GetDebugFont();
        pTextWriter->SetTextColor(controllerBase->IsConnected() ? ActiveColor : DeactiveColor);
        pTextWriter->SetScale(1.0f, 1.0f);
        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->Print("%s : Time Series amplitude%s value", controllerBase->GetName(), isLow ? "Low" : "High");

        for (int vibIdx = 0; vibIdx < vibrationDeviceCount; vibIdx++)
        {
            const VibrationState& v = controllerBase->GetVibrationState(vibIdx);

            float x = baseX + 20.0f + 280.0f * vibIdx;
            float y = baseY + 25.0f;

            pTextWriter->SetTextColor(controllerBase->IsConnected() ? ActiveColor : DeactiveColor);
            pTextWriter->SetScale(0.8f, 0.8f);
            pTextWriter->SetCursor(x, y);

            switch (v.deviceInfo.position)
            {
            case nn::hid::VibrationDevicePosition_Left:
                pTextWriter->Print("Left");
                break;
            case nn::hid::VibrationDevicePosition_Right:
                pTextWriter->Print("Right");
                break;
            default:
                pTextWriter->Print("Unknown");
                break;
            }

            VibrationValueDrawer vibDrawer(pTextWriter, &g_pGraphicsSystem->GetCommandBuffer(), &g_pGraphicsSystem->GetPrimitiveRenderer());
            const VibrationValueBuffer& buffer = controllerBase->GetVibrationValueBuffer(vibIdx);
            vibDrawer.SetPosition(x, y + 15.0f);
            vibDrawer.SetBrightColor(controllerBase->IsConnected());
            vibDrawer.DrawTimeSeriesGraph(buffer, isLow);
        }
    }

    //!< 振動値を表示する関数
    void DrawVibrationValue(NpadControllerBase* controllerBase, int ctrlIdx) NN_NOEXCEPT
    {
        int vibrationDeviceCount = controllerBase->GetVibrationDeviceCount();

        float baseX = 30.0f + 600.0f * (ctrlIdx / 4);
        float baseY = 50.0f + 160.0f * (ctrlIdx % 4);

        nn::gfx::util::DebugFontTextWriter* pTextWriter = &g_pGraphicsSystem->GetDebugFont();
        pTextWriter->SetTextColor(controllerBase->IsConnected() ? ActiveColor : DeactiveColor);
        pTextWriter->SetScale(1.0f, 1.0f);
        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->Print("%s", controllerBase->GetName());

        for (int vibIdx = 0; vibIdx < vibrationDeviceCount; vibIdx++)
        {
            const VibrationState& v = controllerBase->GetVibrationState(vibIdx);

            float x = baseX + 20.0f + 280.0f * vibIdx;
            float y = baseY + 25.0f;

            pTextWriter->SetTextColor(controllerBase->IsConnected() ? ActiveColor : DeactiveColor);
            pTextWriter->SetScale(0.8f, 0.8f);
            pTextWriter->SetCursor(x, y);
            switch (v.deviceInfo.position)
            {
            case nn::hid::VibrationDevicePosition_Left:
                pTextWriter->Print("Left");
                break;
            case nn::hid::VibrationDevicePosition_Right:
                pTextWriter->Print("Right");
                break;
            default:
                pTextWriter->Print("Unknown");
                break;
            }

            VibrationValueDrawer vibDrawer(pTextWriter, &g_pGraphicsSystem->GetCommandBuffer(), &g_pGraphicsSystem->GetPrimitiveRenderer());
            vibDrawer.SetPosition(x, y + 15.0f);
            vibDrawer.SetScale(1.0f);
            vibDrawer.SetBrightColor(controllerBase->IsConnected());
            vibDrawer.DrawVibrationValue(v.actualVibrationValue, true);
        }
    }

    void PrintMessage(nn::gfx::util::DebugFontTextWriter* pTextWriter,
        const float baseX, const float baseY, const float offsetX, const char* key, const char* message) NN_NOEXCEPT
    {
        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->Print("%s", key);
        pTextWriter->SetCursor(baseX + offsetX, baseY);
        pTextWriter->Print(": %s", message);
    }

    void UpdateGraphics(std::vector<NpadControllerBase*> controllers) NN_NOEXCEPT
    {
        nn::gfx::util::DebugFontTextWriter* pTextWriter = &g_pGraphicsSystem->GetDebugFont();
        g_pGraphicsSystem->BeginDraw();
        pTextWriter->Draw(&g_pGraphicsSystem->GetCommandBuffer());

        {
            const float scale = 2.0f;
            pTextWriter->SetScale(scale, scale);
            pTextWriter->SetCursor(10.0f, 10.0f);
            pTextWriter->SetTextColor(TextColor);
            pTextWriter->Print("%s", ProgramName);
        }

        {
            const float scale = 1.5;
            const float offsetX = 80.0f;
            const float offsetY = 30.0f;

            float baseX = 40.0f;
            float baseY = 100.0f;

            pTextWriter->SetScale(scale, scale);

            pTextWriter->SetCursor(baseX, baseY);
            pTextWriter->SetTextColor(TextColor);
            PrintMessage(pTextWriter, baseX, baseY, offsetX, "L", "Play VibrationFileA");
            pTextWriter->SetCursor(baseX + 5 * offsetX, baseY);
            pTextWriter->SetTextColor(g_PlayerA.IsPlaying() ? RedColor : TextColor);
            pTextWriter->Print("%s", g_PlayerA.IsPlaying() ? "Playing" : "Stop");
            baseY += offsetY;

            pTextWriter->SetCursor(baseX, baseY);
            pTextWriter->SetTextColor(TextColor);
            PrintMessage(pTextWriter, baseX, baseY, offsetX, "R", "Play VibrationFileB");
            pTextWriter->SetCursor(baseX + 5 * offsetX, baseY);
            pTextWriter->SetTextColor(g_PlayerB.IsPlaying() ? RedColor : TextColor);
            pTextWriter->Print("%s", g_PlayerB.IsPlaying() ? "Playing" : "Stop");
            baseY += offsetY + 20.0f;

            pTextWriter->SetTextColor(TextColor);
            PrintMessage(pTextWriter, baseX, baseY, offsetX, "A","Increase Frequency");
            baseY += offsetY;
            PrintMessage(pTextWriter, baseX, baseY, offsetX, "Y","Reduce Frequency");
            baseY += offsetY;
            PrintMessage(pTextWriter, baseX, baseY, offsetX, "X","Volume Up");
            baseY += offsetY;
            PrintMessage(pTextWriter, baseX, baseY, offsetX, "B","Volume Down");
            baseY += offsetY + 20.0f;

            PrintMessage(pTextWriter, baseX, baseY, offsetX, "Up", "Speed Up");
            baseY += offsetY;
            PrintMessage(pTextWriter, baseX, baseY, offsetX, "Down", "Speed Down");
            baseY += offsetY;
            PrintMessage(pTextWriter, baseX, baseY, offsetX, "Left", "Pan left");
            baseY += offsetY;
            PrintMessage(pTextWriter, baseX, baseY, offsetX, "Right", "Pan right");
            baseY += offsetY + 20.0f;
        }

        {
            NpadControllerBase* controller = controllers.front();
            int ctrlIdx = 5;
            DrawVibrationValue(controller, ctrlIdx++);
            DrawTimeSeriesGraph(controller, ctrlIdx++, true);
            DrawTimeSeriesGraph(controller, ctrlIdx++, false);
        }

        g_pGraphicsSystem->EndDraw();
        g_pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000L * 1000L * 1000L / FrameRate));
    }

    void FinalizeGraphics() NN_NOEXCEPT
    {
        g_pGraphicsSystem->Finalize();
        delete g_pGraphicsSystem;

        g_pAppAllocator->Finalize();
        delete g_pAppAllocator;

        delete[] g_pAppMemory;
    }

} // anonymous-namespace

extern "C" void nnMain()
{
    NN_LOG("%s Start.\n", ProgramName);

    InitializeGraphics();

    nn::hid::InitializeNpad();

    // 使用する操作形態を設定します
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask);

    // 使用する Npad を設定します
    nn::hid::NpadIdType npadIds[] =
    {
        nn::hid::NpadId::No1,
    };

    nn::hid::SetSupportedNpadIdType(npadIds, NN_ARRAY_SIZE(npadIds));

    // 管理対象の操作形態に関する初期化
    std::vector<NpadControllerBase*> controllers;
    controllers.push_back(new NpadFullKeyController(nn::hid::NpadId::No1, "Id:1 FullKeyStyle"));

    // 管理対象の操作形態に関する初期化
    for (std::vector<NpadControllerBase*>::iterator it = controllers.begin();
        it != controllers.end();
        ++it)
    {
        (*it)->Initialize();
    }

    // メモリ上に振動ファイルのデータを展開する
    LoadVibrationResources();

    nns::hid::SpeedChangeableVibrationPlayer* players[2] = { &g_PlayerA, &g_PlayerB };

    // 管理対象の操作形態に関する初期化
    for (std::vector<NpadControllerBase*>::iterator it = controllers.begin();
        it != controllers.end();
        ++it)
    {
        (*it)->SetVibrationPlayers(players, NN_ARRAY_SIZE(players));
        (*it)->InitializeVibrationNode();
    }

    g_IsQuitRequired = false;

    nn::os::ThreadType  thread;
    nn::Result          result;
    // タイマーイベントを初期化する
    nn::os::InitializeTimerEvent(&g_TimerEvent, nn::os::EventClearMode_AutoClear);
    // スレッドを生成する
    result = nn::os::CreateThread(&thread, VibrationNodeThreadFunction, NULL, g_ThreadStack, StackSize, nn::os::DefaultThreadPriority);
    NN_ASSERT(result.IsSuccess(), "Cannot create thread.");
    // スレッドの実行を開始する
    nn::os::StartThread(&thread);
    // タイマーイベントは周期タイマーイベントとして開始する
    const nn::TimeSpan interval = nn::hid::VibrationNode::DefaultVibrationSampleInterval;
    nn::os::StartPeriodicTimerEvent(&g_TimerEvent, interval, interval);

    while (!g_IsQuitRequired)
    {
        for (std::vector<NpadControllerBase*>::iterator it = controllers.begin();
            it != controllers.end();
            ++it)
        {
            (*it)->Update();

            if ((*it)->IsQuitRequired())
            {
                g_IsQuitRequired = true;
            }
        }

        UpdateGraphics(controllers);
    }

    // 管理対象の操作形態に関する削除
    for (std::vector<NpadControllerBase*>::iterator it = controllers.begin();
        it != controllers.end();
        ++it)
    {
        delete *it;
    }

    // スレッドが終了するのを待機する
    nn::os::WaitThread(&thread);
    // スレッドを破棄する
    nn::os::DestroyThread(&thread);
    // タイマーイベントをファイナライズする
    nn::os::FinalizeTimerEvent(&g_TimerEvent);

    FinalizeGraphics();

    NN_LOG("%s Done\n", ProgramName);
}
