﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cmath>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_Constant.h>
#include <nn/util/util_Arithmetic.h>
#include "DumpedVibrationGenerator.h"
#include "Color.h"

#include <nn/nn_Log.h>

namespace {

    const nn::util::Unorm8x4 MainColor[] = { Color::White, Color::DimGray };
    const nn::util::Unorm8x4 GroupColor[] = { Color::Salmon, Color::Black };
    const nn::util::Unorm8x4 ParameterColor[] = { Color::Red, Color::DimGray };
    const nn::util::Unorm8x4 LineColor[] = { Color::Aqua, Color::DimGray };
    const nn::util::Unorm8x4 PointColor[] = { Color::Red, Color::Black };

    //!< BasicLockable な MutexType を表す構造体です。
    struct LockableMutexType final
    {
        ::nn::os::MutexType _mutex;

        void lock() NN_NOEXCEPT
        {
            ::nn::os::LockMutex(&_mutex);
        }

        void unlock() NN_NOEXCEPT
        {
            ::nn::os::UnlockMutex(&_mutex);
        }
    };

    //!< SpeedChangeableVibrationPlayer 内の排他処理に用いるミューテックス
    LockableMutexType s_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

    const int AxisHeight = 160;
    const int AxisWidth  = 640;
    const int ParaColumnMax = 6;

    const char* ParameterName[ParaColumnMax] =
    {
        "MaxAmplitude", "CurbeMode", "StartFrequency", "EndFrequency", "Time", "Offset",
    };

    const char* CurbeName[] =
    {
        "A", "B", "C", "D",
    };

    float calc(float x, DumpedPattern p)
    {
        float y = 0;
        switch (p)
        {
        case DumpedPattern_A_Curbe:
            y = std::expf(-3.0f * x);
            break;
        case DumpedPattern_B_Curbe:
            y = (1.0f - x);
            break;
        case DumpedPattern_C_Curbe:
            y = nn::util::CosEst(0.5f * nn::util::FloatPi * x);
            break;
        case DumpedPattern_D_Curbe:
            y = 1.0f;
            break;
        default:
            y = 0;
            break;
        }
        return y;
    }
}

const int DumpedVibrationEditer::ColumnMax[] = { 2, 2, 2 };
const int DumpedVibrationEditer::RowMax = NN_ARRAY_SIZE(ColumnMax);

void DumpedVibrationDrawer::Draw(const DumpedVibrationParameter& parameter) const NN_NOEXCEPT
{
    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);

    m_pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    m_pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    m_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    DrawLine(parameter);
    DrawText(parameter);
}

void DumpedVibrationDrawer::DrawText(const DumpedVibrationParameter& parameter) const NN_NOEXCEPT
{
    //文字の描画
    const float x = m_X + 5.0f;
    float y = m_Y + 15.0f;
    //const float w = AxisWidth * m_Scale;
    const float h = AxisHeight * m_Scale;

    int offset = 0;
    const float width = 300.0f * m_Scale;
    const float height = 20.0f * m_Scale;
    const float textFirstPosX = x + 10.0f;
    const float textSecondPosX = textFirstPosX + width;
    m_pWriter->SetTextColor(MainColor[m_ColorPatternId]);
    m_pWriter->SetScale(m_Scale, m_Scale);

    //第１行描画
    if (m_SelectedParameter == 0)
    {
        m_pPrimitiveRenderer->SetColor(GroupColor[m_ColorPatternId]);
        m_pPrimitiveRenderer->Draw2DFrame(m_pCommandBuffer, x, y + h, width, height);
    }
    else if (m_SelectedParameter == 1)
    {
        m_pPrimitiveRenderer->SetColor(GroupColor[m_ColorPatternId]);
        m_pPrimitiveRenderer->Draw2DFrame(m_pCommandBuffer, x + width, y + h, width, height);
    }
    m_pWriter->SetCursor(textFirstPosX, y + h);
    m_pWriter->SetTextColor(m_SelectedParameter == 0 ? ParameterColor[m_ColorPatternId] : MainColor[m_ColorPatternId]);
    m_pWriter->Print("%s:%.2f", ParameterName[offset++], parameter.amplitude);
    m_pWriter->SetCursor(textSecondPosX, y + h);
    m_pWriter->SetTextColor(m_SelectedParameter == 1 ? ParameterColor[m_ColorPatternId] : MainColor[m_ColorPatternId]);
    m_pWriter->Print("%s:%s", ParameterName[offset++], CurbeName[parameter.pattern]);
    y += height;

    //第２行描画
    if (m_SelectedParameter == 2)
    {
        m_pPrimitiveRenderer->SetColor(GroupColor[m_ColorPatternId]);
        m_pPrimitiveRenderer->Draw2DFrame(m_pCommandBuffer, x, y + h, width, height);
    }
    else if (m_SelectedParameter == 3)
    {
        m_pPrimitiveRenderer->SetColor(GroupColor[m_ColorPatternId]);
        m_pPrimitiveRenderer->Draw2DFrame(m_pCommandBuffer, x + width, y + h, width, height);
    }
    m_pWriter->SetCursor(textFirstPosX, y + h);
    m_pWriter->SetTextColor(m_SelectedParameter == 2 ? ParameterColor[m_ColorPatternId] : MainColor[m_ColorPatternId]);
    m_pWriter->Print("%s:%.1fHz", ParameterName[offset++], parameter.freqFirst);
    m_pWriter->SetCursor(textSecondPosX, y + h);
    m_pWriter->SetTextColor(m_SelectedParameter == 3 ? ParameterColor[m_ColorPatternId] : MainColor[m_ColorPatternId]);
    m_pWriter->Print("%s:%.1fHz", ParameterName[offset++], parameter.freqEnd);
    y += height;

    //第３行描画
    if (m_SelectedParameter == 4)
    {
        m_pPrimitiveRenderer->SetColor(GroupColor[m_ColorPatternId]);
        m_pPrimitiveRenderer->Draw2DFrame(m_pCommandBuffer, x, y + h, width, height);
    }
    else if (m_SelectedParameter == 5)
    {
        m_pPrimitiveRenderer->SetColor(GroupColor[m_ColorPatternId]);
        m_pPrimitiveRenderer->Draw2DFrame(m_pCommandBuffer, x + width, y + h, width, height);
    }
    m_pWriter->SetCursor(textFirstPosX, y + h);
    m_pWriter->SetTextColor(m_SelectedParameter == 4 ? ParameterColor[m_ColorPatternId] : MainColor[m_ColorPatternId]);
    m_pWriter->Print("%s:%dms", ParameterName[offset++], parameter.time);
    m_pWriter->SetCursor(textSecondPosX, y + h);
    m_pWriter->SetTextColor(m_SelectedParameter == 5 ? ParameterColor[m_ColorPatternId] : MainColor[m_ColorPatternId]);
    m_pWriter->Print("%s:%dms", ParameterName[offset++], parameter.offset);
    y += height;
}

void DumpedVibrationDrawer::DrawLine(const DumpedVibrationParameter& parameter) const NN_NOEXCEPT
{
    const float x = m_X + 5.0f;
    const float y = m_Y + 5.0f;
    const float w = AxisWidth  * m_Scale;
    const float h = AxisHeight * m_Scale;

    const float scale = 0.75f;
    const float t0 = x;
    const float t1 = t0 + parameter.offset * scale;
    const float t2 = t1 + parameter.time * scale;
    const float a0 = y + h;
    const float a1 = a0 - h * parameter.amplitude;

    //波形の分割数
    const int NumOfPoint = 100;
    float dx[NumOfPoint];
    float dy[NumOfPoint];

    for (int i = 0; i < NumOfPoint; i++)
    {
        float rate = 1.0f * (i + 1) / NumOfPoint;
        dx[i] = t1 + parameter.time * rate * scale;
        dy[i] = a0 - h * parameter.amplitude * calc(rate, parameter.pattern);
    }

    //枠の描画
    {
        m_pPrimitiveRenderer->SetLineWidth(1.0f);
        m_pPrimitiveRenderer->SetColor(MainColor[m_ColorPatternId]);
        m_pPrimitiveRenderer->Draw2DFrame(m_pCommandBuffer, x, y, w, h);
    }

    //線の描画
    {
        m_pPrimitiveRenderer->SetLineWidth(m_ColorPatternId == 0 ? 3.0f : 1.0f);
        m_pPrimitiveRenderer->SetColor(LineColor[m_ColorPatternId]);
        if (m_SelectedParameter < 4)
        {
            m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, t0, a0, t1, a0);

            if (m_SelectedParameter == 0)
            {
                m_pPrimitiveRenderer->SetColor(ParameterColor[m_ColorPatternId]);
            }
            m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, t1, a0, t1, a1);

            if (m_SelectedParameter == 1)
            {
                m_pPrimitiveRenderer->SetColor(ParameterColor[m_ColorPatternId]);
            }
            m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, t1, a1, dx[0], dy[0]);

            for (int i = 0; i < NumOfPoint - 1; i++)
            {
                m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, dx[i], dy[i], dx[i + 1], dy[i + 1] );
            }
            m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, dx[NumOfPoint - 1], dy[NumOfPoint - 1], t2, a0);

            //始点の描画
            if (m_SelectedParameter == 2)
            {
                m_pPrimitiveRenderer->SetColor(PointColor[m_ColorPatternId]);
                m_pPrimitiveRenderer->Draw2DRect(m_pCommandBuffer, t1 - 5.0f, a0 - 5.0f, 10.0f, 10.0f);
            }
            //終点の描画
            else if (m_SelectedParameter == 3)
            {
                m_pPrimitiveRenderer->SetColor(PointColor[m_ColorPatternId]);
                m_pPrimitiveRenderer->Draw2DRect(m_pCommandBuffer, t2 - 5.0f, a0 - 5.0f, 10.0f, 10.0f);
            }
        }
        else
        {
            m_pPrimitiveRenderer->SetColor(m_SelectedParameter == 5 ? ParameterColor[m_ColorPatternId] : LineColor[m_ColorPatternId]);
            m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, t0, a0, t1, a0);
            m_pPrimitiveRenderer->SetColor(m_SelectedParameter == 4 ? ParameterColor[m_ColorPatternId] : LineColor[m_ColorPatternId]);
            m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, t1, a0, t1, a1);
            m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, t1, a1, dx[0], dy[0]);

            for (int i = 0; i < NumOfPoint - 1; i++)
            {
                m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, dx[i], dy[i], dx[i + 1], dy[i + 1]);
            }

            m_pPrimitiveRenderer->Draw2DLine(m_pCommandBuffer, dx[NumOfPoint - 1], dy[NumOfPoint - 1], t2, a0);
        }
    }
}

DumpedVibrationGenerator::DumpedVibrationGenerator() NN_NOEXCEPT
    : m_CurrentTime(0)
    , isPlay(false)
{
    for(int i = 0; i < NumOfSuband; i++)
    {
        m_Parameters[i].amplitude = 0.5f;
        m_Parameters[i].time      = 100;
        m_Parameters[i].offset    = 0;
        m_Parameters[i].freqFirst = 160.0f;
        m_Parameters[i].freqEnd   = 160.0f;
        m_Parameters[i].pattern   = DumpedPattern_A_Curbe;
    }
}

void DumpedVibrationGenerator::OnNextSampleRequired(
    nn::hid::VibrationValue* pValue,
    nn::hid::VibrationNodeConnection::List* pInputConnections) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_Mutex)> lock(s_Mutex);

    NN_ASSERT_NOT_NULL(pValue);
    NN_UNUSED(pInputConnections);
    NN_ASSERT_GREATER_EQUAL(m_CurrentTime, 0);

    if (isPlay)
    {
        float ampl[NumOfSuband];
        float freq[NumOfSuband];

        for (int i = 0; i < NumOfSuband; i++)
        {
            const auto& p = m_Parameters[i];
            const int t0 = 0;
            const int t1 = t0 + p.offset;

            if (m_CurrentTime < t1)
            {
                ampl[i] = 0.0f;
                freq[i] = p.freqFirst;
            }
            else
            {
                float x = 1.0f * (m_CurrentTime - t1) / p.time;

                if (x > 1.0f)
                {
                    x = 1.0f;
                }

                ampl[i] = p.amplitude * calc(x, p.pattern);
                freq[i] = p.freqFirst * (1.0f - x) + p.freqEnd * x;
            }
        }

        *pValue = nn::hid::VibrationValue::Make(ampl[0], freq[0], ampl[1], freq[1]);

        m_CurrentTime += 5;

        if (m_Parameters[0].time + m_Parameters[0].offset <= m_CurrentTime && m_Parameters[1].time + m_Parameters[1].offset <= m_CurrentTime)
        {
            isPlay = false;
        }
    }
    else
    {
        *pValue = nn::hid::VibrationValue::Make();
    }
}
