﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/hid/hid_NpadFullKey.h>
#include <nn/hid/hid_NpadJoyDual.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>

#include <nn/hid/system/hid_NpadSystem.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/system/hid_SixAxisSensorCalibration.h>
#include <nn/hid/system/hid_UniquePad.h>

#include "HidSixAxisSensorCalibration_InitialState.h"
#include "HidSixAxisSensorCalibration_State.h"

namespace
{
    ::nn::Result ExtractFeatures(::nn::hid::SixAxisSensorHandle* pOutHandle,
                                 ::nn::hid::system::UniqueSixAxisSensorHandle* pOutUniquePadHandle,
                                 ::nn::hid::system::UniquePadId* pOutId,
                                 const ::nn::hid::NpadButtonSet& buttons,
                                 const ::nn::hid::NpadIdType& npadId) NN_NOEXCEPT
    {
        ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax] = {};
        int uniquePadIdCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                                 sizeof(uniquePadIds) / sizeof(uniquePadIds[0]));

        for (int i = 0; i < uniquePadIdCount; i++)
        {
            int uniquePadControllerNumber;
            ::nn::hid::system::UniquePadInterface uniquePadInterface;
            ::nn::hid::system::UniquePadType uniquePadType;

            NN_RESULT_DO(::nn::hid::system::GetUniquePadControllerNumber(&uniquePadControllerNumber,
                                                                         uniquePadIds[i]));
            NN_RESULT_DO(::nn::hid::system::GetUniquePadInterface(&uniquePadInterface,
                                                                  uniquePadIds[i]));
            uniquePadType = ::nn::hid::system::GetUniquePadType(uniquePadIds[i]);

            if (uniquePadControllerNumber - 1 == npadId)
            {
                if ((uniquePadType == ::nn::hid::system::UniquePadType_Embedded) ||
                    (uniquePadType == ::nn::hid::system::UniquePadType_FullKeyController))
                {
                    ::nn::hid::SixAxisSensorHandle handle = {};
                    int count = ::nn::hid::GetSixAxisSensorHandles(&handle,
                                                                   1,
                                                                   npadId,
                                                                   ::nn::hid::NpadStyleFullKey::Mask);
                    NN_ASSERT_EQUAL(1, count);

                    *pOutHandle = handle;
                }
                else if (uniquePadType == ::nn::hid::system::UniquePadType_RightController &&
                         buttons.Test<::nn::hid::NpadButton::A>())
                {
                    ::nn::hid::SixAxisSensorHandle handles[2] = {};
                    int count = ::nn::hid::GetSixAxisSensorHandles(handles,
                                                                   2,
                                                                   npadId,
                                                                   ::nn::hid::NpadStyleJoyDual::Mask);
                    NN_ASSERT_EQUAL(2, count);
                    *pOutHandle = handles[1];
                }
                else if (uniquePadType == ::nn::hid::system::UniquePadType_LeftController &&
                         buttons.Test<::nn::hid::NpadButton::Right>())
                {
                    ::nn::hid::SixAxisSensorHandle handles[2] = {};
                    int count = ::nn::hid::GetSixAxisSensorHandles(handles,
                                                                   2,
                                                                   npadId,
                                                                   ::nn::hid::NpadStyleJoyDual::Mask);
                    NN_ASSERT_EQUAL(2, count);
                    *pOutHandle = handles[0];
                }
                else
                {
                    continue;
                }

                // UniqueSixAxisSensorHandle を取得
                ::nn::hid::system::UniqueSixAxisSensorHandle handles[::nn::hid::system::UniqueSixAxisSensorHandleCountMax];

                int handleCount = ::nn::hid::system::ListSixAxisSensorHandles(handles,
                                                                              uniquePadIds[i],
                                                                              ::nn::hid::system::UniqueSixAxisSensorHandleCountMax);

                NN_ABORT_UNLESS_EQUAL(1, handleCount);

                *pOutUniquePadHandle = handles[0];
                *pOutId = uniquePadIds[i];

                NN_RESULT_SUCCESS;
            }
        }
        NN_LOG("No available pads\n");
        return ::nn::hid::system::ResultUniquePadNotAvailable();
    }

    // フォーカスしているコントローラの特徴量を抽出します。
    bool GetFocusedPadFeatures(FocusedPadFeatures* pOutFeatures) NN_NOEXCEPT
    {
        const ::nn::hid::NpadIdType NpadIds[] = {
            ::nn::hid::NpadId::No1,
            ::nn::hid::NpadId::No2,
            ::nn::hid::NpadId::No3,
            ::nn::hid::NpadId::No4,
            ::nn::hid::NpadId::No5,
            ::nn::hid::NpadId::No6,
            ::nn::hid::NpadId::No7,
            ::nn::hid::NpadId::No8,
            ::nn::hid::NpadId::Handheld,
        };

        nn::hid::system::NpadSystemState state = {};
        for (const auto& npadId : NpadIds)
        {
            ::nn::hid::system::GetNpadState(&state, npadId);
            if (state.buttons.Test<nn::hid::NpadButton::A>() ||
                state.buttons.Test<nn::hid::NpadButton::Right>())
            {
                NN_LOG("[NpadId = %x] A or Right button pressed\n", npadId);

                ::nn::hid::SixAxisSensorHandle handle;
                ::nn::hid::system::UniqueSixAxisSensorHandle uniquePadHandle;
                ::nn::hid::system::UniquePadId uniquePadId;

                auto result = ExtractFeatures(&handle,
                                              &uniquePadHandle,
                                              &uniquePadId,
                                              state.buttons,
                                              npadId);
                if (result.IsFailure())
                {
                    return false;
                }

                pOutFeatures->buttons = state.buttons;
                pOutFeatures->handle = handle;
                pOutFeatures->npadId = npadId;
                pOutFeatures->uniquePadHandle = uniquePadHandle;
                pOutFeatures->uniquePadId = uniquePadId;

                return true;
            }
        }
        return false;
    }


} // namespace

// InitialState クラス
InitialState::InitialState() NN_NOEXCEPT
{
    // 何もしない
}

InitialState::~InitialState() NN_NOEXCEPT
{
    // 何もしない
}

void InitialState::Update(Context* pContext) NN_NOEXCEPT
{
    FocusedPadFeatures features = {};

    if (GetFocusedPadFeatures(&features))
    {
        // サポート有無のチェック
        if (::nn::hid::system::IsSixAxisSensorUserCalibrationSupported(features.uniquePadHandle) == false)
        {
            NN_LOG("[%s] Calibration Not Supported. Go back to InitialState...\n", __FUNCTION__);
            pContext->SetState(&State::s_InitialState);
            return;
        }

        pContext->SetFocusedPadFeatures(features);
        pContext->SetState(&State::s_StabilizingState);
        return;
    }
}

void InitialState::Enter(Context* pContext) NN_NOEXCEPT
{
    NN_LOG("[%s] InitialState\n", __FUNCTION__);
}

void InitialState::Exit(Context* pContext) NN_NOEXCEPT
{
    NN_LOG("[%s] InitialState\n", __FUNCTION__);
}
