﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>

#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_SixAxisSensorAccurateCalibration.h>

#include "HidSixAxisSensorAccurateCalibration_Context.h"
#include "HidSixAxisSensorAccurateCalibration_State.h"

namespace {

template <typename TState>
::nn::hid::NpadButtonSet GetNpadButtons(nn::hid::NpadIdType id)
{
    TState state = {};
    ::nn::hid::GetNpadState(&state, id);
    return state.buttons;
}

::nn::hid::NpadButtonSet GetButtons() NN_NOEXCEPT
{
    ::nn::hid::NpadButtonSet buttons = {};
    buttons.Reset();

    // トリガー入力を取得
    buttons |= GetNpadButtons<::nn::hid::NpadFullKeyState>(::nn::hid::NpadId::No1);
    buttons |= GetNpadButtons<::nn::hid::NpadJoyDualState>(::nn::hid::NpadId::No1);
    buttons |= GetNpadButtons<::nn::hid::NpadHandheldState>(::nn::hid::NpadId::Handheld);

    return buttons;
}

} // namespace

Context::Context() NN_NOEXCEPT
    : m_pState(&State::s_InitialState)
    , m_LastButtons()
    , m_CurrentButtons()
{
    // 何もしない
}

void Context::Update() NN_NOEXCEPT
{
    m_LastButtons = m_CurrentButtons;
    m_CurrentButtons = GetButtons();

    m_pState->Update(this);
}

void Context::SetState(IState* pState) NN_NOEXCEPT
{
    if (m_pState)
    {
        m_pState->Exit(this);
    }

    m_pState = pState;

    m_pState->Enter(this);
}

void Context::Initialize() NN_NOEXCEPT
{
    const ::nn::hid::NpadIdType NpadIds[] = {
        ::nn::hid::NpadId::No1,
        ::nn::hid::NpadId::Handheld,
    };

    const ::nn::hid::NpadStyleSet NpadStyleSet = (
        ::nn::hid::NpadStyleFullKey::Mask |
        ::nn::hid::NpadStyleJoyDual::Mask |
        ::nn::hid::NpadStyleHandheld::Mask
    );

    ::nn::hid::SetSupportedNpadIdType(NpadIds, NN_ARRAY_SIZE(NpadIds));
    ::nn::hid::SetSupportedNpadStyleSet(NpadStyleSet);

    // 注意:
    // UniquePad に本体が列挙されず、適切な handle を取得する実装が入っていないため暫定対処。
    // StartSixAxisSensorAccurateUserCalibration() は形式的に handle を引数としますが、
    // handle に依らずキャリブレーション処理はすべて対象を本体とします。
    m_UniqueSixAxisSensorHandle = ::nn::hid::system::UniqueSixAxisSensorHandle();

    // サポート有無を確認
    if (!::nn::hid::system::IsSixAxisSensorAccurateUserCalibrationSupported(m_UniqueSixAxisSensorHandle))
    {
        // NN_ABORT("Accurate User Calibration for SixAxisSensor is not supported.\n");
    }

}

::nn::hid::NpadButtonSet Context::GetTriggerButtons() const NN_NOEXCEPT
{
    return (m_CurrentButtons ^ m_LastButtons) & m_CurrentButtons;
}

::nn::hid::system::UniqueSixAxisSensorHandle Context::GetUniqueSixAxisSensorHandle() const NN_NOEXCEPT
{
    return m_UniqueSixAxisSensorHandle;
}

void Context::PrintCalibrationStatus() const NN_NOEXCEPT
{
    ::nn::hid::system::SixAxisSensorAccurateUserCalibrationState state;

    const auto Result = ::nn::hid::system::GetSixAxisSensorAccurateUserCalibrationState(&state,
                                                                                        m_UniqueSixAxisSensorHandle);
    if (Result.IsFailure())
    {
        NN_ABORT("Calibration Failure\n");
    }

    NN_LOG("Calibrating : %x, %f\n", state.stage, state.progress);

    switch (state.stage)
    {
    case ::nn::hid::system::SixAxisSensorAccurateUserCalibrationStage_InitialStage:
        // Add Instructions...
        break;
    case ::nn::hid::system::SixAxisSensorAccurateUserCalibrationStage_Completed:
        // Add Instructions...
        break;
    default:
        break;
    }
}
