﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include "HidNpadIntegrate_PlatformDependentPluginBase.h"
#include "HidNpadIntegrate_PluginBase.h"
#include "HidNpadIntegrate_ScenePluginBase.h"
#include "HidNpadIntegrate_SettingApiPluginBase.h"
#include "HidNpadIntegrate_StylePluginBase.h"

class PluginManager
{
    NN_DISALLOW_COPY(PluginManager);
    NN_DISALLOW_MOVE(PluginManager);

public:
    PluginManager() NN_NOEXCEPT : m_Mutex(false){}

    void AddPlugin(ScenePluginBase* pSceneBase) NN_NOEXCEPT
    {
        m_Mutex.Lock();
        m_SceneList.push_back(pSceneBase);
        m_Mutex.Unlock();
    }

    void AddPluginToHead(ScenePluginBase* pSceneBase) NN_NOEXCEPT
    {
        m_Mutex.Lock();
        m_SceneList.insert(m_SceneList.begin(), pSceneBase);
        m_Mutex.Unlock();
    }

    void AddPlugin(NpadStylePluginBase* pStyleBase) NN_NOEXCEPT
    {
        m_Mutex.Lock();
        m_StyleList.push_back(pStyleBase);
        m_Mutex.Unlock();
    }

    void AddPluginToHead(NpadStylePluginBase* pStyleBase) NN_NOEXCEPT
    {
        m_Mutex.Lock();
        m_StyleList.insert(m_StyleList.begin(), pStyleBase);
        m_Mutex.Unlock();
    }

    void AddPlugin(PlatformDependentPluginBase* pPlatformDependent) NN_NOEXCEPT
    {
        m_Mutex.Lock();
        m_PlatformDependentList.push_back(pPlatformDependent);
        m_Mutex.Unlock();
    }

    void AddPluginToHead(PlatformDependentPluginBase* pPlatformDependent) NN_NOEXCEPT
    {
        m_Mutex.Lock();
        m_PlatformDependentList.insert(m_PlatformDependentList.begin(), pPlatformDependent);
        m_Mutex.Unlock();
    }

    void AddPlugin(SettingApiPluginBase* pSettingApi) NN_NOEXCEPT
    {
        m_Mutex.Lock();
        m_SettingApiList.push_back(pSettingApi);
        m_Mutex.Unlock();
    }

    void AddPluginToHead(SettingApiPluginBase* pSettingApi) NN_NOEXCEPT
    {
        m_Mutex.Lock();
        m_SettingApiList.insert(m_SettingApiList.begin(), pSettingApi);
        m_Mutex.Unlock();
    }

    void InitializeScenePlugin(ApplicationHeap* pAppAllocator,
                                GraphicsSystem*  pGraphicsSystem) NN_NOEXCEPT
    {
        for (std::vector<ScenePluginBase*>::iterator it = m_SceneList.begin();
             it != m_SceneList.end();
             ++it)
        {
            (*it)->InitializeScene(pAppAllocator, pGraphicsSystem);
        }
    }

    void FinalizeScenePlugin(ApplicationHeap* pAppAllocator) NN_NOEXCEPT
    {
        for (std::vector<ScenePluginBase*>::reverse_iterator it = m_SceneList.rbegin();
             it != m_SceneList.rend();
             ++it)
        {
            (*it)->FinalizeScene(pAppAllocator);
            delete (*it);
        }
        m_SceneList.clear();
    }

    void InitializeNpadPlugin() NN_NOEXCEPT
    {
        for (std::vector<NpadStylePluginBase*>::iterator it =
                m_StyleList.begin();
            it != m_StyleList.end();
            ++it)
        {
            (*it)->Initialize();
        }
    }

    void FinalizeNpadPlugin() NN_NOEXCEPT
    {
        for (std::vector<NpadStylePluginBase*>::reverse_iterator it =
                m_StyleList.rbegin();
            it != m_StyleList.rend();
            ++it)
        {
            (*it)->Finalize();
            delete (*it);
        }
        m_StyleList.clear();
    }

    void InitializePlatformPlugin() NN_NOEXCEPT
    {
        for (std::vector<PlatformDependentPluginBase*>::iterator it =
                m_PlatformDependentList.begin();
            it != m_PlatformDependentList.end();
            ++it)
        {
            (*it)->Initialize();
        }
    }

    void FinalizePlatformPlugin() NN_NOEXCEPT
    {
        for (std::vector<PlatformDependentPluginBase*>::reverse_iterator it =
                m_PlatformDependentList.rbegin();
            it != m_PlatformDependentList.rend();
            ++it)
        {
            (*it)->Finalize();
            delete (*it);
        }
        m_PlatformDependentList.clear();
    }

    void InitializeSettingApiPlugin() NN_NOEXCEPT
    {
        for (std::vector<SettingApiPluginBase*>::iterator it =
                m_SettingApiList.begin();
            it != m_SettingApiList.end();
            ++it)
        {
            (*it)->Initialize();
        }
    }

    void FinalizeSettingApiPlugin() NN_NOEXCEPT
    {
        for (std::vector<SettingApiPluginBase*>::reverse_iterator it =
                m_SettingApiList.rbegin();
            it != m_SettingApiList.rend();
            ++it)
        {
            (*it)->Finalize();
            delete (*it);
        }
        m_SettingApiList.clear();
    }

    void StopScenePlugin() NN_NOEXCEPT
    {
        for (std::vector<ScenePluginBase*>::iterator it = m_SceneList.begin();
             it != m_SceneList.end();
             ++it)
        {
            (*it)->StopScene();
        }
    }

    void RestartScenePlugin() NN_NOEXCEPT
    {
        for (std::vector<ScenePluginBase*>::iterator it = m_SceneList.begin();
             it != m_SceneList.end();
             ++it)
        {
            (*it)->RestartScene();
        }
    }

    int GetScenePluginCount() NN_NOEXCEPT
    {
        return static_cast<int>(m_SceneList.size());
    }

    ScenePluginBase* GetScene(int cnt) NN_NOEXCEPT
    {
        return m_SceneList[cnt];
    }

    int GetCurrentSceneNum() NN_NOEXCEPT
    {
        return m_CurrentActiveScene;
    }

    int GetOldSceneNum() NN_NOEXCEPT
    {
        return m_OldActiveScene;
    }

    void NextScene() NN_NOEXCEPT
    {
        m_OldActiveScene = m_CurrentActiveScene;
        m_CurrentActiveScene++;

        if(m_CurrentActiveScene == GetScenePluginCount())
        {
            m_CurrentActiveScene = 0;
        }
    }

    void PreviousScene() NN_NOEXCEPT
    {
        m_OldActiveScene = m_CurrentActiveScene;
        m_CurrentActiveScene--;

        if(m_CurrentActiveScene < 0)
        {
            m_CurrentActiveScene = GetScenePluginCount() - 1;
        }
    }

    nn::hid::NpadStyleSet GetAvailableNpadStyleSet() NN_NOEXCEPT
    {
        nn::hid::NpadStyleSet mask = {};

        for (std::vector<NpadStylePluginBase*>::iterator it =
                m_StyleList.begin();
            it != m_StyleList.end();
            ++it)
        {
            mask |= (*it)->GetNpadStyleSet();
        }

        return mask;
    }

    NpadStylePluginBase* GetEnableNpad(nn::hid::NpadStyleSet style) NN_NOEXCEPT
    {
        for (std::vector<NpadStylePluginBase*>::iterator it =
                m_StyleList.begin();
            it != m_StyleList.end();
            ++it)
        {
            if ((*it)->IsEnableNpadStyle(style))
            {
                return (*it);
            }
        }
        return NULL;
    }

    int GetNpadStylePluginCount() NN_NOEXCEPT
    {
        return static_cast<int>(m_StyleList.size());
    }

    NpadStylePluginBase* GetNpadStyle(int cnt) NN_NOEXCEPT
    {
        return m_StyleList[cnt];
    }

    int GetPlatformDependentPluginCount() NN_NOEXCEPT
    {
        return static_cast<int>(m_PlatformDependentList.size());
    }

    PlatformDependentPluginBase* GetPlatformDependent(int cnt) NN_NOEXCEPT
    {
        return m_PlatformDependentList[cnt];
    }

    int GetSettingApiPluginCount() NN_NOEXCEPT
    {
        return static_cast<int>(m_SettingApiList.size());
    }

    SettingApiPluginBase* GetSettingApi(int cnt) NN_NOEXCEPT
    {
        return m_SettingApiList[cnt];
    }

private:
    std::vector<ScenePluginBase*> m_SceneList;
    std::vector<NpadStylePluginBase*> m_StyleList;
    std::vector<PlatformDependentPluginBase*> m_PlatformDependentList;
    std::vector<SettingApiPluginBase*> m_SettingApiList;

    nn::os::Mutex m_Mutex;
    int m_OldActiveScene;
    int m_CurrentActiveScene;

};

PluginManager& GetPluginManager() NN_NOEXCEPT;

#define SET_PLUGIN(plugin, name) \
class plugin##Plugin \
{ \
public: \
    plugin##Plugin () NN_NOEXCEPT \
    { \
        plugin* pPlugin = new plugin(); \
        pPlugin->SetName(name);\
        GetPluginManager().AddPlugin(pPlugin); \
    }; \
}; \
plugin##Plugin g_##plugin##Plugin \

