﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{HidControllerSequence_Game.cpp,PageSampleHidControllerSequence_Game}

    @brief
    サンプルアプリケーションのゲーム画面でのコントローラーサポートアプレットの呼び出し方
*/

/**
    @page PageSampleHidControllerSequence_Game ゲーム画面でのコントローラーサポートアプレットの呼び出し方
    @tableofcontents

    @brief
    サンプルアプリケーションのゲーム画面においてのコントローラーサポートアプレットの呼び出し方の解説です

    @section PageSampleHidControllerSequence_Game_SectionBrief 概要
    サンプルアプリケーションのゲーム画面においてのコントローラーサポートアプレットの呼び出し方の解説です

    @section PageSampleHidControllerSequence_Game_SectionFileStructure ファイル構成
    本サンプルプログラムは
    @link ../../../Samples/Sources/Applications/HidControllerSequence
    Samples/Sources/Applications/HidControllerSequence @endlink 以下にあります。

    @section PageSampleHidControllerSequence_Game_SectionNecessaryEnvironment 必要な環境
    事前に実機とコントローラーをペアリングしてください。

    @section PageSampleHidControllerSequence_Game_SectionHowToOperate 操作方法
    左右方向ボタンもしくはスティックを倒す事でプレイヤーが移動します。@n
    [A]を押下げるとプレイヤーがジャンプし、[B]を押下げるとプレイヤーが加速します。@n
    ([Plus] or [Minus]) を押すと中断ダイアログが開きます。@n
    2人以上でプレイしている場合、中断ダイアログの操作選択権はダイアログを開いたプレイヤーのみとなります。

    @section PageSampleHidControllerSequence_Game_SectionPrecaution 注意事項
    コントローラーは十分に充電した状態でお使いください。

    @section PageSampleHidControllerSequence_Game_SectionHowToExecute 実行手順
    サンプルアプリケーションをビルドし、実行してください。

    @section PageSampleHidControllerSequence_Game_SectionDetail 解説

    ゲーム画面での処理の流れは以下の通りです
    - コントローラーの接続確認
    - 接続確認ダイアログの表示
    - プレイヤーのコントローラーの入力確認
    - 中断ダイアログの表示
    - タイトル画面に移動

    シングルプレイ時
    @image html Applications\HidControllerSequence\HidControllerSequence_Game_Single.png

    シングルプレイ時は以下の場合にコントローラーサポートアプレットが呼び出されます。
    - マスタープレイヤーのコントローラーが消失
    - 新しい無線コントローラーの接続を検知
    - 異なるコントローラーからの入力を検知

    @image html Applications\HidControllerSequence\HidControllerSequence_Single.png

    マルチプレイ時
    @image html Applications\HidControllerSequence\HidControllerSequence_Game_Multi.png

    マルチプレイ時は以下の場合にコントローラーサポートアプレットが呼び出されます。
    - 操作プレイヤーのコントローラーが消失

    マルチプレイ時では、マージンを設けずプレイヤーのコントローラーの切断を検知した場合、@n
    即時コントローラーサポートアプレットを呼び出します。

*/

#include "./HidControllerSequence_Game.h"

const nns::hid::ButtonSet Player::MoveButtonSet = nns::hid::Button::Left::Mask | nns::hid::Button::Right::Mask;
const float Player::MaxSpeed = 16.f;
const nn::util::Float4 Player::MoveArea = NN_UTIL_FLOAT_4_INITIALIZER(0.f, 0.f, 1280.f, 656.f);

Player::Player(const PlayerState& state) :
    m_FrameCount(0),
    m_State(state),
    m_isShowInfo(false)
{
    m_Accel = NN_UTIL_FLOAT_2_INITIALIZER(0.f, 0.f);
    m_Scale = NN_UTIL_FLOAT_2_INITIALIZER(1.f, 1.f);
}

void Player::Update()
{
    m_isShowInfo = false;

    m_State.Style = nn::hid::GetNpadStyleSet(m_State.Id);

    m_State.Pos.x += m_Accel.x;
    m_State.Pos.y += m_Accel.y;
    m_Accel.x *= 0.88f;             // 摩擦
    m_Accel.y *= 0.98f;             // 空気抵抗
    m_Accel.y += 0.98f;             // 重力

    m_AnimeCounter.CurFrameCount = m_FrameCount - m_AnimeCounter.StartFrameCount;

    if (m_State.Pos.y < MoveArea.w - m_State.Size.y - 0.5f)
    {
        // 着地
        m_AnimeCounter.IsStart = false;
        m_Scale = nn::util::MakeFloat2(1.f, 1.f);
    }
    else
    {
        if (m_AnimeCounter.IsStart)
        {
            if (m_AnimeCounter.CurFrameCount < 30)
            {
                const float baseScale = 1.f - (static_cast<float>(m_AnimeCounter.CurFrameCount) / 30.f);
                float scale = ((nn::util::SinEst(nn::util::DegreeToRadian(static_cast<float>(m_AnimeCounter.CurFrameCount * 36.0)))) / 5.f) * baseScale;
                m_Scale = nn::util::MakeFloat2( scale + 1.f, -scale + 1.f);
            }
            else
            {
                m_Scale = nn::util::MakeFloat2(1.f, 1.f);
            }
        }
        else
        {
            m_AnimeCounter.IsStart = true;
            m_AnimeCounter.StartFrameCount = m_FrameCount;
            m_Scale = nn::util::MakeFloat2(1.f, 1.f);
        }
    }

    if (m_State.Pos.x < MoveArea.x || m_State.Pos.x + m_State.Size.x >= MoveArea.z)
    {
        m_Accel.x = 0.f;
        if (m_State.Pos.x < MoveArea.x) { m_State.Pos.x = MoveArea.x; }
        if (m_State.Pos.x + m_State.Size.x >= MoveArea.z) { m_State.Pos.x = MoveArea.z - m_State.Size.x; }
    }
    if (m_State.Pos.y < MoveArea.y) { m_State.Pos.y = MoveArea.y; }
    if (m_State.Pos.y + m_State.Size.y >= MoveArea.w) { m_State.Pos.y = MoveArea.w - m_State.Size.y; }

    ++m_FrameCount;
}

void Player::Draw(nn::gfx::CommandBuffer* command, nns::gfx::PrimitiveRenderer::Renderer* renderer)
{
    renderer->SetColor(m_State.Color);
    renderer->Draw2DRect(command,
        m_State.Pos.x + (m_State.Size.x - m_State.Size.x * m_Scale.x) / 2.f,
        m_State.Pos.y + (m_State.Size.y - m_State.Size.y * m_Scale.y),
        m_State.Size.x * m_Scale.x,
        m_State.Size.y * m_Scale.y);

    const uint8_t invWeight =
        static_cast<uint8_t>(0.298912f * m_State.Color.GetR()) +
        static_cast<uint8_t>(0.586611f * m_State.Color.GetG()) +
        static_cast<uint8_t>(0.114478f * m_State.Color.GetB());

    const nn::util::Color4u8 color = invWeight < 128 ? nn::util::Color4u8::White() : nn::util::Color4u8::Black();

    renderer->SetColor(color);

    for (int32_t styleIndex = 0; styleIndex < m_State.Style.GetCount(); ++styleIndex)
    {
        if (m_State.Style.Test(styleIndex))
        {
            nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(m_State.Pos.x + (m_State.Size.x - m_State.Size.x * m_Scale.x) / 2.f, m_State.Pos.y + (m_State.Size.y - m_State.Size.y * m_Scale.y));
            nn::util::Float2 size = NN_UTIL_FLOAT_2_INITIALIZER(m_State.Size.x * m_Scale.x, m_State.Size.y * m_Scale.y);

            DrawStyleIcon(pos, size, styleIndex);
            break;
        }
    }
}

void Player::DrawInfo()
{
    m_isShowInfo = true;
    nn::util::Float2 drawPos = NN_UTIL_FLOAT_2_INITIALIZER(0.f, MoveArea.w + 24.f);

    for (size_t i = 0; i < NN_ARRAY_SIZE(ApplicationState::NpadIds); ++i)
    {
        if (ApplicationState::NpadIds[i] == m_State.Id)
        {
            int32_t npadIndex = (ApplicationState::NpadIds[i] == nn::hid::NpadId::Handheld) ? 0 : static_cast<int32_t>(i);
            drawPos.x = static_cast<float>(npadIndex) * 210.f + 300.f;
            g_pGraphicsSystem->GetDebugFont().SetCursor(drawPos.x, drawPos.y);
            g_pGraphicsSystem->GetDebugFont().SetTextColor(m_State.Color);
            g_pGraphicsSystem->GetDebugFont().Print("■ Player %d", npadIndex + 1);
        }
    }
}

PlayerState Player::GetState()
{
    return m_State;
}

void Player::SetNpadId(nn::hid::NpadIdType id)
{
    m_State.Id = id;
}

void Player::Move(const nns::hid::Controller& controller)
{
    const nns::hid::ButtonSet& button = controller.GetButtons();
    const nn::util::Float2& stick = controller.GetLeftStick();

    float speed             = 1.f;

    // Bボタンで加速
    if (button.Test<nns::hid::Button::B>())
    {
        speed = (m_State.Pos.y < MoveArea.w - m_State.Size.y) ? 1.25f : 2.f;
    }

    // 移動
    if ((button & MoveButtonSet).IsAnyOn() &&
        !(m_State.Style.Test<nn::hid::NpadStyleJoyLeft>() || m_State.Style.Test<nn::hid::NpadStyleJoyRight>()))
    {
        // ボタン入力時の処理
        speed *= button.Test<nns::hid::Button::Right>() ? MaxSpeed / 10.f : -MaxSpeed / 10.f;
        m_Accel.x += speed;
    }
    else
    {
        // スティック入力時の処理
        speed *= (stick.x * stick.x * stick.x * MaxSpeed) / 10.f;
        m_Accel.x += speed;
    }

    // ジャンプ処理
    if (m_State.Pos.y > (MoveArea.w - m_State.Size.y - 0.5f))
    {
        // 接地中
        if (controller.GetButtonsDown().Test<nns::hid::Button::A>())
        {
            m_Accel.y = -MaxSpeed * 2.f;
        }
    }
    else
    {
        // 空中
        if (controller.GetButtons().Test<nns::hid::Button::A>())
        {
            if ((m_Accel.y < 0))
            {
                // 上昇中にAを押している場合
                m_Accel.y -= 0.49f;
            }
        }
    }
}

GameSceneThread::GameSceneThread() : ThreadState()
{

}

GameSceneThread::GameSceneThread(const char* name, nn::os::ThreadFunction func) : ThreadState(name, func)
{
}

void GameSceneThread::Initialize()
{
    //==============================================
    // 前処理
    //==============================================
    // グラフィックス 関連の機能
    m_pRenderer     = &g_pGraphicsSystem->GetPrimitiveRenderer();
    m_pTextWriter   = &g_pGraphicsSystem->GetDebugFont();
    m_pCommand      = &g_pGraphicsSystem->GetCommandBuffer();

    m_IsExit        = false;

    // ダイアログの初期化
    InitializeDialog();

    // プレイ人数が一人の場合と複数の場合で処理が変化します
    m_IsSingleMode = (ApplicationState::GetPlayerCount() == 1);

    // 接続状態のNpadIdを記憶します
    std::vector<nns::hid::Controller*>list = g_pControllerManager->GetControllerList();
    for (std::vector<nns::hid::Controller*>::iterator it = list.begin(); it != list.end(); ++it)
    {
        const size_t controllerIndex = (it - list.begin());
        const nn::hid::NpadIdType id = ApplicationState::NpadIds[controllerIndex];
        ApplicationState::InitialConnectedController[controllerIndex].Reset();
        if (id != nn::hid::NpadId::Handheld)
        {
            ApplicationState::InitialConnectedController[controllerIndex].Set<nn::hid::NpadJoyAttribute::IsLeftConnected>(
                (*it)->IsConnected() || static_cast<nns::hid::GamePadNx*>((*it))->GetAttributesSet().Test<nns::hid::Attribute::IsLeftConnected>());
            ApplicationState::InitialConnectedController[controllerIndex].Set<nn::hid::NpadJoyAttribute::IsRightConnected>(
                (*it)->IsConnected() || static_cast<nns::hid::GamePadNx*>((*it))->GetAttributesSet().Test<nns::hid::Attribute::IsRightConnected>());
        }
    }

    const nn::util::Color4u8 colorList[] =
    {
        nn::util::Color4u8::Red(),
        nn::util::Color4u8::Green(),
        nn::util::Color4u8::Blue(),
        nn::util::Color4u8::Yellow()
    };

    m_PlayerList.clear();
    m_PlayerList.resize(0);
    for (size_t i = 0; i < ApplicationState::GetPlayerCount(); ++i)
    {
        PlayerState state;
        state.Color = colorList[i % NN_ARRAY_SIZE(colorList)];
        state.Style = nn::hid::GetNpadStyleSet(ApplicationState::IsHandheldMode() ? nn::hid::NpadId::Handheld : ApplicationState::NpadIds[i]);
        state.Id = ApplicationState::IsHandheldMode() ? nn::hid::NpadId::Handheld : ApplicationState::NpadIds[i];
        state.Size = NN_UTIL_FLOAT_2_INITIALIZER(48, 48);
        state.Pos = NN_UTIL_FLOAT_2_INITIALIZER(300.f + i * 210.f, 360);
        m_PlayerList.push_back(new Player(state));
    }

    g_pGraphicsSystem->StartScanScreen();
}

void GameSceneThread::Update()
{
    nn::os::LockMutex(&g_Mutex);
    {
        g_pControllerManager->Update();
    }
    nn::os::UnlockMutex(&g_Mutex);

    // コントローラーの接続確認
    nn::Result appletResult;
    nn::hid::ControllerSupportResultInfo controllerResultInfo;

    CheckPatternSet check;
    check.Reset();
    m_IsSingleMode ? check.Set<CheckPattern::Single>() : check.Set<CheckPattern::PlayerCountShort>();
    check.Set<CheckPattern::ChangeMasterController>();
    check.Set<CheckPattern::CallInterval>();
    check.Set<CheckPattern::Resume>();
    if (CheckConnectController(check))
    {
        int playerCount = static_cast<int>(std::min(ApplicationState::GetPlayerCount(), ApplicationState::GetPlayerCount()));
        // コントローラーの接続の追加が必要な場合、コントローラーサポートアプレットを呼び出す
        int32_t dialogResult = 0;

        appletResult = CallControllerSupportApplet(&controllerResultInfo, playerCount);
        if (!m_IsSingleMode)
        {
            bool IsSuccess = (appletResult.IsSuccess() && controllerResultInfo.playerCount != 0);
            if (IsSuccess == false) { BlurScreen(); }
            while (IsSuccess == false)
            {
                dialogResult = ShowDialog(m_ControllerLostDialog);
                if (dialogResult != m_ControllerLostDialog.CancelNum)
                {
                    // 終了を選択した場合
                    m_IsExit = true;
                    break;
                }
                else
                {
                    // キャンセルを選択した場合、コントローラーを追加接続するまで無限ループさせる
                    appletResult = CallControllerSupportApplet(&controllerResultInfo, playerCount);
                }
                IsSuccess = (appletResult.IsSuccess() && controllerResultInfo.playerCount != 0);
            }
        }
    }
    else
    {
        bool                isCalledDialog      = false;
        nn::hid::NpadIdType dialogCalledNpadId;

        //==============================================
        // コントローラー の処理
        //==============================================
        // シングルプレイ時のマスターコントローラーを指定
        if (m_IsSingleMode)
        {
            m_PlayerList.at(0)->SetNpadId(ApplicationState::IsHandheldMode() ? nn::hid::NpadId::Handheld : nn::hid::NpadId::No1);
        }

        // 接続中のコントローラーの入力処理
        for (
            std::vector<nns::hid::Controller*>::iterator it = g_pControllerManager->GetControllerList().begin();
            !m_IsExit && it != g_pControllerManager->GetControllerList().end();
            ++it
            )
        {
            bool    isConnectedPlayerController = false;
            size_t  playerIndex                 = 0;
            const nn::hid::NpadIdType npadId    = ApplicationState::NpadIds[it - g_pControllerManager->GetControllerList().begin()];

            // 操作権をもつ接続中のコントローラーか否か確認する
            if ((*it)->IsConnected())
            {
                for (size_t i = 0; i < m_PlayerList.size(); ++i)
                {
                    if (m_PlayerList.at(i)->GetState().Id == npadId)
                    {
                        isConnectedPlayerController = true;
                        playerIndex = i;
                        break;
                    }
                }
            }

            if (
                !isCalledDialog &&
                ((*it)->GetButtonsDown().Test<nns::hid::Button::Plus>() || (*it)->GetButtonsDown().Test<nns::hid::Button::Minus>())
                )
            {
                if ((!m_IsSingleMode && npadId == nn::hid::NpadId::Handheld) ||
                    isConnectedPlayerController)
                {
                    isCalledDialog = true;
                    dialogCalledNpadId = npadId;
                }
                else
                {
                    // 操作権を持たないコントローラー
                }
            }

            // 操作できないコントローラーの場合処理を中断します
            if (!isConnectedPlayerController)
            {
                continue;
            }

            // プレイヤーの操作
            m_PlayerList.at(playerIndex)->Move((**it));
        }

        //==============================================
        // プレイヤーの更新
        //==============================================
        for (
            std::vector<Player*>::iterator it = m_PlayerList.begin();
            !m_IsExit && it != m_PlayerList.end();
            ++it
            )
        {
            (*it)->Update();
            (*it)->DrawInfo();
        }

        //==============================================
        // ポーズ画面の呼び出し処理
        //==============================================
        if (isCalledDialog)
        {
            CallDialog(dialogCalledNpadId);
        }
    }

    if (m_IsExit)
    {
        g_pGraphicsSystem->StopScanScreen();
        nn::os::LockMutex(&g_Mutex);
        {
            ThreadState* pThreadState = nullptr;
            NN_ASSERT(g_ThreadManager.GetThreadStateFromIndex(ThreadManager::ThreadList_SceneGame, &pThreadState));
            g_ThreadManager.SetEndThread(pThreadState->GetThreadType());
            NN_ASSERT(g_ThreadManager.GetThreadStateFromIndex(ThreadManager::ThreadList_SceneTitle, &pThreadState));
            g_ThreadManager.SetNextThread(pThreadState->GetThreadType());
        }
        nn::os::UnlockMutex(&g_Mutex);
    }
} // NOLINT(impl/function_size)

void GameSceneThread::Draw()
{
    nn::os::LockMutex(&g_Mutex);
    {
        g_pGraphicsSystem->BeginDraw();
        {
            m_pRenderer->Draw2DLine(m_pCommand, 0, 720.f - 64.f, 1280, 720.f - 64.f);

            for (
                std::vector<Player*>::iterator it = m_PlayerList.begin();
                !m_IsExit && it != m_PlayerList.end();
                ++it
                )
            {
                (*it)->Draw(m_pCommand, m_pRenderer);
            }


            m_pRenderer->SetColor(nn::util::Color4u8::Gray());
            m_pRenderer->Draw2DRect(
                m_pCommand, 0, 0,
                640.f / 3.5f, 64.f / 3.5f, g_TextureDescriptor[ResourceType_Title], g_SamplerDescriptor);

            m_pTextWriter->Draw(m_pCommand);
        }
        g_pGraphicsSystem->EndDraw();
        g_pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / g_FrameRate));
    }
    nn::os::UnlockMutex(&g_Mutex);
}

void GameSceneThread::Finalize()
{
    for (std::vector<Player*>::iterator it = m_PlayerList.begin(); it != m_PlayerList.end(); ++it)
    {
        delete (*it);
        (*it) = nullptr;
    }
}

void GameSceneThread::InitializeDialog()
{
    // ポーズ画面
    {
        m_PauseDialog.SetDefault();
        m_PauseDialog.Size.y = 0.f;
        m_PauseDialog.Title = "\nPAUSE\n";
        m_PauseDialog.Message = "";
        m_PauseDialog.Answer.clear();
        m_PauseDialog.Answer.push_back("Resume");
        m_PauseDialog.Answer.push_back("Go to App title");
        m_PauseDialog.CancelNum = 0;
    }
    // 継続確認画面
    DialogInfo insControllerDialogInfo;
    {
        m_ControllerLostDialog.SetDefault();
        m_ControllerLostDialog.Size.y = 0.f;
        m_ControllerLostDialog.Title = "\nStop playing and\n\nreturn to App title?\n";
        m_ControllerLostDialog.Message = "";
        m_ControllerLostDialog.Answer.clear();
        m_ControllerLostDialog.Answer.push_back("Cancel");
        m_ControllerLostDialog.Answer.push_back("Return to App title");
        m_ControllerLostDialog.CancelNum = 0;
        m_ControllerLostDialog.ControllerCheck.Reset();
    }
}

void GameSceneThread::CallDialog(const nn::hid::NpadIdType& npadId)
{
    BlurScreen();
    nn::hid::ControllerSupportResultInfo appletResultInfo;
    while (!m_IsExit)
    {
        int32_t dialogResult = ShowDialog(m_PauseDialog, npadId);

        if (dialogResult == -1)
        {
            while (!m_IsExit)
            {
                dialogResult = ShowDialog(m_ControllerLostDialog);
                if (dialogResult == -1)
                {

                }
                else if (dialogResult == m_ControllerLostDialog.CancelNum)
                {
                    int playerCount = static_cast<int>(std::min(ApplicationState::GetPlayerCount(), ApplicationState::MaxPlayerCount));
                    if (CallControllerSupportApplet(&appletResultInfo, playerCount).IsSuccess())
                    {
                        break;
                    }
                }
                else
                {
                    // 終了を選択した場合
                    m_IsExit = true;
                }
            }
        }
        else if (dialogResult == m_PauseDialog.CancelNum)
        {
            break;
        }
        else
        {
            m_IsExit = true;
            break;
        }
    }
}

void GameSceneThread::BlurScreen()
{
    //// 画面のぼかし
    nn::os::LockMutex(&g_Mutex);
    {
        g_pGraphicsSystem->StartScanScreen();
        for (size_t i = 0; i < 4; ++i)
        {
            g_pGraphicsSystem->BeginDraw();
            {
                m_pRenderer->SetUserPixelShader(g_pGraphicsSystem->GetBlurShader());
                m_pRenderer->Draw2DRect(m_pCommand,
                    0, 720, 1280, -720, g_ScanScreenDescriptor, g_SamplerDescriptor);
                m_pRenderer->SetUserPixelShader(NULL);
            }
            g_pGraphicsSystem->EndDraw();
            g_pGraphicsSystem->Synchronize(
                nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / g_FrameRate));
        }
        g_pGraphicsSystem->StopScanScreen();
    }
    nn::os::UnlockMutex(&g_Mutex);
}
