﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


#include <stdint.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_VibrationValueTypes.h>

namespace nns
{

extern const float g_freqs[256];

/**
 * @brief Decoder for vibration data
 *
 * Vibration data is made of two bands, each band being composed of a frequency and an amplitude.
 * Band data is updated every 5 ms.
 *
 * Over a 5 ms chunk, bands are generated individually and then summed together to compose the final signal.
 * Each band possesses its own state made of three values: phase (angle), frequency and amplitude. A band's
 * signal is generated sample by sample as follows:
 *  # increment frequency and amplitude so that they reach the specified values at the end of the chunk
 *  # increment the phase based on the current frequency
 *  # the sample's value is simply `amplitude * sin(phase)`
 */
class VibrationDecoder
{
private:
    struct State
    {
        uint32_t phase;     ///< Current phase (8.24 fixed point)
        uint32_t frequency; ///< Frequency in Hz (12.20 fixed point)
        uint32_t amplitude; ///< Current amplitude in [0; 255]
    };

    static const int FrequencyPrecision = 20; ///< Size of the fractional part for fixed point representation of frequency
    static const int AmplitudePrecision = 16; ///< Size of the fractional part for fixed point representation of amplitude
    static const int PhasePrecision     = 24; ///< Size of the fractional part for fixed point representation of phase

public:
    nn::Result Init(uint32_t samplingRate) NN_NOEXCEPT;

    size_t GenerateChunk(uint32_t fm1, uint32_t am1, uint32_t fm2, uint32_t am2, int16_t* dest) NN_NOEXCEPT;
    size_t GenerateChunk(const nn::hid::VibrationValue& vib, int16_t* dest) NN_NOEXCEPT;

private:
    size_t DoGenerateChunk(uint32_t freq1, uint32_t am1, uint32_t freq2, uint32_t am2, int16_t* dest) NN_NOEXCEPT;

    static int Sin(uint32_t phase) NN_NOEXCEPT;

private:
    State    m_states[2];
    uint32_t m_phaseRate;  ///< Precomputed ratio for increasing the phase at each step
    size_t   m_chunkLength;
};


} // namespace nns
