﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3ddemo_DemoUtility.h"
#include "g3ddemo_GfxUtility.h"

namespace nn { namespace g3d { namespace demo {

namespace {

const size_t g_HeapSize = 450 * 1024 * 1024;
void* g_pMemory = nullptr;
nn::mem::StandardAllocator g_StandardAllocator;
int g_AllocateCount = 0;
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
HWND g_ConsoleHwnd = nullptr;
#endif
nns::hid::ControllerManager g_ControllerManager;
int g_ScreenTouchHold = 0;

void* g_MountRomCacheBuffer = nullptr;

Pad g_Pad;

void LookAt(nn::util::Vector3fType* pCameraPos, nn::util::Vector3fType* pCameraTarget, const nn::g3d::Sphere* pBounding)
{
    if (pBounding == nullptr)
    {
        return;
    }

    NN_ASSERT(!isnan(pBounding->radius));
    float distance = pBounding->radius / (std::tan(nn::util::DegreeToRadian(45.0f) * 0.5f));
    if (distance < 1e-5)
    {
        // バウンディングの半径が 0 の場合は適当な値を入れておく
        distance = 5.0f;
    }

    nn::util::Vector3fType dir;
    VectorSubtract(&dir, *pCameraPos, *pCameraTarget);
    NN_ASSERT(nn::util::VectorLength(dir) != 0.0f);

    VectorNormalize(&dir, dir);
    VectorMultiply(&dir, dir, distance);

    *pCameraTarget = pBounding->center;
    VectorAdd(pCameraPos, pBounding->center, dir);
}

} // anonymous namespace

//--------------------------------------------------------------------------------------------------
void Initialize() NN_NOEXCEPT
{
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)

    _CrtSetDbgFlag(_CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);
    _CrtSetBreakAlloc(-1);

    setlocale(LC_CTYPE, "");

#if !defined(NN_BUILD_TARGET_PLATFORM_ADDRESS_64)
    // FPU の精度を 32 ビットに設定し、非正規化数を 0 として扱う。
    unsigned int oldVal;
    int err = _controlfp_s(&oldVal, _DN_FLUSH | _PC_24, _MCW_DN | MCW_PC);
    (void)err;
    NN_ASSERT(err == 0);
#endif

    wchar_t titleName[256];
    GetConsoleTitleW(titleName, 256);
    g_ConsoleHwnd = FindWindow(nullptr, titleName);
#endif
    g_pMemory = malloc(g_HeapSize);
    NN_ASSERT_NOT_NULL(g_pMemory);

    InitializeGraphicMemory();

    // コントローラマネージャを標準構成に設定
    nns::hid::util::SetControllerManagerWithDefault(&g_ControllerManager);
}

void Finalize() NN_NOEXCEPT
{
    free(g_pMemory);
    g_pMemory = nullptr;
}

void InitializeDemo() NN_NOEXCEPT
{
    g_StandardAllocator.Initialize(g_pMemory, g_HeapSize);

    // リソースをマウント
    size_t cacheSize = 0;
    nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT(result.IsSuccess());

    g_MountRomCacheBuffer = AllocateMemory(cacheSize);
    NN_ASSERT_NOT_NULL(g_MountRomCacheBuffer);

    result = nn::fs::MountRom("Resource", g_MountRomCacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    InitializeGfx();

    g_Pad.Initialize();
}

void FinalizeDemo() NN_NOEXCEPT
{
    ShutdownGfx();
    nn::fs::Unmount("Resource");
    FreeMemory(g_MountRomCacheBuffer);
    g_MountRomCacheBuffer = nullptr;
    g_StandardAllocator.Finalize();
}

//--------------------------------------------------------------------------------------------------
// Memory

void* AllocateMemory(size_t size, size_t alignment /*= DEFAULT_ALIGNMENT*/) NN_NOEXCEPT
{
    NN_ASSERT(IsPowerOfTwo(alignment));
    NN_ASSERT(size > 0);
    void* ptr = nullptr;
    ptr = g_StandardAllocator.Allocate(size, alignment);
    if (ptr == nullptr)
    {
        // メモリー確保に失敗した場合に強制終了したくないことがあるのでログ出力のみ
        NN_LOG("Memory allocation failed\n");
    }
    ++g_AllocateCount;
    return ptr;
}

void FreeMemory(void* ptr) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(ptr);
    g_StandardAllocator.Free(ptr);
    --g_AllocateCount;
}

nn::mem::StandardAllocator* GetAllocator() NN_NOEXCEPT
{
    return &g_StandardAllocator;
}

//--------------------------------------------------------------------------------------------------
// Pad

Pad& GetPad() NN_NOEXCEPT
{
    return g_Pad;
}

void Pad::Initialize() NN_NOEXCEPT
{
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    // マウスを使用する
    nn::hid::InitializeMouse();

    // デバッグパッドにキーマッピングを設定する
    nn::settings::GetDebugPadKeyboardMap(&m_DebugPadKeyboardMap);
    m_DebugPadKeyboardMap.buttonA = nn::hid::KeyboardKey::A::Index;
    m_DebugPadKeyboardMap.buttonB = nn::hid::KeyboardKey::B::Index;
    m_DebugPadKeyboardMap.buttonX = nn::hid::KeyboardKey::X::Index;
    m_DebugPadKeyboardMap.buttonY = nn::hid::KeyboardKey::Y::Index;
    m_DebugPadKeyboardMap.buttonL = nn::hid::KeyboardKey::L::Index;
    m_DebugPadKeyboardMap.buttonR = nn::hid::KeyboardKey::R::Index;
    m_DebugPadKeyboardMap.buttonZL = nn::hid::KeyboardKey::Z::Index;
    m_DebugPadKeyboardMap.buttonZR = nn::hid::KeyboardKey::Z::Index;
    m_DebugPadKeyboardMap.buttonLeft = nn::hid::KeyboardKey::LeftArrow::Index;
    m_DebugPadKeyboardMap.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    m_DebugPadKeyboardMap.buttonUp = nn::hid::KeyboardKey::UpArrow::Index;
    m_DebugPadKeyboardMap.buttonDown = nn::hid::KeyboardKey::DownArrow::Index;
    m_DebugPadKeyboardMap.buttonStart = nn::hid::KeyboardKey::Space::Index;
    m_DebugPadKeyboardMap.buttonSelect = nn::hid::KeyboardKey::Return::Index;
    nn::settings::SetDebugPadKeyboardMap(m_DebugPadKeyboardMap);
#endif
}

void Pad::InitializeNpad() NN_NOEXCEPT
{
#if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    nn::hid::InitializeNpad();

    // 使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleHandheld::Mask);

    // 使用する Npad を設定
    m_NpadIds[0] = nn::hid::NpadId::No1;
    m_NpadIds[1] = nn::hid::NpadId::Handheld;

    nn::hid::SetSupportedNpadIdType(m_NpadIds, NPADID_NUM);

    m_InitNpad = true;
#endif
}

void Pad::Reset() NN_NOEXCEPT
{
    m_AnalogStick.leftX = m_AnalogStick.leftY = m_AnalogStick.rightX = m_AnalogStick.rightY = 0.0f;
    m_Stick.leftX = m_Stick.leftY = m_Stick.rightX = m_Stick.rightY = 0;
    m_LastStick.leftX = m_LastStick.leftY = m_LastStick.rightX = m_LastStick.rightY = 0;
    m_Button = 0;
    m_LastButton = 0;
    m_Released = 0;
    m_Triggered = 0;

    return;
}

bool Pad::Read() NN_NOEXCEPT
{
    Bit32 button = 0;
    Stick stick = { 0, 0, 0, 0 };
    bool IsAnalogStickLData = false;
    bool IsAnalogStickRData = false;

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    // マウス操作
    {
        nn::hid::MouseState state;
        nn::hid::GetMouseState(&state);
        int displayWidth = GetGfxFramework()->GetDisplayWidth();
        int displayHeight = GetGfxFramework()->GetDisplayHeight();

        if (0 <= state.x && state.x <= displayWidth && 0 <= state.y && state.y <= displayHeight)
        {
            static const int8_t RADIUS = 127;
            float invRadius = 1.0f / std::min(displayWidth, displayHeight);
            float rx = invRadius * ((state.x << 1) - displayWidth);
            float ry = invRadius * (displayHeight - (state.y << 1));
            float lengthSq = rx * rx + ry * ry;
            if (lengthSq > 1.0f)
            {
                float rcpLen = 1.0f / std::sqrt(lengthSq);
                rx *= rcpLen;
                ry *= rcpLen;
            }
            if (state.buttons.Test<nn::hid::MouseButton::Left>())
            {
                stick.leftX += static_cast<int8_t>(rx * RADIUS);
                stick.leftY += static_cast<int8_t>(ry * RADIUS);
                IsAnalogStickLData = true;
            }
            if (state.buttons.Test<nn::hid::MouseButton::Right>())
            {
                stick.rightX += static_cast<int8_t>(rx * RADIUS);
                stick.rightY += static_cast<int8_t>(ry * RADIUS);
                IsAnalogStickRData = true;
            }
        }
    }
#endif

    g_ControllerManager.Update();

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    // ウインドウフォーカスが無い場合はデバッグパッドに設定されているキーボードのマッピングを無効にする
    HWND hWnd = GetForegroundWindow();
    if (!(hWnd && hWnd == GetWindowHandle()))
    {
        nn::settings::ResetDebugPadKeyboardMap();
    }
    else
    {
        nn::settings::SetDebugPadKeyboardMap(m_DebugPadKeyboardMap);
    }
#endif

    // DebugPad の入力を取得
    {
        nn::hid::DebugPadState state;
        nn::hid::GetDebugPadState(&state);
        button |= state.buttons.Test<nn::hid::DebugPadButton::Start>() ? BUTTON_START : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::Left>() ? BUTTON_LEFT : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::Up>() ? BUTTON_UP : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::Right>() ? BUTTON_RIGHT : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::Down>() ? BUTTON_DOWN : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::A>() ? BUTTON_A : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::B>() ? BUTTON_B : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::X>() ? BUTTON_X : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::Y>() ? BUTTON_Y : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::R>() ? TRIGGER_R : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::L>() ? TRIGGER_L : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::ZL>() ? TRIGGER_Z : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::ZR>() ? TRIGGER_Z : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::Select>() ? BUTTON_A : 0;

        if (!IsAnalogStickLData)
        {
            stick.leftX = static_cast<int8_t>(state.analogStickL.x >> 8);
            stick.leftY = static_cast<int8_t>(state.analogStickL.y >> 8);
        }
        if (!IsAnalogStickRData)
        {
            stick.rightX = static_cast<int8_t>(state.analogStickR.x >> 8);
            stick.rightY = static_cast<int8_t>(state.analogStickR.y >> 8);
        }
    }

    ReadNpad(&button, &stick);

    static const int STICK_THRESHOLD = 32;
    button |= (m_Stick.leftX < -STICK_THRESHOLD) ? STICK_L_LEFT : 0;
    button |= (m_Stick.leftX > STICK_THRESHOLD) ? STICK_L_RIGHT : 0;
    button |= (m_Stick.leftY < -STICK_THRESHOLD) ? STICK_L_DOWN : 0;
    button |= (m_Stick.leftY > STICK_THRESHOLD) ? STICK_L_UP : 0;
    button |= (m_Stick.rightX < -STICK_THRESHOLD) ? STICK_R_LEFT : 0;
    button |= (m_Stick.rightX > STICK_THRESHOLD) ? STICK_R_RIGHT : 0;
    button |= (m_Stick.rightY < -STICK_THRESHOLD) ? STICK_R_DOWN : 0;
    button |= (m_Stick.rightY > STICK_THRESHOLD) ? STICK_R_UP : 0;

    m_LastButton = m_Button;
    m_Button = button;
    m_LastStick = m_Stick;
    m_Stick = stick;

    m_AnalogStick.leftX +=
        static_cast<float>(m_Stick.leftX - m_LastStick.leftX) / 56.0f;
    m_AnalogStick.leftY +=
        static_cast<float>(m_Stick.leftY - m_LastStick.leftY) / 56.0f;
    m_AnalogStick.rightX +=
        static_cast<float>(m_Stick.rightX - m_LastStick.rightX) / 56.0f;
    m_AnalogStick.rightY +=
        static_cast<float>(m_Stick.rightY - m_LastStick.rightY) / 56.0f;

    Bit32 changed = m_Button ^ m_LastButton;
    m_Triggered = changed & m_Button;
    m_Released = changed & m_LastButton;

    return true;
} // NOLINT

void Pad::ReadNpad(Bit32* pButton, Stick* pStick) NN_NOEXCEPT
{
#if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)

    if (!m_InitNpad)
    {
        return;
    }

    nn::hid::NpadButtonSet npadState[NPADID_NUM];
    nn::hid::AnalogStickState npadAnalogStickL;
    nn::hid::AnalogStickState npadAnalogStickR;
    Bit32 result = GetNpadState(npadState, &npadAnalogStickL, &npadAnalogStickR);
    if (nn::g3d::CheckFlag(result, NpadState_Buttons))
    {
        for (int idxNPad = 0; idxNPad < NPADID_NUM; ++idxNPad)
        {
            nn::hid::NpadButtonSet* pNpadState = &npadState[idxNPad];
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::A>() ? BUTTON_A : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::B>() ? BUTTON_B : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::X>() ? BUTTON_X : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::Y>() ? BUTTON_Y : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::Plus>() ? BUTTON_START : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::Left>() ? BUTTON_LEFT : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::Up>() ? BUTTON_UP : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::Right>() ? BUTTON_RIGHT : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::Down>() ? BUTTON_DOWN : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::R>() ? TRIGGER_R : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::L>() ? TRIGGER_L : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::ZR>() ? TRIGGER_Z : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::ZL>() ? TRIGGER_Z : 0;
        }
    }

    if (nn::g3d::CheckFlag(result, NpadState_AnalogStickL))
    {
        pStick->leftX = static_cast<int8_t>(npadAnalogStickL.x >> 8);
        pStick->leftY = static_cast<int8_t>(npadAnalogStickL.y >> 8);
    }
    if (nn::g3d::CheckFlag(result, NpadState_AnalogStickR))
    {
        pStick->rightX = static_cast<int8_t>(npadAnalogStickR.x >> 8);
        pStick->rightY = static_cast<int8_t>(npadAnalogStickR.y >> 8);
    }
#else
    NN_UNUSED(pButton);
    NN_UNUSED(pStick);
#endif
}

Bit32 Pad::GetNpadState(nn::hid::NpadButtonSet* pNpadState,
                        nn::hid::AnalogStickState* pAnalogStickL,
                        nn::hid::AnalogStickState* pAnalogStickR
) NN_NOEXCEPT
{
    Bit32 result = 0;

#if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)

    pAnalogStickL->x = 0;
    pAnalogStickL->y = 0;
    pAnalogStickR->x = 0;
    pAnalogStickR->y = 0;

    for (int idxNpad = 0; idxNpad < NPADID_NUM; ++idxNpad)
    {
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(m_NpadIds[idxNpad]);

        pNpadState[idxNpad].Reset();

        // ジョイコン操作が有効な場合
        if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
        {
            //最新のNpadのステートを取得
            nn::hid::NpadJoyDualState joyDualState;
            nn::hid::GetNpadState(&joyDualState, m_NpadIds[idxNpad]);
            pNpadState[idxNpad] = joyDualState.buttons;
            result |= NpadState_Buttons;

            // analogStickの確認
            if (pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLRight>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLUp>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLLeft>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLDown>())
            {
                *pAnalogStickL = joyDualState.analogStickL;
                result |= NpadState_AnalogStickL;
            }

            if (pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRRight>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRUp>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRLeft>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRDown>())
            {
                *pAnalogStickR = joyDualState.analogStickR;
                result |= NpadState_AnalogStickR;
            }
        }

        // 携帯機コントローラー操作が有効な場合
        if (style.Test<nn::hid::NpadStyleHandheld>() == true)
        {
            //最新のNpadのステートを取得
            nn::hid::NpadHandheldState handheldState;
            nn::hid::GetNpadState(&handheldState, m_NpadIds[idxNpad]);
            pNpadState[idxNpad] = handheldState.buttons;
            result |= NpadState_Buttons;

            // analogStickの確認
            if (pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLRight>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLUp>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLLeft>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLDown>())
            {
                *pAnalogStickL = handheldState.analogStickL;
                result |= NpadState_AnalogStickL;
            }

            if (pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRRight>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRUp>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRLeft>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRDown>())
            {
                *pAnalogStickR = handheldState.analogStickR;
                result |= NpadState_AnalogStickR;
            }
        }
    }
#else
    NN_UNUSED(pNpadState);
    NN_UNUSED(pAnalogStickL);
    NN_UNUSED(pAnalogStickR);
#endif

    return result;
}

//--------------------------------------------------------------------------------------------------
// CameraController

void ControlCamera(nn::util::Vector3fType* pos, nn::util::Vector3fType* up, nn::util::Vector3fType* target, const nn::util::Matrix4x3fType* viewMtx, const Pad* pad) NN_NOEXCEPT
{
    const float RotSpeed     = 0.05f;
    const float ZoomSpeed    = 30.0f;
    const float ZoomMin      = 0.001f;
    const float ZoomMax      = 1000000.0f;
    const float MoveScaleMin = 0.0001f;
    const float MoveScaleMax = 1000.0f;
    const float AtMoveSpeed  = 25.0f;

    float zoomInOut = 0.0f;
    float upDown = 0.0f;

    if (pad->IsHold(Pad::BUTTON_Y))
    {
        zoomInOut = 1.0f;
    }
    else if (pad->IsHold(Pad::BUTTON_X))
    {
        zoomInOut = -1.0f;
    }

    if (pad->IsHold(Pad::BUTTON_A))
    {
        upDown = 1.0f;
    }
    else if (pad->IsHold(Pad::BUTTON_B))
    {
        upDown = -1.0f;
    }

    nn::util::Vector3fType right;
    nn::util::Vector3fType camUp;
    nn::util::Vector3fType look;
    MatrixGetColumn0(&right, *viewMtx);
    MatrixGetColumn1(&camUp, *viewMtx);
    MatrixGetColumn2(&look, *viewMtx);

    nn::util::Vector3fType diff;
    VectorSubtract(&diff, *target, *pos);

    float length = VectorLengthSquared (diff);
    float dist = length / std::sqrt(length);
    VectorNormalize(&diff, diff);
    float ratio     = ( dist - ZoomMin ) / ( ZoomMax - ZoomMin );
    float moveScale = 0.5f * ( MoveScaleMin + ( MoveScaleMax - MoveScaleMin ) * ratio );
    dist += zoomInOut * ZoomSpeed * moveScale;
    dist = ( ZoomMax <= dist ) ? ZoomMax : ( ( dist <= ZoomMin ) ? ZoomMin : dist );

    nn::util::Vector3fType dirX;
    nn::util::Vector3fType dirY;
    nn::util::Vector3fType dirZ;
    VectorSet(&dirZ, 0.0f, 1.0f, 0.0f);
    VectorCross(&dirZ, right, dirZ);
    VectorNormalize(&dirZ, dirZ);
    VectorCross(&dirY, dirZ, right);
    VectorNormalize(&dirY, dirY);

    const Pad::AnalogStick& analogStick = pad->GetAnalogStick();

    VectorMultiply(&dirX, right, AtMoveSpeed * moveScale * analogStick.rightX);
    VectorAdd(target, dirX, *target);
    VectorMultiply(&dirY, dirY, AtMoveSpeed * moveScale * upDown);
    VectorAdd(target, dirY, *target);
    VectorMultiply(&dirZ, dirZ, AtMoveSpeed * moveScale * ( -analogStick.rightY));
    VectorAdd(target, dirZ, *target);

    nn::util::Vector4fType qt;
    nn::util::Vector4fType qtX;
    nn::util::Vector4fType qtY;
    {
        float ang = analogStick.leftY * -RotSpeed * 0.5f;
        float c = std::cos(ang);
        float s = std::sin(ang);
        VectorSet(&qtX,
                  s * VectorGetX(right),
                  s * VectorGetY(right),
                  s * VectorGetZ(right),
                  c);
    }
    {
        float ang = analogStick.leftX * RotSpeed * 0.5f;
        float c = std::cos(ang);
        float s = std::sin(ang);
        VectorSet(&qtY,
                  0.0f,
                  s,
                  0.0f,
                  c);
    }

    QuaternionMultiply(&qt, qtY, qtX);

    nn::util::Matrix4x3fType qtMtx;
    MatrixSetRotate(&qtMtx, qt);

    VectorTransformNormal(&diff, diff, qtMtx);
    VectorNormalize(&diff, diff);

    VectorMultiply(pos, diff, dist);
    VectorSubtract(pos, *target, *pos);

    VectorCross(up, right, diff);
    VectorSet(up,
              0.0f,
              (VectorGetY(*up) >= 0) ? 1.0f : -1.0f,
              0.0f);

#if 0
    NN_LOG("------values------\n");
    NN_LOG("left  : %f, %f\n", analogStick.leftX, analogStick.leftY);
    NN_LOG("right : %f, %f\n", analogStick.rightX, analogStick.rightY);

    NN_LOG("pos   : %f, %f, %f\n", pos->x, pos->y, pos->z);
    NN_LOG("up    : %f, %f, %f\n", up->x, up->y, up->z);
    NN_LOG("target: %f, %f, %f\n", target->x, target->y, target->z);
#endif
}

void LookAtModel(nn::util::Vector3fType* pCameraPos, nn::util::Vector3fType* pCameraTarget, const nn::g3d::ModelObj* pTarget) NN_NOEXCEPT
{
    LookAt(pCameraPos, pCameraTarget, pTarget->GetBounding());
}

void LookAtShape(nn::util::Vector3fType* pCameraPos, nn::util::Vector3fType* pCameraTarget, const nn::g3d::ShapeObj* pTarget) NN_NOEXCEPT
{
    LookAt(pCameraPos, pCameraTarget, pTarget->GetBounding());
}

void LookAtSubMesh(
    nn::util::Vector3fType* pCameraPos,
    nn::util::Vector3fType* pCameraTarget,
    const nn::g3d::ShapeObj* pTarget,
    int meshIndex,
    int submeshIndex) NN_NOEXCEPT
{
    NN_ASSERT(meshIndex < pTarget->GetMeshCount());
    NN_ASSERT(submeshIndex < pTarget->GetSubMeshCount());
    const nn::g3d::Aabb* pSubMeshBoundingArray = pTarget->GetSubMeshBoundingArray(meshIndex);
    if (pSubMeshBoundingArray == nullptr)
    {
        return;
    }

    const Aabb* pSubMeshBounding = &pSubMeshBoundingArray[submeshIndex];
    nn::g3d::Sphere boundingSphere;
    nn::util::VectorAdd(&boundingSphere.center, pSubMeshBounding->max, pSubMeshBounding->min);
    nn::util::VectorMultiply(&boundingSphere.center, boundingSphere.center, 0.5f);
    boundingSphere.radius = nn::util::VectorDistance(boundingSphere.center, pSubMeshBounding->max);
    LookAt(pCameraPos, pCameraTarget, &boundingSphere);
}

void LookAtBone(
    nn::util::Vector3fType* pCameraPos,
    nn::util::Vector3fType* pCameraTarget,
    const nn::g3d::ModelObj* pTarget,
    int boneIndex
) NN_NOEXCEPT
{
    LookAtModel(pCameraPos, pCameraTarget, pTarget);
    const nn::g3d::SkeletonObj* pSkeletonObj = pTarget->GetSkeleton();
    const int boneCount = pSkeletonObj->GetBoneCount();
    NN_ASSERT_RANGE(boneIndex, 0, boneCount);

    nn::util::Vector3fType diff;
    nn::util::VectorSubtract(&diff, *pCameraPos, *pCameraTarget);
    nn::util::VectorMultiply(&diff, diff, 0.25f); // モデルのバウンディングボックスの大きさから 1/4 の距離とする

    nn::util::MatrixGetAxisW(pCameraTarget, pSkeletonObj->GetWorldMtxArray()[boneIndex]);
    nn::util::VectorAdd(pCameraPos, *pCameraTarget, diff);
}

//--------------------------------------------------------------------------------------------------
// File

int OpenFile(FileHandle* pHandle, const char* path, const char* mode) NN_NOEXCEPT
{
    NN_ASSERT(path != nullptr);
    NN_ASSERT(mode != nullptr);

    int fsMode = 0;
    if (std::strstr(mode, "r") != nullptr)
    {
        fsMode |= nn::fs::OpenMode_Read;
    }
    if (std::strstr(mode, "w") != nullptr)
    {
        fsMode |= nn::fs::OpenMode_Write;
    }
    if (std::strstr(mode, "a") != nullptr)
    {
        fsMode |= nn::fs::OpenMode_AllowAppend;
    }
    nn::Result result = nn::fs::OpenFile(pHandle, path, fsMode);
    return result.IsSuccess() ? 0 : 1;
}

int CloseFile(FileHandle handle) NN_NOEXCEPT
{
    nn::fs::CloseFile(handle);
    return 0;
}

int GetFileSize(FileHandle handle, size_t* pSize) NN_NOEXCEPT
{
    NN_ASSERT(pSize != nullptr);
    int64_t size;
    nn::Result result = nn::fs::GetFileSize(&size, handle);
    *pSize = static_cast<size_t>(size);
    return result.IsSuccess() ? 0 : -1;
}

int ReadFile(FileHandle handle, void* pDst, size_t length) NN_NOEXCEPT
{
    NN_ASSERT(pDst != nullptr);
    size_t readSize;
    nn::Result result = nn::fs::ReadFile(&readSize, handle, 0, pDst, length);
    return result.IsSuccess() ? 0 : -1;
}

int WriteFile(FileHandle handle, const void* pSrc, size_t length) NN_NOEXCEPT
{
    NN_ASSERT(pSrc != nullptr);
    nn::fs::WriteOption option = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag::WriteOptionFlag_Flush);
    nn::Result result = nn::fs::WriteFile(handle, 0, pSrc, length, option);
    return result.IsSuccess() ? 0 : -1;
}

void* LoadFile(const char* path, size_t* pSize /*= nullptr*/, size_t alignment /*= DEFAULT_ALIGNMENT*/) NN_NOEXCEPT
{
    NN_ASSERT(path != nullptr);
    NN_ASSERT(IsPowerOfTwo(alignment));

    int result = 0;
    FileHandle handle;
    result = nn::g3d::demo::OpenFile(&handle, path, "rb");
    NN_ASSERT(result == 0, "OpenFile failed.\n");

    size_t size = 0;
    result = nn::g3d::demo::GetFileSize(handle, &size);
    NN_ASSERT(result == 0, "GetFileSize failed.\n");
    NN_ASSERT(size != 0, "file size is 0.\n");

    // テキストファイル NUL 終端させるために多めにメモリー確保して 0 で埋める。
    void* ptr = AllocateMemory(size + DEFAULT_ALIGNMENT, alignment);
    NN_ASSERT_NOT_NULL(ptr);
    result = nn::g3d::demo::ReadFile(handle, ptr, size);
    NN_ASSERT(result == 0, "ReadFile failed.\n");

    memset(AddOffset(ptr, size), 0, DEFAULT_ALIGNMENT);
    result = nn::g3d::demo::CloseFile(handle);
    NN_ASSERT(result == 0, "CloseFile failed.\n");

    size += DEFAULT_ALIGNMENT;

    if (pSize)
    {
        *pSize = size;
    }

    NN_UNUSED(result);

    return ptr;
}

// メモリープール上にファイルをロードする
void* LoadFile(const char* path, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize) NN_NOEXCEPT
{
    NN_ASSERT(path != nullptr);

    int result = 0;
    FileHandle handle;
    result = nn::g3d::demo::OpenFile(&handle, path, "rb");
    NN_ASSERT(result == 0, "OpenFile failed.\n");

    size_t size = 0;
    result = nn::g3d::demo::GetFileSize(handle, &size);
    NN_ASSERT(result == 0, "GetFileSize failed.\n");
    NN_ASSERT(memoryPoolSize > size, "file size is 0.\n");

    void* pFile = NULL;

    // メモリープールをマップしてファイルを書き込む
    void* memoryPoolPtr = pMemoryPool->Map();
    pFile = nn::util::BytePtr(memoryPoolPtr).Advance(memoryPoolOffset).Get();
    NN_ASSERT_NOT_NULL(pFile);
    result = nn::g3d::demo::ReadFile(handle, pFile, memoryPoolSize);
    NN_ASSERT(result == 0, "ReadFile failed.\n");
    memset(AddOffset(pFile, memoryPoolSize), 0, DEFAULT_ALIGNMENT);
    pMemoryPool->FlushMappedRange(memoryPoolOffset, memoryPoolSize);
    pMemoryPool->Unmap();

    result = nn::g3d::demo::CloseFile(handle);
    NN_ASSERT(result == 0, "CloseFile failed.\n");

    return pFile;
}

void SaveFile(const char* path, const void* pData, size_t size) NN_NOEXCEPT
{
    NN_ASSERT(path != nullptr);
    NN_ASSERT(pData != nullptr);

    int result = 0;
    FileHandle handle;
    result = nn::g3d::demo::OpenFile(&handle, path, "wb");
    NN_ASSERT(result == 0, "OpenFile failed.\n");

    result = nn::g3d::demo::WriteFile(handle, pData, size);
    NN_ASSERT(result == 0, "WriteFile failed.\n");

    result = nn::g3d::demo::CloseFile(handle);
    NN_ASSERT(result == 0, "CloseFile failed.\n");

    NN_UNUSED(result);
}

// ResFileのアライメントはBinaryFileHeaderから取得
size_t GetFileAlignment(const char* path) NN_NOEXCEPT
{
    int result = 0;
    FileHandle handle;
    result = nn::g3d::demo::OpenFile(&handle, path, "rb");
    NN_ASSERT(result == 0, "OpenFile failed.\n");

    void* ptr = AllocateMemory(sizeof(nn::util::BinaryFileHeader));
    NN_ASSERT_NOT_NULL(ptr);
    result = nn::g3d::demo::ReadFile(handle, ptr, sizeof(nn::util::BinaryFileHeader));
    NN_ASSERT(result == 0, "ReadFile failed.\n");

    result = nn::g3d::demo::CloseFile(handle);
    NN_ASSERT(result == 0, "CloseFile failed.\n");

    size_t alignment = reinterpret_cast<nn::util::BinaryFileHeader*>(ptr)->GetAlignment();
    FreeMemory(ptr);

    return alignment;
}

size_t GetFileSize(const char* path) NN_NOEXCEPT
{
    int result = 0;
    FileHandle handle;
    result = nn::g3d::demo::OpenFile(&handle, path, "rb");
    NN_ASSERT(result == 0, "OpenFile failed.\n");

    size_t size = 0;
    result = nn::g3d::demo::GetFileSize(handle, &size);
    NN_ASSERT(result == 0, "GetFileSize failed.\n");

    result = nn::g3d::demo::CloseFile(handle);
    NN_ASSERT(result == 0, "CloseFile failed.\n");

    return size;
}

//--------------------------------------------------------------------------------------------------
// Menu

void CmdMenu::PrintMenu() const NN_NOEXCEPT
{
    NN_LOG("[menu]\n");
    NN_LOG("   0: %s\n", m_pItems[m_NumItem].title);
    for (int idxItem = 0; idxItem < m_NumItem; ++idxItem)
    {
        NN_LOG("  %2d: %s\n", idxItem + 1, m_pItems[idxItem].title);
    }
}

int CmdMenu::Run(int item) const NN_NOEXCEPT
{
    NN_LOG("\n");
    if (item <= 0 || m_NumItem < item)
    {
        return 0;
    }
    int returnCode = EXIT_SUCCESS;
    do
    {
        NN_LOG("[%s]\n", m_pItems[item - 1].title);
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
        _CrtMemState memState;
        (void)memState;
        _CrtMemCheckpoint(&memState);
#endif
        int beginAllocateCount = g_AllocateCount;
        returnCode = m_pItems[item - 1].pFunc();
        int endAllocateCount = g_AllocateCount;
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
        _CrtMemDumpAllObjectsSince(&memState);
#endif
        NN_ASSERT(beginAllocateCount == endAllocateCount, "Memory Leek detected. begin: %d end: %d\n", beginAllocateCount, endAllocateCount);
        if (returnCode > EXIT_SUCCESS)
        {
            NN_LOG("EXIT_FAILURE\n");
        }
        NN_LOG("\n");
    } while (returnCode == -1);
    return returnCode;
}

int CmdMenu::Select() NN_NOEXCEPT
{
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    m_SelectMenu = 0;
    NN_LOG("\n");
    NN_LOG("> ");
    int c = getchar();
    while (('\0' < c && c < '0') || '9' < c)
    {
        c = getchar();
    }
    while ('0' <= c && c <= '9')
    {
        m_SelectMenu *= 10;
        m_SelectMenu += c - '0';
        c = getchar();
    }
    NN_LOG("\n");
    return m_SelectMenu;
#else
    NN_LOG("\n");
    if (m_SelectMenu == 0)
    {
        NN_LOG(">  0: %s\n", m_pItems[m_NumItem].title);
    }
    else
    {
        NN_LOG("> %2d: %s\n", m_SelectMenu, m_pItems[m_SelectMenu - 1].title);
    }



    Pad pad;
    pad.Reset();

    while (1)
    {
        pad.Read();

        bool updated = false;
        if (pad.IsTriggered(Pad::BUTTON_STICK_L_UP))
        {
            ++m_SelectMenu;
            updated = true;
        }
        else if (pad.IsTriggered(Pad::BUTTON_STICK_L_DOWN))
        {
            --m_SelectMenu;
            updated = true;
        }
        else if (pad.IsTriggered(Pad::BUTTON_STICK_L_RIGHT))
        {
            m_SelectMenu += 10;
            updated = true;
        }
        else if (pad.IsTriggered(Pad::BUTTON_STICK_L_LEFT))
        {
            m_SelectMenu -= 10;
            updated = true;
        }

        m_SelectMenu = m_SelectMenu < 0 ? m_NumItem : (m_SelectMenu > m_NumItem ? 0 : m_SelectMenu);

        if (updated)
        {
            if (m_SelectMenu == 0)
            {
                NN_LOG(">  0: %s\n", m_pItems[m_NumItem].title);
            }
            else
            {
                NN_LOG("> %2d: %s\n", m_SelectMenu, m_pItems[m_SelectMenu - 1].title);
            }
        }

        if (pad.IsTriggered(Pad::BUTTON_A))
        {
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(16));
    }
    return m_SelectMenu;
#endif
}

int CmdMenu::Loop() NN_NOEXCEPT
{
    // 引数から起動
    if (nn::os::GetHostArgc() > 1)
    {
        int item = atoi(nn::os::GetHostArgv()[1]);
        return Run(item);
    }

    // メニューから選択。
    int returnCode = 0;
    while (NN_STATIC_CONDITION(1))
    {
        PrintMenu();
        int item = Select();
        if (item == 0)
        {
            break;
        }
        returnCode = Run(item);
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
        SetForegroundWindow(g_ConsoleHwnd);
#endif
    }

    return returnCode;
}

// タッチ長押しで終了させる
bool isExitDemo() NN_NOEXCEPT
{
#if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    const int MaxScreenTouchHold = 60;
    const nns::hid::TouchScreen* const pTouchScreen =
        reinterpret_cast<nns::hid::TouchScreen*>(g_ControllerManager.GetController(nns::hid::ControllerId_TouchScreen, 0));
    const std::vector<nns::hid::TouchScreen::TouchState>& states = pTouchScreen->GetTouchStates();
    if (states.size() > 0)
    {
        const nns::hid::TouchScreen::TouchState& state = states[0];
        if (state.IsBegan() || state.IsEnded())
        {
            g_ScreenTouchHold = 0;
        }
        else
        {
            if(++g_ScreenTouchHold >= MaxScreenTouchHold)
            {
                g_ScreenTouchHold = 0;
                return true;
            }
        }
    }
#endif
    return false;
}

nns::hid::ControllerManager* GetControllerManager() NN_NOEXCEPT
{
    return &g_ControllerManager;
}

#if defined( NN_BUILD_CONFIG_OS_WIN )
int GetFixedCoreNumber() NN_NOEXCEPT
{
    return 0;
}
#endif

}}} // namespace nn::g3d::demo
