﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3ddemo_ViewerUtility.h"
#include "Town.h"
#include <nn/g3d/g3d_Viewer.h>

namespace g3ddemo = nn::g3d::demo;

MenuFlagSet g_MenuFlag;
int         g_SelectedModelIndex;
int         g_SelectedShapeIndex;
int         g_SelectedSubmeshIndex;
int         g_SelectedViewType;
int         g_SelectedFrameBufferType;
int         g_InvalidForceLodLevel = -1;

extern nn::util::Vector3fType   g_CameraTarget;
extern nn::util::Vector3fType   g_CameraPosition;
extern g3ddemo::ResourceHolder  g_Holder;
extern TownViewer               g_TownViewer;

namespace {

// エディットモードの項目
enum EditMode
{
    EditMode_SelectModel,
    EditMode_AnimationSpeed,

    EditMode_Count,
};

// デバッグモード1の項目
// 描画に関するデバッグを行うためのモード
enum DebugMode1
{
    DebugMode1_ModelAnimation,
    DebugMode1_LightAnimation,
    DebugMode1_Wireframe,
    DebugMode1_Ice,
    DebugMode1_ShadowRendering,
    DebugMode1_WaterRendering,
    DebugMode1_GeometryRendering,
    DebugMode1_LightRendering,
    DebugMode1_Average,

    DebugMode1_Count,
};

// デバッグモード2の項目
// モデルに関するデバッグを行うためのモード
enum DebugMode2
{
    DebugMode2_SelectModel,
    DebugMode2_SelectShape,
    DebugMode2_SelectSubmesh,
    DebugMode2_BoundingSphere,
    DebugMode2_BoundingBox,
    DebugMode2_SingleShape,
    DebugMode2_SingleSubmesh,
    DebugMode2_LodLevel,
    DebugMode2_ForceLodLevel,
    DebugMode2_MipLevel,

    DebugMode2_Count,
};

// デバッグモード3の項目
// ビューに関するデバッグを行うためのモード
enum DebugMode3
{
    DebugMode3_ViewType,
    DebugMode3_DrawViewVolume,
    DebugMode3_FreezeViewVolume,

    DebugMode3_DrawDebugFrameBuffer,
    DebugMode3_FrameBufferType,

    DebugMode3_Count,
};

// メニュー項目のシフト方向
enum ShiftDirection
{
    ShiftDirection_Forward,
    ShiftDirection_Backward,
};

bool        g_EnableMenu;
int         g_MenuMode;
int         g_pSelectMenuModeIndex[MenuType_Count];
int         g_ForceLodLevel;

// 各メニュー項目の最大数
const int g_MenuModeCount[MenuType_Count] =
{
    EditMode_Count,
    DebugMode1_Count,
    DebugMode2_Count,
    DebugMode3_Count
};

// モデル数を取得
int GetModelObjCount() NN_NOEXCEPT
{
    return g_Holder.renderModelObjs.GetCount();
}

// モデル名を取得
const char* GetModelObjName(int objIndex) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(objIndex, 0, g_Holder.renderModelObjs.GetCount());
    return g_Holder.renderModelObjs[objIndex]->GetModelObj()->GetName();
}

// モデルのシェイプ数を取得
int GetShapeCount(int objIndex) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(objIndex, 0, g_Holder.renderModelObjs.GetCount());
    return g_Holder.renderModelObjs[objIndex]->GetModelObj()->GetShapeCount();
}

// モデルのシェイプ名を取得
const char* GetShapeName(int objIndex, int shapeIndex) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(objIndex, 0, g_Holder.renderModelObjs.GetCount());

    nns::g3d::RenderModelObj* pRenderModelObj = g_Holder.renderModelObjs[objIndex];
    NN_ASSERT_RANGE(shapeIndex, 0, pRenderModelObj->GetModelObj()->GetShapeCount());

    return pRenderModelObj->GetModelObj()->GetShapeName(shapeIndex);
}

// モデルのサブメッシュ数を取得
int GetSubmeshCount(int objIndex, int shapeIndex) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(objIndex, 0, g_Holder.renderModelObjs.GetCount());

    nns::g3d::RenderModelObj* pRenderModelObj = g_Holder.renderModelObjs[objIndex];
    NN_ASSERT_RANGE(shapeIndex, 0, pRenderModelObj->GetModelObj()->GetShapeCount());

    return pRenderModelObj->GetModelObj()->GetShape(shapeIndex)->GetSubMeshCount();
}

// アニメーションのステップ値を変更
void ChangeAnimStep(int objIndex, float step)
{
    if (objIndex >= 0 && objIndex < GetModelObjCount())
    {
        nns::g3d::ModelAnimObj* pModelAnimObj = g_Holder.modelAnimObjs[objIndex];
        int* pAnimId = g_Holder.animationIds[objIndex];
        NN_ASSERT_NOT_NULL(pAnimId);

        for (int animationIndex = 0; animationIndex < AnimationType_Count; ++animationIndex)
        {
            int id = pAnimId[animationIndex];
            if (id == nns::g3d::ModelAnimObj::InvalidAnimId)
            {
                continue;
            }

            nn::g3d::ModelAnimObj* pAnimObj = pModelAnimObj->GetAnimObj(id);
            NN_ASSERT_NOT_NULL(pAnimObj);
            pAnimObj->GetFrameCtrl().SetStep(pAnimObj->GetFrameCtrl().GetStep() + step);
        }
    }
}

// アニメーションのステップ値を取得
float GetAnimStep(int objIndex)
{
    if (objIndex >= 0 && objIndex < GetModelObjCount())
    {
        nns::g3d::ModelAnimObj* pModelAnimObj = g_Holder.modelAnimObjs[objIndex];
        int* pAnimId = g_Holder.animationIds[objIndex];
        NN_ASSERT_NOT_NULL(pAnimId);

        for (int animationIndex = 0; animationIndex < AnimationType_Count; ++animationIndex)
        {
            int id = pAnimId[animationIndex];
            if (id == nns::g3d::ModelAnimObj::InvalidAnimId)
            {
                continue;
            }

            nn::g3d::ModelAnimObj* pAnimObj = pModelAnimObj->GetAnimObj(id);
            NN_ASSERT_NOT_NULL(pAnimObj);
            return pAnimObj->GetFrameCtrl().GetStep();
        }
    }

    return 0.0f;
}

const char* GetViewTypeName() NN_NOEXCEPT
{
    switch (g_SelectedViewType)
    {
    case ViewType_Default:
        {
            return "Default";
        }
        break;
    case ViewType_Water:
        {
            return "Water";
        }
        break;
    case ViewType_Light0:
        {
            return "Light0";
        }
        break;
    case ViewType_Light1:
        {
            return "Light1";
        }
        break;
    case ViewType_Light2:
        {
            return "Light2";
        }
        break;
    case ViewType_Light3:
        {
            return "Light3";
        }
        break;
    default:
        break;
    }

    NN_ASSERT_RANGE(g_SelectedViewType, ViewType_Default, ViewType_Count);
    return nullptr;
}

const char* GetFrameBufferTypeName() NN_NOEXCEPT
{
    switch (g_SelectedFrameBufferType)
    {
    case FrameBufferType_Water:
        {
            return "Water";
        }
        break;
    case FrameBufferType_Shadow:
        {
            return "DepthShadow";
        }
        break;
    case FrameBufferType_Geometry:
        {
            return "Geometry";
        }
        break;
    case FrameBufferType_Light:
        {
            return "Light";
        }
        break;
    default:
        break;
    }

    NN_ASSERT_RANGE(g_SelectedFrameBufferType, FrameBufferType_Water, FrameBufferType_Count);
    return nullptr;
}

// シェーダーオプションの切り替え
void SetShaderOption(nns::g3d::RenderModelObj* pRenderModelObj, const char* pOptonName, const char* pEnabled)
{
    nn::g3d::ModelObj* pModelObj = pRenderModelObj->GetModelObj();
    int shapeCount = pModelObj->GetShapeCount();
    for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
    {
        nns::g3d::RenderUnitObj* pRenderUnitObj = pRenderModelObj->GetRenderUnitObj(shapeIndex);
        nn::g3d::ShaderSelector* pShaderSelector = pRenderUnitObj->GetShaderSelector();
        int optionIndex = pShaderSelector->FindDynamicOptionIndex(pOptonName);
        if (optionIndex >= 0)
        {
            const nn::g3d::ResShaderOption* pResShaderOption = pShaderSelector->GetDynamicOption(optionIndex);
            int choiceIndex = pResShaderOption->FindChoiceIndex(pEnabled);
            pShaderSelector->WriteDynamicKey(optionIndex, choiceIndex);
        }
    }
}

// 氷オプションの切り替え
void SetIceOption(nns::g3d::RenderModelObj* pRenderModelObj, const char* pEnabled)
{
    nn::g3d::ModelObj* pModelObj = pRenderModelObj->GetModelObj();
    int shapeCount = pModelObj->GetShapeCount();
    for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
    {
        nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(shapeIndex);
        nn::g3d::MaterialObj* pMaterial = pModelObj->GetMaterial(pShapeObj->GetMaterialIndex());
        const nn::g3d::ResRenderInfo* pIceInfo = pMaterial->GetResource()->FindRenderInfo("ice");
        if (pIceInfo && pIceInfo->GetInt(0) != 0)
        {
            nns::g3d::RenderUnitObj* pRenderUnitObj = pRenderModelObj->GetRenderUnitObj(shapeIndex);
            nn::g3d::ShaderSelector* pShaderSelector = pRenderUnitObj->GetShaderSelector();
            int optionIndex = pShaderSelector->FindDynamicOptionIndex("ice_world");
            if (optionIndex >= 0)
            {
                const nn::g3d::ResShaderOption* pResShaderOption = pShaderSelector->GetDynamicOption(optionIndex);
                int choiceIndex = pResShaderOption->FindChoiceIndex(pEnabled);
                pShaderSelector->WriteDynamicKey(optionIndex, choiceIndex);
            }
        }
    }
}

// 氷オプションの切り替え
void SetIceOption(bool isEnabled)
{
    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        SetIceOption(pRenderModelObj, isEnabled ? "1" : "0");
    }
}

// ミップレベルオプションの切り替え
void SetMipLevelOption(bool isEnabled)
{
    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        SetShaderOption(pRenderModelObj, "mip_level", isEnabled ? "1" : "0");
    }
}

// LODレベルの表示オプションの切り替え
void SetLodLevelOption(bool isEnabled)
{
    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        SetShaderOption(pRenderModelObj, "lod_level", isEnabled ? "1" : "0");
    }
}

// ワイヤーフレーム切り替え
void SetWireframe(bool isEnabled)
{
    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        nn::g3d::ModelObj* pModelObj = pRenderModelObj->GetModelObj();
        int materialCount = pModelObj->GetMaterialCount();
        for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
        {
            g3ddemo::FillMode fillMode = isEnabled ? g3ddemo::FillMode_Wireframe : g3ddemo::FillMode_Solid;
            ::SetWireframe(pRenderModelObj, fillMode, materialIndex);
        }
    }
}

// 現在選択されているメニューの項目番号を取得
int GetSelectMenuModeIndex()
{
    return g_pSelectMenuModeIndex[g_MenuMode];
}

// 現在選択されているメニュー項目を前へ移動
void ShiftSelectedMenuItemForward()
{
    int menuModeIndex = g_pSelectMenuModeIndex[g_MenuMode];
    if (++menuModeIndex >= g_MenuModeCount[g_MenuMode])
    {
        menuModeIndex = 0;
    }
    g_pSelectMenuModeIndex[g_MenuMode] = menuModeIndex;
}

// 現在選択されているメニュー項目を後ろへ移動
void ShiftSelectedMenuItemBackward()
{
    int menuModeIndex = g_pSelectMenuModeIndex[g_MenuMode];
    if (--menuModeIndex < 0)
    {
        menuModeIndex = g_MenuModeCount[g_MenuMode] - 1;
    }
    g_pSelectMenuModeIndex[g_MenuMode] = menuModeIndex;
}

// 現在選択されているモデル番号を前へ移動
void ShiftSelectedModelForward()
{
    if (++g_SelectedModelIndex >= GetModelObjCount())
    {
        g_SelectedModelIndex = 0;
    }
    g_SelectedShapeIndex = 0;
    g_SelectedSubmeshIndex = 0;
    g_ForceLodLevel = -1;
}

// 現在選択されているモデル番号を後ろへ移動
void ShiftSelectedModelBackward()
{
    if (--g_SelectedModelIndex < 0)
    {
        g_SelectedModelIndex = GetModelObjCount() - 1;
    }
    g_SelectedShapeIndex = 0;
    g_SelectedSubmeshIndex = 0;
    g_ForceLodLevel = -1;
}

// 現在選択されているシェイプ番号を前へ移動
void ShiftSelectedShapeForward()
{
    int shapeCount = GetShapeCount(g_SelectedModelIndex);
    if (++g_SelectedShapeIndex >= shapeCount)
    {
        g_SelectedShapeIndex = 0;
    }
    g_SelectedSubmeshIndex = 0;
}

// 現在選択されているシェイプ番号を後ろへ移動
void ShiftSelectedShapeBackward()
{
    if (--g_SelectedShapeIndex < 0)
    {
        int shapeCount = GetShapeCount(g_SelectedModelIndex);
        g_SelectedShapeIndex = shapeCount - 1;
    }
    g_SelectedSubmeshIndex = 0;
}

// 現在選択されているサブメッシュ番号を前へ移動
void ShiftSelectedSubmeshForward()
{
    int submeshCount = GetSubmeshCount(g_SelectedModelIndex, g_SelectedShapeIndex);
    if (++g_SelectedSubmeshIndex >= submeshCount)
    {
        g_SelectedSubmeshIndex = 0;
    }
}

// 現在選択されているサブメッシュ番号を後ろへ移動
void ShiftSelectedSubmeshBackward()
{
    if (--g_SelectedSubmeshIndex < 0)
    {
        int submeshCount = GetSubmeshCount(g_SelectedModelIndex, g_SelectedShapeIndex);
        g_SelectedSubmeshIndex = submeshCount - 1;
    }
}

// 強制 LOD レベル表示の値を前へ移動
void ShiftForceLodLevelForward()
{
    NN_ASSERT_RANGE(g_SelectedModelIndex, 0, g_Holder.renderModelObjs.GetCount());
    nn::g3d::ModelObj* pModelObj = g_Holder.renderModelObjs[g_SelectedModelIndex]->GetModelObj();
    if (pModelObj)
    {
        int lodCount = pModelObj->GetLodCount();
        if (++g_ForceLodLevel >= lodCount)
        {
            g_ForceLodLevel = -1;
        }
    }
}

// 強制 LOD レベル表示の値を後ろへ移動
void ShiftForceLodLevelBackward()
{
    if (--g_ForceLodLevel < -1)
    {
        NN_ASSERT_RANGE(g_SelectedModelIndex, 0, g_Holder.renderModelObjs.GetCount());
        nn::g3d::ModelObj* pModelObj = g_Holder.renderModelObjs[g_SelectedModelIndex]->GetModelObj();
        if (pModelObj)
        {
            int lodCount = pModelObj->GetLodCount();
            g_ForceLodLevel = lodCount - 1;
        }
    }
}

// ワイヤーフレーム切り替え
void ChangeDebugModeWireframe()
{
    g_MenuFlag.Flip<MenuFlag::Wireframe>();
    SetWireframe(g_MenuFlag.Test<MenuFlag::Wireframe>());
}

// 氷設定切り替え
void ChangeDebugModeIce()
{
    g_MenuFlag.Flip<MenuFlag::Ice>();
    SetIceOption(g_MenuFlag.Test<MenuFlag::Ice>());
}

// Mipレベル表示の設定切り替え
void ChangeDebugMipLevel()
{
    g_MenuFlag.Flip<MenuFlag::MipLevel>();
    SetMipLevelOption(g_MenuFlag.Test<MenuFlag::MipLevel>());
}

// LODレベル表示の設定切り替え
void ChangeDebugLodLevel()
{
    g_MenuFlag.Flip<MenuFlag::LodLevel>();
    SetLodLevelOption(g_MenuFlag.Test<MenuFlag::LodLevel>());
}

void AttachToEdit(nn::g3d::ModelObj* pModelObj)
{
    NN_ASSERT_NOT_NULL(pModelObj);
    if (nn::g3d::viewer::ViewerServer::GetInstance().IsConnected() &&
        !(nn::g3d::viewer::ViewerServer::GetInstance().IsModelAttached(pModelObj)
            || nn::g3d::viewer::ViewerServer::GetInstance().IsModelAttaching(pModelObj)))
    {
        nn::g3d::viewer::ViewerServer::SendUserMessageArg userMessageArg(
            "モデルのアタッチが実行されました。",
            nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageType_Info,
            nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageDestination_Log);
        nn::g3d::viewer::ViewerServer::GetInstance().QueueSendUserMessageCommand(userMessageArg);

        NN_LOG("Attach %s\n", pModelObj->GetResource()->GetName());
        nn::g3d::viewer::ViewerResult result = nn::g3d::viewer::ViewerServer::GetInstance().QueueAttachModelCommand(pModelObj);
        NN_ASSERT(result == nn::g3d::viewer::ViewerResult_Success
            || result == nn::g3d::viewer::ViewerResult_AlreadyAttached);
    }
}

// デバッグモード1の項目の切り替え
void ShiftSelectedMenuValueForDebugMode1(ShiftDirection shiftDirection)
{
    NN_UNUSED(shiftDirection);
    switch (GetSelectMenuModeIndex())
    {
    case DebugMode1_ModelAnimation:
    {
        g_MenuFlag.Flip<MenuFlag::ModelAnimation>();
    }
    break;
    case DebugMode1_LightAnimation:
    {
        g_MenuFlag.Flip<MenuFlag::LightAnimation>();
    }
    break;
    case DebugMode1_Wireframe:
    {
        ChangeDebugModeWireframe();
    }
    break;
    case DebugMode1_Ice:
    {
        ChangeDebugModeIce();
    }
    break;
    case DebugMode1_ShadowRendering:
    {
        g_MenuFlag.Flip<MenuFlag::ShadowRendering>();
    }
    break;
    case DebugMode1_WaterRendering:
    {
        g_MenuFlag.Flip<MenuFlag::WaterRendering>();
    }
    break;
    case DebugMode1_GeometryRendering:
    {
        g_MenuFlag.Flip<MenuFlag::GeometryRendering>();
    }
    break;
    case DebugMode1_LightRendering:
    {
        g_MenuFlag.Flip<MenuFlag::LightRendering>();
    }
    break;
    case DebugMode1_Average:
    {
        g_MenuFlag.Flip<MenuFlag::TextureCompute>();
    }
    break;
    default:
        break;
    }
}

// デバッグモード2の項目の切り替え
void ShiftSelectedMenuValueForDebugMode2(ShiftDirection shiftDirection)
{
    switch (GetSelectMenuModeIndex())
    {
    case DebugMode2_SelectModel:
        {
            if (shiftDirection == ShiftDirection_Forward)
            {
                ShiftSelectedModelForward();
            }
            else if (shiftDirection == ShiftDirection_Backward)
            {
                ShiftSelectedModelBackward();
            }
        }
    break;
    case DebugMode2_SelectShape:
        {
            if (shiftDirection == ShiftDirection_Forward)
            {
                ShiftSelectedShapeForward();
            }
            else if (shiftDirection == ShiftDirection_Backward)
            {
                ShiftSelectedShapeBackward();
            }
        }
    break;
    case DebugMode2_SelectSubmesh:
        {
            if (shiftDirection == ShiftDirection_Forward)
            {
                ShiftSelectedSubmeshForward();
            }
            else if (shiftDirection == ShiftDirection_Backward)
            {
                ShiftSelectedSubmeshBackward();
            }
        }
    break;
    case DebugMode2_BoundingSphere:
        {
            g_MenuFlag.Flip<MenuFlag::BoundingSphere>();
        }
    break;
    case DebugMode2_BoundingBox:
        {
            g_MenuFlag.Flip<MenuFlag::BoundingBox>();
        }
    break;
    case DebugMode2_SingleShape:
        {
            g_MenuFlag.Flip<MenuFlag::SingleShape>();
        }
    break;
    case DebugMode2_SingleSubmesh:
        {
            g_MenuFlag.Flip<MenuFlag::SingleSubmesh>();
            if (g_MenuFlag.Test<MenuFlag::SingleSubmesh>())
            {
                g_MenuFlag.Set<MenuFlag::SingleShape>(true);
            }
        }
    break;
    case DebugMode2_LodLevel:
        {
            ChangeDebugLodLevel();
        }
    break;
    case DebugMode2_ForceLodLevel:
        {
            if (shiftDirection == ShiftDirection_Forward)
            {
                ShiftForceLodLevelForward();
            }
            else if (shiftDirection == ShiftDirection_Backward)
            {
                ShiftForceLodLevelBackward();
            }
        }
    break;
    case DebugMode2_MipLevel:
        {
            ChangeDebugMipLevel();
        }
    break;
    default:
        break;
    }
}

void ShiftSelectedMenuValueForDebugMode3(ShiftDirection shiftDirection)
{
    switch (GetSelectMenuModeIndex())
    {
    case DebugMode3_ViewType:
        {
            g_SelectedViewType += shiftDirection == ShiftDirection_Forward ? 1 : ViewType_Count - 1;
            g_SelectedViewType %= ViewType_Count;

            g_MenuFlag.Set<MenuFlag::FreezeViewVolume>(false);
        }
        break;

    case DebugMode3_DrawViewVolume:
        {
            g_MenuFlag.Flip<MenuFlag::DrawViewVolume>();
        }
        break;

    case DebugMode3_FreezeViewVolume:
        {
            g_MenuFlag.Flip<MenuFlag::FreezeViewVolume>();
        }
        break;

    case DebugMode3_DrawDebugFrameBuffer:
        {
            g_MenuFlag.Flip<MenuFlag::DrawDebugFrameBuffer>();
        }
        break;

    case DebugMode3_FrameBufferType:
        {
            g_SelectedFrameBufferType += shiftDirection == ShiftDirection_Forward ? 1 : FrameBufferType_Count - 1;
            g_SelectedFrameBufferType %= FrameBufferType_Count;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

// 選択されたメニュー項目の強調
void SetMenuTextColor(g3ddemo::ScreenInfo& screenInfo, int index)
{
    if (index == GetSelectMenuModeIndex())
    {
        screenInfo.SetTextColor(0, 150, 50, 255);
    }
    else
    {
        screenInfo.SetTextColor(255, 255, 255, 255);
    }
}

const char* FindAttachOrDetachString(const char* name)
{
    nn::g3d::ResShaderArchive* pShaderArchive = g_TownViewer.FindShaderArchive(name);
    if (pShaderArchive != nullptr &&
        nn::g3d::viewer::ViewerServer::GetInstance().IsShaderArchiveAttached(pShaderArchive))
    {
        return "Detach";
    }

    return "Attach";
}

void MoveFocusToSelected()
{
    NN_ASSERT_RANGE(g_SelectedModelIndex, 0, g_Holder.renderModelObjs.GetCount());
    nn::g3d::ModelObj* pModelObj = g_Holder.renderModelObjs[g_SelectedModelIndex]->GetModelObj();
    if (pModelObj)
    {
        if (g_MenuFlag.Test<MenuFlag::SingleShape>())
        {
            nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(g_SelectedShapeIndex);
            if (g_MenuFlag.Test<MenuFlag::SingleSubmesh>())
            {
                int meshIndex = 0;
                if (g_ForceLodLevel >= 0)
                {
                    meshIndex = g_ForceLodLevel < pShapeObj->GetMeshCount() ? g_ForceLodLevel : pShapeObj->GetMeshCount() - 1;
                }

                // RigidBody でないサブメッシュは個別にバウンディングボリュームを持っていないので、
                // 代わりにサブメッシュが属するシェイプ本体にカメラをフォーカスします。
                if (pShapeObj->IsRigidBody())
                {
                    g3ddemo::LookAtSubMesh(&g_CameraPosition, &g_CameraTarget, pShapeObj, meshIndex, g_SelectedSubmeshIndex);
                }
                else
                {
                    g3ddemo::LookAtShape(&g_CameraPosition, &g_CameraTarget, pShapeObj);
                }
            }
            else
            {
                g3ddemo::LookAtShape(&g_CameraPosition, &g_CameraTarget, pShapeObj);
            }
        }
        else
        {
            g3ddemo::LookAtModel(&g_CameraPosition, &g_CameraTarget, pModelObj);
        }
    }
}
} // anonymous namespace

int GetMenuMode() NN_NOEXCEPT
{
    return g_MenuMode;
}

int GetForceLodLevel() NN_NOEXCEPT
{
    return g_ForceLodLevel;
}

bool IsMenuEnabled() NN_NOEXCEPT
{
    return g_EnableMenu;
}

template<typename MenuFlag>
bool IsMenuFlagEnabled() NN_NOEXCEPT
{
    return g_MenuFlag.Test<MenuFlag>();
}

void InitializeMenu() NN_NOEXCEPT
{
    g_EnableMenu = true;
    g_MenuMode = MenuType_Edit;
    g_SelectedModelIndex = 0;
    g_SelectedShapeIndex = 0;
    g_SelectedSubmeshIndex = 0;
    g_SelectedViewType = ViewType_Default;
    g_SelectedFrameBufferType = FrameBufferType_Shadow;

    memset(g_pSelectMenuModeIndex, 0, sizeof(g_pSelectMenuModeIndex));

    g_ForceLodLevel = g_InvalidForceLodLevel;

    g_MenuFlag.Set<MenuFlag::WaterRendering>(true);
    g_MenuFlag.Set<MenuFlag::GeometryRendering>(true);
    g_MenuFlag.Set<MenuFlag::LightRendering>(true);
    g_MenuFlag.Set<MenuFlag::ShadowRendering>(true);
    g_MenuFlag.Set<MenuFlag::ModelAnimation>(true);
    g_MenuFlag.Set<MenuFlag::TextureCompute>(true);
    g_MenuFlag.Set<MenuFlag::FreezeViewVolume>(false);
    g_MenuFlag.Set<MenuFlag::DrawViewVolume>(true);
    g_MenuFlag.Set<MenuFlag::DrawDebugFrameBuffer>(false);
}

void CalculateMenuInfo(g3ddemo::ScreenInfo& screenInfo, nn::util::Float4& texAverage) NN_NOEXCEPT
{
    screenInfo.StartPrint();
    {
        screenInfo.Print("[Town]\n");

        if (g_MenuMode == MenuType_Edit)
        {
            screenInfo.Print("Edit Mode\n");
            screenInfo.Print("L : Change Mode\n");
            screenInfo.Print("R : Hide Menu\n");
            screenInfo.Print("LStick : Rotate\n");
            screenInfo.Print("RStick : Translate\n");

            SetMenuTextColor(screenInfo, EditMode_SelectModel);
            screenInfo.Print("Selected Model : %s\n", GetModelObjName(g_SelectedModelIndex));

            SetMenuTextColor(screenInfo, EditMode_AnimationSpeed);
            // アニメーションのステップ値を取得
            screenInfo.Print("Anim Speed : %.2f\n", GetAnimStep(g_SelectedModelIndex));
            screenInfo.SetTextColor(255, 255, 255, 255);

            screenInfo.Print("A : Attach Model\n");
            screenInfo.Print("B : %s Average Shader\n", FindAttachOrDetachString("tex_average"));
            screenInfo.Print("X : %s Shadow Shader\n", FindAttachOrDetachString("shadow"));
            screenInfo.Print("Y : %s Geometry Shader\n", FindAttachOrDetachString("gbuffer"));
            screenInfo.Print("ZR : Follow Model\n");
            screenInfo.Print("%d models are registered.\n", GetModelObjCount());
        }
        else if (g_MenuMode == MenuType_Debug1)
        {
            float lineHeight = screenInfo.GetFontHeight();
            float drawLine = lineHeight;
            float quadSizeX = lineHeight * 0.7f;
            float quadSizeY = lineHeight;
            float rightPosition = 180.f;
            float ratePosition = 125.f;
            float referenceFrame = 1000000.f / 60.f;

            screenInfo.Print("Debug Mode 1\n");
            screenInfo.Print("L : Change Mode\n");
            screenInfo.Print("R : Hide Menu\n");
            screenInfo.Print("LStick : Rotate\n");
            screenInfo.Print("RStick : Translate\n");
            drawLine += lineHeight * 5;

            SetMenuTextColor(screenInfo, DebugMode1_ModelAnimation);
            screenInfo.Print("    : Model Animation\n");
            screenInfo.Print(rightPosition, drawLine, "%s\n",
                g_MenuFlag.Test<MenuFlag::ModelAnimation>() ? "  [Enable]" : " [Disable]");
            drawLine += lineHeight;

            SetMenuTextColor(screenInfo, DebugMode1_LightAnimation);
            screenInfo.Print("    : Light Animation\n");
            screenInfo.Print(rightPosition, drawLine, "%s\n",
                g_MenuFlag.Test<MenuFlag::LightAnimation>() ? "  [Enable]" : " [Disable]");
            drawLine += lineHeight;

            SetMenuTextColor(screenInfo, DebugMode1_Wireframe);
            screenInfo.Print("    : Wireframe\n");
            screenInfo.Print(rightPosition, drawLine, "%s\n",
                g_MenuFlag.Test<MenuFlag::Wireframe>() ? "  [Enable]" : " [Disable]");
            drawLine += lineHeight;

            SetMenuTextColor(screenInfo, DebugMode1_Ice);
            screenInfo.Print("    : Ice\n");
            screenInfo.Print(rightPosition, drawLine, "%s\n",
                g_MenuFlag.Test<MenuFlag::Ice>() ? "  [Enable]" : " [Disable]");
            drawLine += lineHeight;

            nn::TimeSpan gpuResult;

            screenInfo.SetTextColor(255, 255, 255, 255);
            screenInfo.Print("    : Clear\n");
            gpuResult = NN_PERF_GET_ELAPSED_TIME_GPU("Clear", 0);
            screenInfo.Print(ratePosition, drawLine, "%4.1f%%\n", 100 * gpuResult.GetMicroSeconds() / referenceFrame);
            screenInfo.AddQuad(0, drawLine, quadSizeX, quadSizeY, nn::util::Color4u8(0, 0, 0, 228));
            drawLine += lineHeight;

            SetMenuTextColor(screenInfo, DebugMode1_ShadowRendering);
            screenInfo.Print("    : Shadow\n");
            gpuResult = NN_PERF_GET_ELAPSED_TIME_GPU("Shadow", 0);
            screenInfo.Print(ratePosition, drawLine, "%4.1f%%\n", 100 * gpuResult.GetMicroSeconds() / referenceFrame);
            screenInfo.Print(rightPosition, drawLine, "%s\n", g_MenuFlag.Test<MenuFlag::ShadowRendering>() ? "  [Enable]" : " [Disable]");
            screenInfo.AddQuad(0, drawLine, quadSizeX, quadSizeY, nn::util::Color4u8(255, 0, 0, 128));
            drawLine += lineHeight;

            SetMenuTextColor(screenInfo, DebugMode1_WaterRendering);
            screenInfo.Print("    : Water\n");
            gpuResult = NN_PERF_GET_ELAPSED_TIME_GPU("Water", 0);
            screenInfo.Print(ratePosition, drawLine, "%4.1f%%\n", 100 * gpuResult.GetMicroSeconds() / referenceFrame);
            screenInfo.Print(rightPosition, drawLine, "%s\n", g_MenuFlag.Test<MenuFlag::WaterRendering>() ? "  [Enable]" : " [Disable]");
            screenInfo.AddQuad(0, drawLine, quadSizeX, quadSizeY, nn::util::Color4u8(0, 0, 255, 128));
            drawLine += lineHeight;

            SetMenuTextColor(screenInfo, DebugMode1_GeometryRendering);
            screenInfo.Print("    : Geometry\n");
            gpuResult = NN_PERF_GET_ELAPSED_TIME_GPU("Geometry", 0);
            screenInfo.Print(ratePosition, drawLine, "%4.1f%%\n", 100 * gpuResult.GetMicroSeconds() / referenceFrame);
            screenInfo.Print(rightPosition, drawLine, "%s\n", g_MenuFlag.Test<MenuFlag::GeometryRendering>() ? "  [Enable]" : " [Disable]");
            screenInfo.AddQuad(0, drawLine, quadSizeX, quadSizeY, nn::util::Color4u8(0, 255, 0, 128));
            drawLine += lineHeight;

            SetMenuTextColor(screenInfo, DebugMode1_LightRendering);
            screenInfo.Print("    : Light\n");
            gpuResult = NN_PERF_GET_ELAPSED_TIME_GPU("Light", 0);
            screenInfo.Print(ratePosition, drawLine, "%4.1f%%\n", 100 * gpuResult.GetMicroSeconds() / referenceFrame);
            screenInfo.Print(rightPosition, drawLine, "%s\n", g_MenuFlag.Test<MenuFlag::LightRendering>() ? "  [Enable]" : " [Disable]");
            screenInfo.AddQuad(0, drawLine, quadSizeX, quadSizeY, nn::util::Color4u8(200, 200, 0, 128));
            drawLine += lineHeight;

            screenInfo.SetTextColor(255, 255, 255, 255);
            screenInfo.Print("    : Opaque\n");
            gpuResult = NN_PERF_GET_ELAPSED_TIME_GPU("Opaque", 0);
            screenInfo.Print(ratePosition, drawLine, "%4.1f%%\n", 100 * gpuResult.GetMicroSeconds() / referenceFrame);
            screenInfo.AddQuad(0, drawLine, quadSizeX, quadSizeY, nn::util::Color4u8(255, 0, 200, 128));
            drawLine += lineHeight;

            screenInfo.Print("    : Copy\n");
            gpuResult = NN_PERF_GET_ELAPSED_TIME_GPU("Copy", 0);
            screenInfo.Print(ratePosition, drawLine, "%4.1f%%\n", 100 * gpuResult.GetMicroSeconds() / referenceFrame);
            screenInfo.AddQuad(0, drawLine, quadSizeX, quadSizeY, nn::util::Color4u8(200, 200, 200, 128));
            drawLine += lineHeight;

            screenInfo.Print("    : Translucent\n");
            gpuResult = NN_PERF_GET_ELAPSED_TIME_GPU("Translucent", 0);
            screenInfo.Print(ratePosition, drawLine, "%4.1f%%\n", 100 * gpuResult.GetMicroSeconds() / referenceFrame);
            screenInfo.AddQuad(0, drawLine, quadSizeX, quadSizeY, nn::util::Color4u8(0, 0, 255, 128));
            drawLine += lineHeight;

            screenInfo.Print("    : Info\n");
            gpuResult = NN_PERF_GET_ELAPSED_TIME_GPU("Info", 0);
            screenInfo.Print(ratePosition, drawLine, "%4.1f%%\n", 100 * gpuResult.GetMicroSeconds() / referenceFrame);
            screenInfo.AddQuad(0, drawLine, quadSizeX, quadSizeY, nn::util::Color4u8(0, 255, 255, 128));
            drawLine += lineHeight;

            SetMenuTextColor(screenInfo, DebugMode1_Average);
            screenInfo.Print("    : Average\n");
            gpuResult = NN_PERF_GET_ELAPSED_TIME_GPU("Calculate", 0);
            screenInfo.Print(ratePosition, drawLine, "%4.1f%%\n", 100 * gpuResult.GetMicroSeconds() / referenceFrame);
            screenInfo.Print(rightPosition, drawLine, g_MenuFlag.Test<MenuFlag::TextureCompute>() ? "  [Enable]" : " [Disable]");
            screenInfo.AddQuad(0, drawLine, quadSizeX, quadSizeY, nn::util::Color4u8(255, 0, 0, 128));
            drawLine += lineHeight;
            screenInfo.SetTextColor(255, 255, 255, 255);

            if (NN_STATIC_CONDITION(NN_PERF_IS_ENABLED()))
            {
                nn::perf::CpuMeter* pFrameMeter = NN_PERF_GET_FRAME_METER();
                float millisecondsPerFrame = static_cast<float>(pFrameMeter->GetLastTotalSpan().GetNanoSeconds()) / 1000000.0f;
                float fps = 1000.f / millisecondsPerFrame;
                screenInfo.Print("%4.1f FPS\n", fps);
                drawLine += lineHeight;
            }

            if (g_MenuFlag.Test<MenuFlag::TextureCompute>())
            {
                screenInfo.Print("Pixel Average:\n");
                drawLine += lineHeight;
                screenInfo.Print("  R %f\n", texAverage.x);
                screenInfo.AddQuad(0, drawLine, texAverage.x * 200, lineHeight, nn::util::Color4u8::Red());
                drawLine += lineHeight;
                screenInfo.Print("  G %f\n", texAverage.y);
                screenInfo.AddQuad(0, drawLine, texAverage.y * 200, lineHeight, nn::util::Color4u8::Green());
                drawLine += lineHeight;
                screenInfo.Print("  B %f\n", texAverage.z);
                screenInfo.AddQuad(0, drawLine, texAverage.z * 200, lineHeight, nn::util::Color4u8::Blue());
                drawLine += lineHeight;
                screenInfo.Print("  A %f\n", texAverage.w);
                screenInfo.AddQuad(0, drawLine, texAverage.w * 200, lineHeight, nn::util::Color4u8::Gray());
            }
        }
        else if (g_MenuMode == MenuType_Debug2)
        {
            float lineHeight = screenInfo.GetFontHeight();
            float drawLine = lineHeight;
            float rightPosition = 230.f;

            screenInfo.Print("Debug Mode 2\n");
            screenInfo.Print("L : Change Mode\n");
            screenInfo.Print("R : Hide Menu\n");
            screenInfo.Print("LStick : Rotate\n");
            screenInfo.Print("RStick : Translate\n");
            drawLine += lineHeight * 8;

            NN_ASSERT(g_SelectedModelIndex < GetModelObjCount());

            SetMenuTextColor(screenInfo, DebugMode2_SelectModel);
            screenInfo.Print("Selected Model : %s\n", GetModelObjName(g_SelectedModelIndex));

            SetMenuTextColor(screenInfo, DebugMode2_SelectShape);
            int shapeCount = GetShapeCount(g_SelectedModelIndex);
            const char* shapeName = GetShapeName(g_SelectedModelIndex, g_SelectedShapeIndex);
            screenInfo.Print("Selected Shape : %s (%d/%d)\n", shapeName, g_SelectedShapeIndex + 1, shapeCount);

            SetMenuTextColor(screenInfo, DebugMode2_SelectSubmesh);
            int submeshCount = GetSubmeshCount(g_SelectedModelIndex, g_SelectedShapeIndex);
            screenInfo.Print("Selected Submesh : %d / %d\n", g_SelectedSubmeshIndex + 1, submeshCount);

            SetMenuTextColor(screenInfo, DebugMode2_BoundingSphere);
            screenInfo.Print("    : Bounding Sphere\n");
            screenInfo.Print(rightPosition, drawLine, "%s\n",
                g_MenuFlag.Test<MenuFlag::BoundingSphere>() ? "  [Enable]" : " [Disable]");
            drawLine += lineHeight;

            SetMenuTextColor(screenInfo, DebugMode2_BoundingBox);
            screenInfo.Print("    : Bounding Box\n");
            screenInfo.Print(rightPosition, drawLine, "%s\n",
                g_MenuFlag.Test<MenuFlag::BoundingBox>() ? "  [Enable]" : " [Disable]");
            drawLine += lineHeight;

            SetMenuTextColor(screenInfo, DebugMode2_SingleShape);
            screenInfo.Print("    : Single Shape\n");
            screenInfo.Print(rightPosition, drawLine, "%s\n",
                g_MenuFlag.Test<MenuFlag::SingleShape>() ? "  [Enable]" : " [Disable]");
            drawLine += lineHeight;

            SetMenuTextColor(screenInfo, DebugMode2_SingleSubmesh);
            screenInfo.Print("    : Single Submesh\n");
            screenInfo.Print(rightPosition, drawLine, "%s\n",
                g_MenuFlag.Test<MenuFlag::SingleSubmesh>() ? "  [Enable]" : " [Disable]");
            drawLine += lineHeight;

            SetMenuTextColor(screenInfo, DebugMode2_LodLevel);
            screenInfo.Print("    : Display LOD Level\n");
            screenInfo.Print(rightPosition, drawLine, "%s\n",
                g_MenuFlag.Test<MenuFlag::LodLevel>() ? "  [Enable]" : " [Disable]");
            drawLine += lineHeight;

            SetMenuTextColor(screenInfo, DebugMode2_ForceLodLevel);
            screenInfo.Print("    : Force LOD Level\n");

            NN_ASSERT_RANGE(g_SelectedModelIndex, 0, g_Holder.renderModelObjs.GetCount());
            nn::g3d::ModelObj* pModelObj = g_Holder.renderModelObjs[g_SelectedModelIndex]->GetModelObj();
            if (g_ForceLodLevel == -1)
            {
                screenInfo.Print(rightPosition, drawLine, "  [Not Specified]\n");
            }
            else if (pModelObj)
            {
                screenInfo.Print(rightPosition, drawLine, "  %d / %d\n",
                    g_ForceLodLevel, pModelObj->GetLodCount() - 1);
            }
            drawLine += lineHeight;

            SetMenuTextColor(screenInfo, DebugMode2_MipLevel);
            screenInfo.Print("    : Display Mip Level\n");
            screenInfo.Print(rightPosition, drawLine, "%s\n",
                g_MenuFlag.Test<MenuFlag::MipLevel>() ? "  [Enable]" : " [Disable]");
            drawLine += lineHeight;



            screenInfo.SetTextColor(255, 255, 255, 255);

            screenInfo.Print("Camera Position: (%.1f, %.1f, %.1f)\n",
                nn::util::VectorGetX(g_CameraPosition),
                nn::util::VectorGetY(g_CameraPosition),
                nn::util::VectorGetZ(g_CameraPosition));

            screenInfo.Print("Camera Target:   (%.1f, %.1f, %.1f)\n",
                nn::util::VectorGetX(g_CameraTarget),
                nn::util::VectorGetY(g_CameraTarget),
                nn::util::VectorGetZ(g_CameraTarget));
        }
        else if (g_MenuMode == MenuType_Debug3)
        {
            screenInfo.Print("Debug Mode 3\n");
            screenInfo.Print("L : Change Mode\n");
            screenInfo.Print("R : Hide Menu\n");
            screenInfo.Print("LStick : Rotate\n");
            screenInfo.Print("RStick : Translate\n");

            SetMenuTextColor(screenInfo, DebugMode3_ViewType);
            screenInfo.Print("Selected View Type : %s\n", GetViewTypeName());

            SetMenuTextColor(screenInfo, DebugMode3_DrawViewVolume);
            screenInfo.Print("    : Draw View Volume : %s\n", g_MenuFlag.Test<MenuFlag::DrawViewVolume>() ? "  [Enable]" : " [Disable]");

            SetMenuTextColor(screenInfo, DebugMode3_FreezeViewVolume);
            screenInfo.Print("    : Freeze View Volume : %s\n", g_MenuFlag.Test<MenuFlag::FreezeViewVolume>() ? " [Enable]" : " [Disable]");

            SetMenuTextColor(screenInfo, DebugMode3_DrawDebugFrameBuffer);
            screenInfo.Print("Draw Debug FrameBuffer : %s\n", g_MenuFlag.Test<MenuFlag::DrawDebugFrameBuffer>() ? " [Enable]" : " [Disable]");

            SetMenuTextColor(screenInfo, DebugMode3_FrameBufferType);
            screenInfo.Print("    : Selected FrameBuffer : %s\n", GetFrameBufferTypeName());

            screenInfo.SetTextColor(255, 255, 255, 255);
        }
    }

    screenInfo.EndPrint();
    screenInfo.AdjustWindowSize();
} // NOLINT

void UpdateMenuParameter(const g3ddemo::Pad& pad) NN_NOEXCEPT
{
    // メニューの表示切り替え
    if (pad.IsTriggered(g3ddemo::Pad::TRIGGER_R))
    {
        g_EnableMenu = !g_EnableMenu;
        return;
    }

    // 非表示中は入力を受け付けない
    if (!g_EnableMenu)
    {
        return;
    }

    // 各モード共通処理
    {
        // モードページの切り替え
        if (pad.IsTriggered(g3ddemo::Pad::TRIGGER_L))
        {
            if (++g_MenuMode >= MenuType_Count)
            {
                g_MenuMode = MenuType_Edit;
            }
            return;
        }

        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_DOWN))
        {
            ShiftSelectedMenuItemForward();
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_UP))
        {
            ShiftSelectedMenuItemBackward();
        }
        if (pad.IsHold(g3ddemo::Pad::TRIGGER_Z))
        {
            MoveFocusToSelected();
        }
    }

    // エディットモード
    if (g_MenuMode == MenuType_Edit)
    {
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
        {
            if (GetSelectMenuModeIndex() == EditMode_SelectModel)
            {
                ShiftSelectedModelBackward();
            }
            else if (GetSelectMenuModeIndex() == EditMode_AnimationSpeed)
            {
                ChangeAnimStep(g_SelectedModelIndex, -0.01f);
            }
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
        {
            if (GetSelectMenuModeIndex() == EditMode_SelectModel)
            {
                ShiftSelectedModelForward();
            }
            else if (GetSelectMenuModeIndex() == EditMode_AnimationSpeed)
            {
                ChangeAnimStep(g_SelectedModelIndex, 0.01f);
            }
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_A))
        {
            nns::g3d::RenderModelObj* pRenderModelObj = g_Holder.renderModelObjs[g_SelectedModelIndex];
            AttachToEdit(pRenderModelObj->GetModelObj());
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_B))
        {
            g_TownViewer.AttachShader("tex_average");
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_Y))
        {
            g_TownViewer.AttachShader("gbuffer");
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_X))
        {
            g_TownViewer.AttachShader("shadow");
        }
    }
    // デバッグモード1
    else if (g_MenuMode == MenuType_Debug1)
    {
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
        {
            ShiftSelectedMenuValueForDebugMode1(ShiftDirection_Backward);
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
        {
            ShiftSelectedMenuValueForDebugMode1(ShiftDirection_Forward);
        }
    }
    // デバッグモード2
    else if (g_MenuMode == MenuType_Debug2)
    {
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
        {
            ShiftSelectedMenuValueForDebugMode2(ShiftDirection_Backward);
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
        {
            ShiftSelectedMenuValueForDebugMode2(ShiftDirection_Forward);
        }
    }
    // デバッグモード3
    else if (g_MenuMode == MenuType_Debug3)
    {
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
        {
            ShiftSelectedMenuValueForDebugMode3(ShiftDirection_Backward);
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
        {
            ShiftSelectedMenuValueForDebugMode3(ShiftDirection_Forward);
        }
    }
} // NOLINT
