﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "g3ddemo_GfxUtility.h"

// 入力テクスチャーの平均輝度を求めるクラス
class TextureAverageShader
{
    NN_DISALLOW_COPY(TextureAverageShader);
public:
    TextureAverageShader() NN_NOEXCEPT
        : m_pSelector(nullptr)
    {
    }

    static const int bufferCount = 2;
    static const int tileDimX = 4;
    static const int tileDimY = 4;
    static const int tileCount = tileDimX * tileDimY;
    static const int bufferSize = tileDimX * tileDimY * sizeof(nn::util::Float4);

    void Initialize(nn::gfx::Device* pDevice, nn::g3d::ResShadingModel* pResShadingModel) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pResShadingModel);

        {
            nn::g3d::ShadingModelObj::Builder shadingModelbuilder(pResShadingModel);
            nn::g3d::ShadingModelObj* pShadingModelObj = nns::g3d::CreateShadingModelObj(pDevice, shadingModelbuilder);
            NN_ASSERT_NOT_NULL(pShadingModelObj);

            nn::g3d::ShaderSelector::Builder selectorBuilder(pShadingModelObj);
            m_pSelector = nns::g3d::CreateShaderSelector(selectorBuilder);
            NN_ASSERT_NOT_NULL(m_pSelector);
        }
        {
            nn::gfx::Sampler::InfoType info;
            info.SetDefault();
            info.SetFilterMode(nn::gfx::FilterMode::FilterMode_MinPoint_MagPoint_MipLinear);
            m_Sampler.Initialize(pDevice, info);
            m_SamplerDescriptorSlot = nn::g3d::demo::RegisterSamplerToDescriptorPool(&m_Sampler);
        }
        m_SamplderIndex = pResShadingModel->FindSamplerIndex("TargetTexture");
        NN_ASSERT(m_SamplderIndex >= 0);

        // コンピュートシェーダー用の演算バッファーを作成
        {
            nns::gfx::GraphicsFramework* pGfxFramework = nn::g3d::demo::GetGfxFramework();
            nn::gfx::BufferInfo info;
            info.SetDefault();
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_UnorderedAccessBuffer | nn::gfx::GpuAccess_Read);
            info.SetSize(bufferSize);
            for (int bufferIndex = 0; bufferIndex < bufferCount; ++bufferIndex)
            {
                ptrdiff_t offset = pGfxFramework->AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Others,
                    bufferSize, nn::gfx::Buffer::GetBufferAlignment(pDevice, info));
                NN_ASSERT(offset != -1);
                m_ExportBuffer[bufferIndex].Initialize(pDevice, info, pGfxFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_Others), offset, bufferSize);
                m_ExportBufferMemoryPoolOffset[bufferIndex] = offset;
            }
        }
    }

    void Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT
    {
        if (m_pSelector)
        {
            nn::g3d::ShadingModelObj* pShadingModelObj = m_pSelector->GetShadingModel();
            nns::g3d::DestroyShadingModelObj(pDevice, pShadingModelObj);
            nns::g3d::DestroyShaderSelector(m_pSelector);
            m_pSelector = nullptr;
        }

        nn::g3d::demo::UnregisterSamplerFromDescriptorPool(&m_Sampler);
        m_Sampler.Finalize(pDevice);

        nns::gfx::GraphicsFramework* pGfxFramework = nn::g3d::demo::GetGfxFramework();
        // コンピュートシェーダー用の演算バッファーを破棄
        for (int bufferIndex = 0; bufferIndex < bufferCount; ++bufferIndex)
        {
            m_ExportBuffer[bufferIndex].Finalize(pDevice);
            pGfxFramework->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Others, m_ExportBufferMemoryPoolOffset[bufferIndex]);
        }
    }

    void SetTileResolution(nn::gfx::Device* pDevice, uint32_t x, uint32_t y) NN_NOEXCEPT
    {
        char str[16];
        sprintf(str, "%d", x);
        int optionIndex = m_pSelector->GetShadingModel()->GetResource()->FindDynamicOptionIndex("tile_resolution_x");
        NN_ASSERT(optionIndex != nn::util::ResDic::Npos);
        int choice = m_pSelector->GetDynamicOption(optionIndex)->FindChoiceIndex(str);
        NN_ASSERT(choice != nn::util::ResDic::Npos);
        m_pSelector->WriteDynamicKey(optionIndex, choice);
        sprintf(str, "%d", y);
        optionIndex = m_pSelector->GetShadingModel()->GetResource()->FindDynamicOptionIndex("tile_resolution_y");
        NN_ASSERT(optionIndex != nn::util::ResDic::Npos);
        choice = m_pSelector->GetDynamicOption(optionIndex)->FindChoiceIndex(str);
        NN_ASSERT(choice != nn::util::ResDic::Npos);
        m_pSelector->WriteDynamicKey(optionIndex, choice);

        bool isSucceeded = m_pSelector->UpdateVariation(pDevice);
        NN_UNUSED(isSucceeded);
        NN_ASSERT(isSucceeded);

        const nn::g3d::ResShadingModel* pResShadingModel = m_pSelector->GetShadingModel()->GetResource();
        int shaderStorageBlockIndex = pResShadingModel->FindShaderStorageBlockIndex("exp_buf");
        m_ExportBufferSlot = m_pSelector->GetProgram()->GetShaderStorageBlockLocation(shaderStorageBlockIndex, nn::g3d::Stage_Compute);
    }

    void MakeCommand(nn::gfx::CommandBuffer* pCommandBuffer, int bufferIndex, nn::gfx::DescriptorSlot inputTextureDescriptorSlot) NN_NOEXCEPT
    {
        if (!m_pSelector->GetProgram()->IsInitialized())
        {
            return;
        }

        m_pSelector->GetProgram()->Load(pCommandBuffer);
        nn::g3d::demo::LoadTextureSampler(pCommandBuffer, inputTextureDescriptorSlot, m_SamplerDescriptorSlot, m_pSelector->GetProgram(), m_SamplderIndex);
        nn::gfx::GpuAddress gpuAddress;
        m_ExportBuffer[bufferIndex].GetGpuAddress(&gpuAddress);
        pCommandBuffer->SetUnorderedAccessBuffer(m_ExportBufferSlot, nn::gfx::ShaderStage_Compute, gpuAddress, bufferSize);

        pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_UnorderedAccessBuffer);
        pCommandBuffer->Dispatch(tileDimX, tileDimY, 1);
        pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_UnorderedAccessBuffer);
    }

    void CalculateAverage(int bufferIndex) NN_NOEXCEPT
    {
        m_AverageResult[bufferIndex] = NN_UTIL_FLOAT_4_INITIALIZER(0.0f, 0.0f, 0.0f, 0.0f);
        {
            nn::util::Float4* buffer = m_ExportBuffer[bufferIndex].Map<nn::util::Float4>();
            m_ExportBuffer[bufferIndex].InvalidateMappedRange(0, bufferSize);
            for (int resultIndex = 0; resultIndex < tileCount; ++resultIndex)
            {
                for (int index = 0; index < 4; ++index)
                {
                    m_AverageResult[bufferIndex].v[index] += buffer[resultIndex].v[index];
                }
            }
            for (int index = 0; index < 4; ++index)
            {
                m_AverageResult[bufferIndex].v[index] /= static_cast<float>(tileCount);
            }
            m_ExportBuffer[bufferIndex].Unmap();
        }
    }

    nn::util::Float4& GetTextureAverage(int bufferIndex) NN_NOEXCEPT
    {
        return m_AverageResult[bufferIndex];
    }

    bool UpdateShader(nn::gfx::Device* pDevice) NN_NOEXCEPT
    {
        return m_pSelector->UpdateVariation(pDevice);
    }

private:
    nn::util::Float4         m_AverageResult[bufferCount];
    nn::gfx::Buffer          m_ExportBuffer[bufferCount];
    nn::gfx::Sampler         m_Sampler;
    nn::gfx::DescriptorSlot  m_SamplerDescriptorSlot;
    nn::g3d::ShaderSelector* m_pSelector;
    int                      m_SamplderIndex;
    int                      m_ExportBufferSlot;
    ptrdiff_t                m_ExportBufferMemoryPoolOffset[bufferCount];
};
