﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{NnsG3dSimple.cpp,PageSampleG3dNnsG3dSimple}
 *
 * @brief
 * nns::g3d を使用したモデル表示のサンプルプログラム
 */

/**
 * @page PageSampleG3dNnsG3dSimple G3dDemo NnsG3dSimple
 * @tableofcontents
 *
 * @image html Applications\G3dDemo\nnsg3dsimple.png
 *
 * @brief
 * サンプルプログラム NnsG3dSimple の解説です。 nns::g3d を使用し、モデル表示を行います。
 *
 * @section PageSampleG3dNnsG3dSimple_SectionBrief 概要
 * nns::g3d を使用したモデル表示を行うサンプルです。
 *
 * @section PageSampleG3dNnsG3dSimple_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/G3dDemo
 * Samples/Sources/Applications/G3dDemo @endlink 以下にあります。
 *
 * @section PageSampleG3dNnsG3dSimple_SectionNecessaryEnvironment 必要な環境
 * 特になし。
 *
 * @section PageSampleG3dNnsG3dSimple_SectionHowToOperate 操作方法
 * Joy-Con、DebugPad、PCの場合キーボード、マウスによる入力を用いて、以下の操作が可能です。
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> Startボタン、スペースキーまたはタッチスクリーン長押し、Joy-Con の+ボタン </td><td> メニューに戻ります。 </td></tr>
 * <tr><td> 左スティックまたはマウス左ボタン押し </td><td> カメラ回転。マウスの場合、画面の中心点からクリック位置への方向、距離によって回転します。 </td></tr>
 * <tr><td> 右スティックまたはマウス右ボタン押し </td><td> カメラ移動。マウスの場合、画面の中心点からクリック位置への方向、距離によって移動します。 </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleG3dNnsG3dSimple_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleG3dNnsG3dSimple_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、メニューから NnsG3dSimple を選択し、実行してください。
 *
 * @section PageSampleG3dNnsG3dSimple_SectionDetail 解説
 * このサンプルプログラムは、 nns::g3d を使用し、モデルを表示します。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - シェーダーアーカイブ読み込み、初期化
 * - モデルリソース読み込み、初期化およびテクスチャー初期化
 * - サンプラーをディスクリプタプールに登録
 * - nns::g3d を使用したモデルのセットアップ
 * - ループ開始
 * - 前フレームで作成したコマンドリスト実行
 * - 各行列更新
 * - モデル更新
 * - コマンドリスト生成
 * - ループ開始に戻る
 *
 */

#include "g3ddemo_GfxUtility.h"
#include "g3ddemo_ModelUtility.h"

namespace g3ddemo = nn::g3d::demo;

namespace {

#define RESOURCE_PATH "Resource:/"
const char* g_BfshaPath      = RESOURCE_PATH "shader/demo.bfsha";
const char* g_EnvBfresPath   = RESOURCE_PATH "env.bfres";
const char* g_ModelBfresPath = RESOURCE_PATH "human.bfres";

nn::util::Vector3fType g_CameraPosition;
nn::util::Vector3fType g_CameraUp;
nn::util::Vector3fType g_CameraTarget;

g3ddemo::ResourceHolder g_Holder;
g3ddemo::ScreenInfo     g_ScreenInfo;
nns::g3d::RenderView    g_RenderView;

const int g_BufferingCount = 2;

void Calculate(nns::gfx::GraphicsFramework* pGfxFramework, void* pUserData)
{
    NN_UNUSED(pGfxFramework);

    int uniformBufferIndex = *reinterpret_cast<int*>(pUserData);

    // 計算
    {
        // カメラコントロール
        g3ddemo::Pad& pad = g3ddemo::GetPad();
        ControlCamera(&g_CameraPosition, &g_CameraUp, &g_CameraTarget, &g_RenderView.GetViewMtx(0), &pad);
        {
            nn::util::Matrix4x3fType matrix;
            MatrixLookAtRightHanded(&matrix,
                g_CameraPosition,
                g_CameraTarget,
                g_CameraUp);
            g_RenderView.SetViewMtx(0, matrix);
        }

        // 投影行列
        {
            nn::util::Matrix4x4fType matrix;
            MatrixPerspectiveFieldOfViewRightHanded(&matrix, nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f, 10.0f, 40000.0f);
            g_RenderView.SetProjectionMtx(0, matrix);
        }

        // ビューの更新処理
        g_RenderView.Calculate(uniformBufferIndex);

        // モデルの更新処理
        for (int index = 0; index < g_Holder.modelAnimObjs.GetCount(); ++index)
        {
            g_Holder.modelAnimObjs[index]->CalculateView(0, uniformBufferIndex, g_RenderView.GetViewMtx(0));
            g_Holder.modelAnimObjs[index]->Calculate();
            g_Holder.modelAnimObjs[index]->CalculateUniformBlock(uniformBufferIndex);
        }
    }
}

void MakeCommand(nns::gfx::GraphicsFramework* pGfxFramework, int bufferIndex, void* pUserData)
{
    int uniformBufferIndex = *reinterpret_cast<int*>(pUserData);

    // コマンド生成
    pGfxFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* pCommandBuffer = pGfxFramework->GetRootCommandBuffer(bufferIndex);

        // カラーターゲットをクリア
        nn::gfx::ColorTargetView* pColor = pGfxFramework->GetColorTargetView();
        pCommandBuffer->ClearColor(pColor, 0.0f, 0.0f, 0.0f, 1.0f, nullptr);

        // 深度ステンシルをクリア
        nn::gfx::DepthStencilView* pDepth = pGfxFramework->GetDepthStencilView();
        pCommandBuffer->ClearDepthStencil(pDepth, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

        // レンダーターゲットをセット
        pCommandBuffer->SetRenderTargets(1, &pColor, pDepth);

        // ビューポートシザーステートをセット
        pCommandBuffer->SetViewportScissorState(pGfxFramework->GetViewportScissorState());

        // モデル表示 コマンド生成
        for (int index = 0; index < g_Holder.renderModelObjs.GetCount(); ++index)
        {
            g_Holder.renderModelObjs[index]->Draw(pCommandBuffer, 0, uniformBufferIndex);
        }

        // テキスト表示 コマンド生成
        {
            g_ScreenInfo.StartPrint();
            g_ScreenInfo.Print("[NnsG3dSimple]\n");
            g_ScreenInfo.EndPrint();
            g_ScreenInfo.AdjustWindowSize();
            g_ScreenInfo.Draw(pCommandBuffer);
        }
    }
    pGfxFramework->EndFrame(bufferIndex);
}

} // anonymous namespace

//--------------------------------------------------------------------------------------------------
int NnsG3dSimpleMain()
{
    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    pGfxFramework->SetFrameworkMode(nns::gfx::GraphicsFramework::FrameworkMode_DeferredSubmission);

    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // スクリーン情報表示を初期化
    g_ScreenInfo.Initialize(pDevice);
    g_ScreenInfo.SetWindowPosition(16, 16);
    g_ScreenInfo.SetFontSize(20);

    g_Holder.Initialize();

    // シェーダーアーカイブの初期化
    nn::g3d::ResShaderArchive* pDemoShaderArchive;
    {
        nn::g3d::ResShaderFile* pFile = g3ddemo::LoadResource<nn::g3d::ResShaderFile>(g_BfshaPath);
        pFile->Setup(pDevice);
        g_Holder.shaderFiles.PushBack(pFile);
        pDemoShaderArchive = pFile->GetResShaderArchive();
    }

    // モデルリソースの初期化
    nn::g3d::ResFile* pEnvModelFile;
    {
        pEnvModelFile = g3ddemo::LoadResource<nn::g3d::ResFile>(g_EnvBfresPath);
        pEnvModelFile->Setup(pDevice);
        g_Holder.files.PushBack(pEnvModelFile);
    }
    nn::g3d::ResFile* pHumanModelFile;
    {
        pHumanModelFile = g3ddemo::LoadResource<nn::g3d::ResFile>(g_ModelBfresPath);
        pHumanModelFile->Setup(pDevice);
        g3ddemo::SetupTexture(pDevice, pHumanModelFile, g3ddemo::TextureBindCallback);
        g3ddemo::RegisterSamplerToDescriptorPool(pHumanModelFile);
        g_Holder.files.PushBack(pHumanModelFile);
    }

    // モデルの頂点ステート、デプスステンシルステート、ブレンドステート、ラスタライザーステート、シェーディングモデルを管理するRenderModelを作成
    nns::g3d::RenderModel* pDemoRenderModel;
    {
        nn::g3d::ResModel* pResModel = pHumanModelFile->GetModel(0);
        pDemoRenderModel = nns::g3d::CreateRenderModel(pDevice, pResModel);
        pDemoRenderModel->CreateDrawPass(pDevice, pDemoShaderArchive);
        g_Holder.renderModels.PushBack(pDemoRenderModel);
    }

    // モデルおよびアニメーションの計算処理を簡略化するModelAnimObjを作成
    nns::g3d::ModelAnimObj* pHumanModelAnimObj;
    {
        nn::g3d::ModelObj::Builder builder(pHumanModelFile->GetModel(0));
        // CPUとGPUで並列実行出来るようにバッファー数に2を指定
        builder.ShapeBufferingCount(g_BufferingCount);
        builder.SkeletonBufferingCount(g_BufferingCount);
        builder.MaterialBufferingCount(g_BufferingCount);
        pHumanModelAnimObj = nns::g3d::CreateModelAnimObj(pDevice, builder);
        g_Holder.modelAnimObjs.PushBack(pHumanModelAnimObj);
        // アニメーション追加
        pHumanModelAnimObj->CreateAnim(pHumanModelFile->FindSkeletalAnim("human_run"));
    }

    // RenderModelとModelObjを組み合わせ、描画を行うRenderModelObjを作成
    nns::g3d::RenderModelObj* pHumanRenderModelObj;
    {
        pHumanRenderModelObj = nns::g3d::CreateRenderModelObj(pHumanModelAnimObj->GetModelObj(), pDemoRenderModel);
        g_Holder.renderModelObjs.PushBack(pHumanRenderModelObj);
        // このデモではシェーダーを動的に切り替えないため、ここでシェーダーを選択
        g3ddemo::WriteSkinningOption(pHumanRenderModelObj);
        pHumanRenderModelObj->UpdateShader(pDevice);
    }

    // 描画時に必要となるg3dの管理外のユニフォームブロックを作成し、RenderModelObjに設定
    // ユニフォームブロックの管理を簡単にするため、別モデルのマテリアルユニフォームブロックとして扱っています
    nn::g3d::ModelObj* pEnvModelObj;
    {
        nn::g3d::ResModel* pResModel = pEnvModelFile->GetModel(0);
        nn::g3d::ShaderUtility::BindShaderParam(pResModel->GetMaterial(0), pDemoShaderArchive->FindShadingModel("env"));
        nn::g3d::ModelObj::Builder builder(pResModel);
        builder.MaterialBufferingCount(g_BufferingCount);
        pEnvModelObj = nns::g3d::CreateModelObj(pDevice, builder);
        // このデモでは毎フレーム更新が必要でないため、ここで1度だけ計算を行う
        pEnvModelObj->CalculateMaterial(0);
        pEnvModelObj->CalculateMaterial(1);
        g_Holder.modelObjs.PushBack(pEnvModelObj);
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = pEnvModelObj->GetMaterial(0)->GetMaterialBlock(0);
        buffers[1] = pEnvModelObj->GetMaterial(0)->GetMaterialBlock(1);
        pHumanRenderModelObj->BindUniformBlock("env", buffers, g_BufferingCount);
    }

    // ビュー用のユニフォームブロックを設定
    g_RenderView.Initialize(pDevice, 1, g_BufferingCount);
    {
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = g_RenderView.GetViewUniformBlock(0, 0);
        buffers[1] = g_RenderView.GetViewUniformBlock(0, 1);
        pHumanRenderModelObj->BindUniformBlock("view", buffers, g_BufferingCount);
    }

    // カメラ初期化
    {
        VectorSet(&g_CameraPosition, 125.4f, 138.6f, 198.0f);
        VectorSet(&g_CameraUp, 0.0f, 1.0f, 0.0f);
        VectorSet(&g_CameraTarget, 0.0f, 82.5f, 0.0f);

        nn::util::Matrix4x3fType matrix;
        MatrixLookAtRightHanded(&matrix,
            g_CameraPosition,
            g_CameraTarget,
            g_CameraUp);
        g_RenderView.SetViewMtx(0, matrix);
    }

    // シェーダーに必要なスクラッチメモリーを設定
    g3ddemo::SetShaderScratchMemory(g_Holder.shaderFiles);

    int uniformBufferIndex = 0;
    pGfxFramework->SetCalculateCallback(Calculate, &uniformBufferIndex);
    pGfxFramework->SetMakeCommandCallback(MakeCommand, &uniformBufferIndex);
    pGfxFramework->ResetFrame();

    g3ddemo::Pad& pad = g3ddemo::GetPad();
    pad.Reset();
    // メインループ
    while (NN_STATIC_CONDITION(1))
    {
        // タッチ長押しで終了
        if (g3ddemo::isExitDemo())
        {
            break;
        }

        if (!pad.Read() || pad.IsTriggered(g3ddemo::Pad::BUTTON_START))
        {
            break;
        }

        // フレーム処理実行
        pGfxFramework->ProcessFrame();

        uniformBufferIndex ^= 1;
    }
    pGfxFramework->QueueFinish();

    // 終了処理
    g3ddemo::DestroyAll(pDevice, &g_Holder);

    // ビュー用ユニフォームブロックの破棄
    {
        g_RenderView.Finalize(pDevice);
    }

    // スクリーン情報表示の破棄
    {
        g_ScreenInfo.Cleanup(pDevice);
    }

    pGfxFramework->SetCalculateCallback(nullptr, nullptr);
    pGfxFramework->SetMakeCommandCallback(nullptr, nullptr);

    return EXIT_SUCCESS;
} // NOLINT
