﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Edit.h"
#include <nn/g3d/g3d_Viewer.h>

namespace
{
    void AttachToEdit(nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pModelObj);
        if (nn::g3d::viewer::ViewerServer::GetInstance().IsConnected() &&
            !(nn::g3d::viewer::ViewerServer::GetInstance().IsModelAttached(pModelObj)
                || nn::g3d::viewer::ViewerServer::GetInstance().IsModelAttaching(pModelObj)))
        {
            nn::g3d::viewer::ViewerServer::SendUserMessageArg userMessageArg(
                "モデルのアタッチが実行されました。",
                nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageType_Info,
                nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageDestination_Log);
            nn::g3d::viewer::ViewerServer::GetInstance().QueueSendUserMessageCommand(userMessageArg);

            NN_LOG("Attach %s\n", pModelObj->GetResource()->GetName());
            nn::g3d::viewer::ViewerResult result = nn::g3d::viewer::ViewerServer::GetInstance().QueueAttachModelCommand(pModelObj);
            NN_ASSERT(result == nn::g3d::viewer::ViewerResult_Success
                || result == nn::g3d::viewer::ViewerResult_AlreadyAttached);
        }
    }

    void DetachFromEdit(nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pModelObj);
        if (nn::g3d::viewer::ViewerServer::GetInstance().IsConnected() &&
            (nn::g3d::viewer::ViewerServer::GetInstance().IsModelAttached(pModelObj)
                || nn::g3d::viewer::ViewerServer::GetInstance().IsModelAttaching(pModelObj)))
        {
            nn::g3d::viewer::ViewerServer::SendUserMessageArg userMessageArg(
                "モデルのデタッチが実行されました。",
                nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageType_Info,
                nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageDestination_Log);
            nn::g3d::viewer::ViewerServer::GetInstance().QueueSendUserMessageCommand(userMessageArg);

            NN_LOG("Detach %s\n", pModelObj->GetResource()->GetName());
            nn::g3d::viewer::ViewerResult result = nn::g3d::viewer::ViewerServer::GetInstance().QueueDetachModelCommand(pModelObj);
            NN_ASSERT(result == nn::g3d::viewer::ViewerResult_Success
                || result == nn::g3d::viewer::ViewerResult_NotAttached);
        }
    }

    void CreateAnim(nns::g3d::ModelAnimObj* pModelAnimObj, void* pResAnim) NN_NOEXCEPT
    {
        const nn::util::BinaryBlockHeader* pHeader =
            reinterpret_cast<const nn::util::BinaryBlockHeader*>(pResAnim);

        if (pHeader->signature.IsValid(static_cast<int32_t>(nn::g3d::ResSkeletalAnim::Signature)))
        {
            const nn::g3d::ResSkeletalAnim* pResSkeltalAnim = reinterpret_cast<const nn::g3d::ResSkeletalAnim*>(pResAnim);
            if (pModelAnimObj->FindAnim(pResSkeltalAnim) == nns::g3d::ModelAnimObj::InvalidAnimId)
            {
                pModelAnimObj->CreateAnim(pResSkeltalAnim);
            }
        }
        else if (pHeader->signature.IsValid(static_cast<int32_t>(nn::g3d::ResBoneVisibilityAnim::Signature)))
        {
            const nn::g3d::ResBoneVisibilityAnim* pResBoneVisiblityAnim = reinterpret_cast<const nn::g3d::ResBoneVisibilityAnim*>(pResAnim);
            if (pModelAnimObj->FindAnim(pResBoneVisiblityAnim) == nns::g3d::ModelAnimObj::InvalidAnimId)
            {
                pModelAnimObj->CreateAnim(pResBoneVisiblityAnim);
            }
        }
        else if (pHeader->signature.IsValid(static_cast<int32_t>(nn::g3d::ResMaterialAnim::Signature)))
        {
            const nn::g3d::ResMaterialAnim* pResMaterialAnim = reinterpret_cast<const nn::g3d::ResMaterialAnim*>(pResAnim);
            if (pModelAnimObj->FindAnim(pResMaterialAnim) == nns::g3d::ModelAnimObj::InvalidAnimId)
            {
                pModelAnimObj->CreateAnim(pResMaterialAnim);
            }
        }
        else if (pHeader->signature.IsValid(static_cast<int32_t>(nn::g3d::ResShapeAnim::Signature)))
        {
            const nn::g3d::ResShapeAnim* pResShapeAnim = reinterpret_cast<const nn::g3d::ResShapeAnim*>(pResAnim);
            if (pModelAnimObj->FindAnim(pResShapeAnim) == nns::g3d::ModelAnimObj::InvalidAnimId)
            {
                pModelAnimObj->CreateAnim(pResShapeAnim);
            }
        }
    }

    int FindAnim(nns::g3d::ModelAnimObj* pModelAnimObj, void* pResAnim) NN_NOEXCEPT
    {
        const nn::util::BinaryBlockHeader* pHeader =
            reinterpret_cast<const nn::util::BinaryBlockHeader*>(pResAnim);

        int id = nns::g3d::ModelAnimObj::InvalidAnimId;
        if (pHeader->signature.IsValid(static_cast<int32_t>(nn::g3d::ResSkeletalAnim::Signature)))
        {
            id = pModelAnimObj->FindAnim(reinterpret_cast<const nn::g3d::ResSkeletalAnim*>(pResAnim));
        }
        else if (pHeader->signature.IsValid(static_cast<int32_t>(nn::g3d::ResBoneVisibilityAnim::Signature)))
        {
            id = pModelAnimObj->FindAnim(reinterpret_cast<const nn::g3d::ResBoneVisibilityAnim*>(pResAnim));
        }
        else if (pHeader->signature.IsValid(static_cast<int32_t>(nn::g3d::ResMaterialAnim::Signature)))
        {
            id = pModelAnimObj->FindAnim(reinterpret_cast<const nn::g3d::ResMaterialAnim*>(pResAnim));
        }
        else if (pHeader->signature.IsValid(static_cast<int32_t>(nn::g3d::ResShapeAnim::Signature)))
        {
            id = pModelAnimObj->FindAnim(reinterpret_cast<const nn::g3d::ResShapeAnim*>(pResAnim));
        }

        return id;
    }

    const char* FindAttachOrDetachString(nn::g3d::ResShaderArchive* pShaderArchive) NN_NOEXCEPT
    {
        if (nn::g3d::viewer::ViewerServer::GetInstance().IsShaderArchiveAttached(pShaderArchive))
        {
            return "Detach";
        }

        return "Attach";
    }

    void Clamp(int& value, int min, int max) NN_NOEXCEPT
    {
        value = std::max NN_PREVENT_MACRO_FUNC(std::min NN_PREVENT_MACRO_FUNC(value, max), min);
    }

} // anonymous namespace

void EditMenu::CalculateCPU(EditViewer* pEditViewer,const g3ddemo::Pad &pad) NN_NOEXCEPT
{
    if (pad.IsTriggered(g3ddemo::Pad::BUTTON_DOWN))
    {
        m_SelectedItemIndex = (m_SelectedItemIndex + 1) % ItemCount;
    }
    if (pad.IsTriggered(g3ddemo::Pad::BUTTON_UP))
    {
        m_SelectedItemIndex = (m_SelectedItemIndex + ItemCount - 1) % ItemCount;
    }
    if (pad.IsTriggered(g3ddemo::Pad::BUTTON_X))
    {
        nn::g3d::ResShaderArchive* debugShaderArchive = m_pResourceHolder->shaderFiles[1]->GetResShaderArchive();
        if (nn::g3d::viewer::ViewerServer::GetInstance().IsConnected())
        {
            bool isAttached = nn::g3d::viewer::ViewerServer::GetInstance().IsShaderArchiveAttached(debugShaderArchive);
            bool isAttaching = nn::g3d::viewer::ViewerServer::GetInstance().IsShaderArchiveAttaching(debugShaderArchive);
            if (!isAttached && !isAttaching)
            {
                // アタッチされていない
                g3ddemo::SendMessageTo3dEditor("debug シェーダーのアタッチが実行されました。");
                nn::g3d::viewer::ViewerResult result = nn::g3d::viewer::ViewerServer::GetInstance().QueueAttachShaderArchiveCommand(debugShaderArchive);
                NN_ASSERT(result == nn::g3d::viewer::ViewerResult_Success
                    || result == nn::g3d::viewer::ViewerResult_AlreadyAttached);
            }
            else if (isAttached)
            {
                g3ddemo::SendMessageTo3dEditor("debug シェーダーのデタッチが実行されました。");
                nn::g3d::viewer::ViewerResult result = nn::g3d::viewer::ViewerServer::GetInstance().QueueDetachShaderArchiveCommand(debugShaderArchive);
                NN_ASSERT(result == nn::g3d::viewer::ViewerResult_Success
                    || result == nn::g3d::viewer::ViewerResult_AlreadyAttached);
            }
        }
    }
    if (pad.IsTriggered(g3ddemo::Pad::BUTTON_Y))
    {
        // Edit.cpp に IsDemoShaderAttached と IsDebugShaderAttached を作る
        nn::g3d::ResShaderArchive* pArchive = m_pBranchShader;
        if (nn::g3d::viewer::ViewerServer::GetInstance().IsConnected())
        {
            if (!nn::g3d::viewer::ViewerServer::GetInstance().IsShaderArchiveAttached(pArchive))
            {
                g3ddemo::SendMessageTo3dEditor("demo シェーダーのアタッチが実行されました。");
                nn::g3d::viewer::ViewerResult result = nn::g3d::viewer::ViewerServer::GetInstance().QueueAttachShaderArchiveCommand(pArchive);
                NN_ASSERT(result == nn::g3d::viewer::ViewerResult_Success
                    || result == nn::g3d::viewer::ViewerResult_AlreadyAttached);
            }
            else
            {
                g3ddemo::SendMessageTo3dEditor("demo シェーダーのデタッチが実行されました。");
                nn::g3d::viewer::ViewerResult result = nn::g3d::viewer::ViewerServer::GetInstance().QueueDetachShaderArchiveCommand(pArchive);
                NN_ASSERT(result == nn::g3d::viewer::ViewerResult_Success
                    || result == nn::g3d::viewer::ViewerResult_AlreadyAttached);
            }
        }
    }

    switch (m_SelectedItemIndex)
    {
    case ItemModelInstanceCount:
    {
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
        {
            ++m_SelectedItemIndexArray[ItemModelInstanceCount];
            pEditViewer->SetModelInstanceCount(m_SelectedItemIndexArray[ItemModelInstanceCount]);
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
        {
            --m_SelectedItemIndexArray[ItemModelInstanceCount];
            m_SelectedItemIndexArray[ItemModelInstanceCount] = m_SelectedItemIndexArray[ItemModelInstanceCount] < 1 ? 1 : m_SelectedItemIndexArray[ItemModelInstanceCount];
            pEditViewer->SetModelInstanceCount(m_SelectedItemIndexArray[ItemModelInstanceCount]);
        }
    }
    break;
    case ItemModel:
    {
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_A))
        {
            // 3DEditor の編集対象として ModelObj を登録します
            // 同じリソースから作られた ModelObj を一括登録するために、通信スレッドを一時停止します
            pEditViewer->PausePollThread();

            nns::g3d::ModelAnimObj* pSelectedModelAnimObj = m_pResourceHolder->modelAnimObjs[m_SelectedItemIndexArray[ItemModel]];
            for (nns::g3d::ModelAnimObj* pModelAnimObj : m_pResourceHolder->modelAnimObjs)
            {
                if (pSelectedModelAnimObj->GetModelObj()->GetResource() == pModelAnimObj->GetModelObj()->GetResource())
                {
                    AttachToEdit(pModelAnimObj->GetModelObj());
                }
            }
            pEditViewer->ResumePollThread();
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_B))
        {
            nns::g3d::ModelAnimObj* pModelAnimObj = m_pResourceHolder->modelAnimObjs[m_SelectedItemIndexArray[ItemModel]];
            DetachFromEdit(pModelAnimObj->GetModelObj());
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
        {
            m_SelectedItemIndexArray[ItemModel] = (m_SelectedItemIndexArray[ItemModel] + 1) % m_pResourceHolder->modelAnimObjs.GetCount();
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
        {
            int modelCount = m_pResourceHolder->modelAnimObjs.GetCount();
            m_SelectedItemIndexArray[ItemModel] = (m_SelectedItemIndexArray[ItemModel] + modelCount - 1) % modelCount;
        }

        Clamp(m_SelectedItemIndexArray[ItemModel], 0, m_pResourceHolder->modelAnimObjs.GetCount() - 1);
    }
    break;
    case ItemAnim:
    {
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_A))
        {
            nns::g3d::ModelAnimObj* pModelAnimObj = m_pResourceHolder->modelAnimObjs[m_SelectedItemIndexArray[ItemModel]];
            void* pResAnim = m_pResourceHolder->modelAnims[m_SelectedItemIndexArray[ItemAnim]];
            CreateAnim(pModelAnimObj, pResAnim);
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_B))
        {
            nns::g3d::ModelAnimObj* pModelAnimObj = m_pResourceHolder->modelAnimObjs[m_SelectedItemIndexArray[ItemModel]];
            void* pResAnim = m_pResourceHolder->modelAnims[m_SelectedItemIndexArray[ItemAnim]];
            int id = FindAnim(pModelAnimObj, pResAnim);
            if (id != nns::g3d::ModelAnimObj::InvalidAnimId)
            {
                pModelAnimObj->DestroyAnim(id);
            }
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
        {
            m_SelectedItemIndexArray[ItemAnim] = (m_SelectedItemIndexArray[ItemAnim] + 1) % m_pResourceHolder->modelAnims.GetCount();
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
        {
            int animCount = m_pResourceHolder->modelAnims.GetCount();
            m_SelectedItemIndexArray[ItemAnim] = (m_SelectedItemIndexArray[ItemAnim] + animCount - 1) % animCount;
        }
    }
    break;
    case ItemShader:
    {
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
        {
            m_SelectedItemIndexArray[ItemShader] = (m_SelectedItemIndexArray[ItemShader] + 1) % m_pOutputOption->GetChoiceCount();
            WriteDynamicKey();
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
        {
            int choiceCount = m_pOutputOption->GetChoiceCount();
            m_SelectedItemIndexArray[ItemShader] = (m_SelectedItemIndexArray[ItemShader] + choiceCount - 1) % choiceCount;
            WriteDynamicKey();
        }
    }
    break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
} // NOLINT

void EditMenu::CalculateGPU(g3ddemo::ScreenInfo& screenInfo) NN_NOEXCEPT
{
    const char* allowString = ">";
    screenInfo.StartPrint();
    screenInfo.Print("[Edit]\n");
    screenInfo.Print("A:   Attach Model\n");
    screenInfo.Print("B:   Detach Model\n");
    screenInfo.Print("X:   %s Debug Shader\n", FindAttachOrDetachString(m_pResourceHolder->shaderFiles[1]->GetResShaderArchive()));
    screenInfo.Print("Y:   %s Demo Shader\n", FindAttachOrDetachString(m_pBranchShader));

    screenInfo.Print("%s Model Instance Count: %d\n",
        m_SelectedItemIndex == ItemModelInstanceCount ? allowString : "  ",
        m_SelectedItemIndexArray[ItemModelInstanceCount]);

    int selectedModelIndex =  m_SelectedItemIndexArray[ItemModel] < m_pResourceHolder->modelAnimObjs.GetCount() ? m_SelectedItemIndexArray[ItemModel] : m_pResourceHolder->modelAnimObjs.GetCount() - 1;

    if (!m_pResourceHolder->renderModelObjs.Empty())
    {
        nns::g3d::ModelAnimObj* pModelAnimObj = m_pResourceHolder->modelAnimObjs[selectedModelIndex];
        nn::g3d::ModelObj* pModelObj = pModelAnimObj->GetModelObj();
        const nn::g3d::ResModel* pResModel = pModelObj->GetResource();

        bool isAttached = nn::g3d::viewer::ViewerServer::GetInstance().IsModelAttached(pModelObj);
        screenInfo.Print("%s [%d/%d] %c %s.fmdb\n",
            m_SelectedItemIndex == ItemModel ? allowString : "  ",
            selectedModelIndex + 1, m_pResourceHolder->modelAnimObjs.GetCount(), isAttached ? '+' : '-',
            pResModel->GetName());
    }
    else
    {
        screenInfo.Print("%s [0/0] No model\n", m_SelectedItemIndex == ItemModel ? allowString : "  ");
    }

    if (!m_pResourceHolder->modelAnims.Empty())
    {
        void* ptr = m_pResourceHolder->modelAnims[m_SelectedItemIndexArray[ItemAnim]];
        nn::util::BinaryBlockHeader* pHeader = reinterpret_cast<nn::util::BinaryBlockHeader*>(ptr);
        const char* name = "";
        const char* ext = "";
        switch (pHeader->signature.GetPacked())
        {
        case nn::g3d::ResSkeletalAnim::Signature:
        {
            nn::g3d::ResSkeletalAnim* pAnim = reinterpret_cast<nn::g3d::ResSkeletalAnim*>(ptr);
            name = pAnim->GetName();
            ext = "fskb";
        }
        break;
        case nn::g3d::ResMaterialAnim::Signature:
        {
            nn::g3d::ResMaterialAnim* pAnim = reinterpret_cast<nn::g3d::ResMaterialAnim*>(ptr);
            name = pAnim->GetName();
            ext = "fmab";
        }
        break;
        case nn::g3d::ResBoneVisibilityAnim::Signature:
        {
            nn::g3d::ResBoneVisibilityAnim* pAnim = reinterpret_cast<nn::g3d::ResBoneVisibilityAnim*>(ptr);
            name = pAnim->GetName();
            ext = "fvbb";
        }
        break;
        case nn::g3d::ResShapeAnim::Signature:
        {
            nn::g3d::ResShapeAnim* pAnim = reinterpret_cast<nn::g3d::ResShapeAnim*>(ptr);
            name = pAnim->GetName();
            ext = "fshb";
        }
        break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        nns::g3d::ModelAnimObj* pModelAnimObj = m_pResourceHolder->modelAnimObjs[selectedModelIndex];
        char enable = FindAnim(pModelAnimObj, ptr) != nns::g3d::ModelAnimObj::InvalidAnimId ? '+' : '-';
        screenInfo.Print("%s [%d/%d] %c %s.%s\n",
            m_SelectedItemIndex == ItemAnim ? allowString : "  ",
            m_SelectedItemIndexArray[ItemAnim] + 1, m_pResourceHolder->modelAnims.GetCount(), enable, name, ext);
    }
    else
    {
        screenInfo.Print("%s [0/0] No animation\n", m_SelectedItemIndex == ItemAnim ? allowString : "  ");
    }
    screenInfo.Print("%s [%d/%d] %s\n",
        m_SelectedItemIndex == ItemShader ? allowString : "  ",
        m_SelectedItemIndexArray[ItemShader] + 1,
        m_pOutputOption->GetChoiceCount(), m_pOutputOption->GetChoiceName(m_SelectedItemIndexArray[ItemShader]));

    screenInfo.EndPrint();
    screenInfo.AdjustWindowSize();

    const float menuWindowWidth = 300.0f;
    screenInfo.SetWindowSize(menuWindowWidth, screenInfo.GetWindowHeight());
}
