﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


/**
    @examplesource{FtmI2cSimple.cpp,PageSampleFtmI2cSimple}

    @brief
    Sample program for I2C driver with Touch IC
*/

/**
    @page PageSampleFtmI2cSimple Simple I2C driver sample program
    @tableofcontents

    @brief
    Sample Program for I2C driver with Touch IC

    @section PageSampleFtmI2cSimple_SectionBrief 概要
    コントローラの最新のボタン入力を取得する方法について説明します。

    @section PageSampleFtmI2cSimple_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/FtmI2cSimple @endlink 以下にあります。

    @section PageSampleFtmI2cSimple_SectionNecessaryEnvironment 必要な環境
    Touch IC I2C access should be disabled before running this program

    @section PageSampleFtmI2cSimple_SectionHowToOperate 操作方法
    This sample program read the Version Address Register of Touch IC and print on the console.

    @section PageSampleFtmI2cSimple_SectionPrecaution 注意事項
    This sample program do not display on the screen

    @section PageSampleFtmI2cSimple_SectionHowToExecute 実行手順
    Build and run the sample program.

    @section PageSampleFtmI2cSimple_SectionDetail 解説
    This sample program works as following.
    - Initialize I2C driver.
    - Generate I2C command list.
    - Execute I2C command list.
*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/i2c/i2c_BusDev.h>
#include <nn/i2c/i2c_Lib.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>

namespace
{
    void InitializeI2c(::nn::i2c::I2cSession* pOutSession) NN_NOEXCEPT
    {
        // Initialize I2C driver
        ::nn::i2c::Initialize();

        // Open session for Ftm port (I2C Bus index = 2, Slave Address = 0x49, Addressing Mode = BitWidth 7, Speed Mode = Fast)
        ::nn::i2c::OpenSessionForDev(pOutSession, 2, 0x49, ::nn::i2c::AddressingMode_BitWidth7, ::nn::i2c::SpeedMode_Fast);

        // Wait for 1msec
        ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(1));
    }

    void ReadVersionAddressRegister(const ::nn::i2c::I2cSession* pSession) NN_NOEXCEPT
    {
        // Write into I2c
        const uint8_t HwCommandReadVersionAddress[] = { 0xD0, 0x00, 0x60 };
        ::nn::i2c::Send(*pSession, HwCommandReadVersionAddress, sizeof(HwCommandReadVersionAddress), ::nn::i2c::TransactionOption_StopCondition);

        // Read From I2C
        uint8_t address[3];
        ::nn::i2c::Receive(address, *pSession, sizeof(address), ::nn::i2c::TransactionOption_StartCondition);

        // Print the result
        // [0:0] Dummy
        // [2:1] Address of register of version information
        NN_LOG("VersionAddressRegister %02x%02xh\n", address[2], address[1]);
    }

} // namespace

extern "C" void nnMain()
{
    NN_LOG("I2C Sample Start.\n");
    ::nn::i2c::I2cSession session;
    InitializeI2c(&session);
    ReadVersionAddressRegister(&session);
    NN_LOG("I2C Sample Done.\n");
}
